# -*- coding: utf-8 -*-

#  Bluemindo
#  notification.py

#    Bluemindo: A really simple but powerful audio player in Python/PyGTK.
#    Copyright (C) 2007-2008  Erwan Briand

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation version 3 of the License.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from os.path import join, isdir, isfile, expanduser
from os import makedirs
from gobject import markup_escape_text
from gtk.glade import XML as glade_XML
from gtk import STOCK_DIALOG_INFO, WRAP_WORD, WIN_POS_CENTER_ALWAYS
from gtk.gdk import pixbuf_new_from_file, INTERP_BILINEAR
from string import replace as str_replace
import ConfigParser
import pynotify

from common.config import ConfigLoader
from common.functions import Functions

class Notification:
    def __init__(self, plugin):   
        self.plugin = {'name': 'Notification',
                       'version': 0.1,
                       'logo': STOCK_DIALOG_INFO,
                       'configurable': True,
                       'authors': 'Erwan Briand <erwan@codingteam.net>',
                       'license': 'GNU General Public License 3',
                       'description': _('Shows desktop notification')}

        plugin.connect('OnModuleConfiguration', self.loadconfig)
        plugin.connect('OnModuleConfigurationSave', self.saveconfig)
        plugin.connect('OnPlayNewSong', self.handler_play_new_song)
        plugin.connect('OnStatusIconAnswer', self.handler_statusicon)
        
        config = ConfigLoader()
        self.plugin_conf = join(config.confdir, 'plugins', 'notification')
        self.player_data = join(config.datadir, 'modules', 'player')
        if not isdir(self.plugin_conf):
            makedirs(self.plugin_conf)

        self.configfile = ConfigParser.ConfigParser()
        self.configfile_ = self.plugin['name'] + '.cfg'

        self.default_title = '<b>{title}</b>'
        self.default_text = '<i>{artist}</i>\n{album}'
        self.default_timeout = 10000
        self.default_active = 1

        if not isfile(join(self.plugin_conf, self.configfile_)):
            self.write_default_config()

        self.notif = None
        pynotify.init('bluemindo')
        
        self.functions = Functions()

    def handler_statusicon(self, sti):
        self.tray = sti

    def handler_play_new_song(self, song):
        track = song[0]
        title = song[1]
        artist = song[2]
        album = song[3]
        genre = song[4]
        length = song[5]

        def parse(str_):
            str_ = str_replace(str_, '{title}', markup_escape_text(title))
            str_ = str_replace(str_, '{artist}', markup_escape_text(artist))
            str_ = str_replace(str_, '{album}', markup_escape_text(album))
            str_ = str_replace(str_, '{track}', str(track))
            str_ = str_replace(str_, '{length}', length)
            str_ = str_replace(str_, '{genre}', markup_escape_text(genre))

            return str_
        
        self.configfile.read(join(self.plugin_conf, self.configfile_))
        form0 = self.configfile.get(self.plugin['name'], 'title')
        form1 = self.configfile.get(self.plugin['name'], 'text')
        form2 = self.configfile.get(self.plugin['name'], 'timeout')
        form3 = int(self.configfile.get(self.plugin['name'], 'active'))

        if form3 == 1:
            notif_title = parse(form0)
            notif_text = parse(form1)
            notif_time = int(form2[:-2])
        
            _file = join(self.player_data,
                         self.functions.get_hash(album, artist) + '.covers')
            if isfile(_file):
                icon = pixbuf_new_from_file(_file)
                icon = icon.scale_simple(50, 50, INTERP_BILINEAR)
            else:
                icon = None

            if self.notif:
                if icon:
                    self.notif.update(notif_title, notif_text)
                else:
                    self.notif.update(notif_title, notif_text, STOCK_DIALOG_INFO)
            else:
                if icon:
                    self.notif = pynotify.Notification(notif_title, notif_text)
                else:
                    self.notif = pynotify.Notification(notif_title, notif_text,
                                                       STOCK_DIALOG_INFO)
                self.notif.set_urgency(pynotify.URGENCY_NORMAL)

                try:
                    self.notif.attach_to_status_icon(self.tray)
                except:
                    print ('Cannot attach notification to `%s` status '
                           'icon! (Issue: '
                           'http://trac.galago-project.org/ticket/121)' %
                           self.tray)

                self.notif.set_timeout(notif_time)

            if icon:
                self.notif.set_icon_from_pixbuf(icon)

            self.notif.show()

    # This function shows the configuration and interface
    def loadconfig(self, (module, confglade)):
        if module == self.plugin['name']:
            # Load the glade and put the vertical box in the module's
            # configuration one
            self.conf_widgets = glade_XML(
                                'plugins/notification/configuration.glade',
                                'vbox1', domain='bluemindo')

            hbox = confglade.get_widget('hbox2')

            try:
                kids = hbox.get_children()
                hbox.remove(kids[2])
            except:
                pass

            hbox.add(self.conf_widgets.get_widget('vbox1'))

            try:
                self.configfile.read(join(self.plugin_conf, self.configfile_))
                form0 = self.configfile.get(self.plugin['name'], 'title')
                form1 = self.configfile.get(self.plugin['name'], 'text')
                form2 = self.configfile.get(self.plugin['name'], 'timeout')
                form3 = self.configfile.get(self.plugin['name'], 'active')

                file_exist = True
            except:
                file_exist = False

            self.conf_widgets.get_widget('label6').set_markup(
                _('Available tags:\n <b><tt>{title}</tt></b> - '
                  'Insert the title\n <b><tt>{artist}</tt></b> - '
                  'Insert the artist\n <b><tt>{album}</tt></b> - '
                  'Insert the album\n <b><tt>{track}</tt></b> - '
                  'Insert the track number\n <b><tt>{length}</tt></b> - '
                  'Insert the length\n <b><tt>{genre}</tt></b> - '
                  'Insert the genre'))

            textentry = self.conf_widgets.get_widget('entry1')
            spinbutton = self.conf_widgets.get_widget('spinbutton1')
            check = self.conf_widgets.get_widget('checkbutton1')

            check.set_active(int(form3))
            text = self.conf_widgets.get_widget('textview1')
            text.set_wrap_mode(WRAP_WORD)
            textview = text.get_buffer()

            if file_exist:
                textentry.set_text(form0)
                spinbutton.set_value(float(form2))
                textview.set_text(form1)

    # This function saves the configuration
    def saveconfig(self, extension):
        if extension == self.plugin['name']:
            try:
                check = self.conf_widgets.get_widget('checkbutton1')
                textentry = self.conf_widgets.get_widget('entry1')
                spinbutton = self.conf_widgets.get_widget('spinbutton1')
                textview = self.conf_widgets.get_widget('textview1')

                form0 = textentry.get_text()
                form2 = spinbutton.get_value()
                form3 = int(check.get_active())

                buf = textview.get_buffer()
                form1 = buf.get_text(buf.get_start_iter(), buf.get_end_iter())

                widgets_retrieved = True
            except:
                widgets_retrieved = False

            if widgets_retrieved:
                try:
                    self.configfile.add_section(self.plugin['name'])
                except:
                    pass

                self.configfile.set(self.plugin['name'], 'title', form0)
                self.configfile.set(self.plugin['name'], 'text', form1)
                self.configfile.set(self.plugin['name'], 'timeout', form2)
                self.configfile.set(self.plugin['name'], 'active', form3)

                self.configfile.write(open(join(self.plugin_conf,
                                                self.configfile_), 'w'))

    # This function writes the default configuration
    def write_default_config(self):
        self.configfile.add_section(self.plugin['name'])
        self.configfile.set(self.plugin['name'], 'title', self.default_title)
        self.configfile.set(self.plugin['name'], 'text', self.default_text)
        self.configfile.set(self.plugin['name'],
                                        'timeout', self.default_timeout)
        self.configfile.set(self.plugin['name'], 'active', self.default_active)

        self.configfile.write(open(join(self.plugin_conf,
                                        self.configfile_), 'w'))
