# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from threading import Thread
from gobject import (TYPE_STRING as gString, markup_escape_text as escape,
                     timeout_add)
from gtk.gdk import Pixbuf as gPixbuf, threads_enter, threads_leave
from gtk import(STOCK_DIRECTORY, STOCK_CDROM, STOCK_FILE, TreeStore,
                CellRendererText, TreeViewColumn, CellRendererPixbuf,
                ICON_SIZE_MENU)

from modules.explorer.playlist import Playlist
from modules.explorer.musicdb import MusicDatabase
from modules.explorer import viewscommon
unescape = viewscommon.unescape
insensitive_sort = viewscommon.insensitive_sort

class BasicView(object):
    ref = None
    ref2 = None

    def __new__(cls, *args, **kws):
        # Singleton
        if cls.ref is None:
            cls.ref = super(BasicView, cls).__new__(cls, args, kws)
        return cls.ref

    def __init__(self, glade_file, config, userconf):
        self.playlist = Playlist(glade_file, config, userconf)
        self.glade_file = glade_file
        self.config = config
        self.userconf = userconf

        # Show widget
        self.glade_file.get_widget('vbox2').show()
        self.glade_file.get_widget('scrolledwindow7').show()

        if BasicView.ref2 is None: 
            BasicView.ref2 = 42 
            self.launch()

    def launch(self):
        treestore = TreeStore(gPixbuf, gString)
        column = TreeViewColumn()

        render_pixbuf = CellRendererPixbuf()
        column.pack_start(render_pixbuf, expand=False)
        column.add_attribute(render_pixbuf, 'pixbuf', 0)

        render_text = CellRendererText()
        column.pack_start(render_text, expand=True)
        column.add_attribute(render_text, 'markup', 1)

        tview = self.glade_file.get_widget('treeview7')
        tview.set_model(treestore)
        tview.append_column(column)
        tview.connect('row-activated', self.add_to_playlist)

        # Add artists
        self.reload_database()

    def reload_database(self, force_scan=False):
        tview = self.glade_file.get_widget('treeview7')
        treestore = tview.get_model()

        pbar = self.glade_file.get_widget('progressbar1')
        pbar.set_text(_('Opening folder…'))
        pbar.show()

        thread = Thread(group=None, target=self.insert_artists,
                        name='artists', args=(pbar, treestore, tview,
                                              force_scan))
        self.thread_name = thread

        thread.start()

    def timeout_progress(self, pbar):
        if self.thread_name.isAlive():
            pbar.pulse()
            return True
        else:
            pbar.hide()

    def insert_artists(self, pbar, treestore, tview, force_scan):
        """This function is called at the start-up and add artists."""
        threads_enter()

        # Call database
        musicdb = MusicDatabase(self.config, pbar)

        if self.config['scan'] or force_scan:
            musicdb.scan()

        pbar.set_text(_('Loading music…'))
        self.database = musicdb.load()

        timeout_add(100, self.timeout_progress, pbar)
        self.songs_tree = viewscommon.get_representation(self.database,
                                                        self.config['artwork'])
        self.continue_to_pulse = False

        pbar.set_text(_('Loading artists…'))
        items = self.songs_tree.items()
        items.sort(insensitive_sort)
        items_sort = [key for key, value in items]

        treestore.clear()
        for artist in items_sort:
            logo = tview.render_icon(STOCK_DIRECTORY, ICON_SIZE_MENU)
            t = treestore.append(None, (logo, '<b>%s</b>' % escape(artist)))

            for album in self.songs_tree[artist]:
                logo = tview.render_icon(STOCK_CDROM, ICON_SIZE_MENU)
                u = treestore.append(t, (logo, '<i>%s</i>' % escape(album)))

                for song in self.songs_tree[artist][album]:
                    logo = tview.render_icon(STOCK_FILE, ICON_SIZE_MENU)
                    v = treestore.append(u, (logo, escape(song[0])))

        threads_leave()

    def add_to_playlist(self, tview, path, column):
        """This function handles row activation."""
        (mod, iter_) = tview.get_selection().get_selected()
        iter_has_depth = tview.get_model().iter_depth(iter_)

        if not iter_has_depth:
            artist_ = unescape(mod.get_value(iter_, 1))[3:-4]

            songs = []
            for album in self.songs_tree[artist_]:
                for sgs in self.songs_tree[artist_][album]:
                    songs.append(sgs)

        elif iter_has_depth == 1:
            artist = tview.get_model().iter_parent(iter_)
            artist_ = unescape(mod.get_value(artist, 1))[3:-4]
            album_ = unescape(mod.get_value(iter_, 1))[3:-4]

            songs = []
            for sgs in self.songs_tree[artist_][album_]:
                songs.append(sgs)

        elif iter_has_depth == 2:
            album = tview.get_model().iter_parent(iter_)
            artist = tview.get_model().iter_parent(album)

            song_ = unescape(mod.get_value(iter_, 1))
            artist_ = unescape(mod.get_value(artist, 1))[3:-4]
            album_ = unescape(mod.get_value(album, 1))[3:-4]

            songs = []
            for sgs in self.songs_tree[artist_][album_]:
                if song_ == sgs[0]:
                    songs.append(sgs)

        try:
            self.playlist.add_songs(songs)
        except NameError:
            pass
