/*
*
*  A2DPD - Bluetooth A2DP daemon for Linux
*
*  Copyright (C) 2006-2007  Frédéric DALLEAU <frederic.dalleau@palmsource.com>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/poll.h>
#include <unistd.h>
#include <signal.h>
#include "a2dpd_dbus.h"
#include "a2dpd_protocol.h"
#include "a2dpd_ipc.h"

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>

// D-Bus constants
#define A2DPD_SERVICE_NAME			"com.access.a2dpd"
#define A2DPD_SERVICE_PATH			"/com/access/a2dpd"
#define A2DPD_SERVER_INTERFACE_NAME		A2DPD_SERVICE_NAME ".server"

#define A2DPD_SERVER_FAILED_ERROR		"com.access.Error.Failed"
#define A2DPD_SERVER_PARAM_ERROR		"com.access.Error.ParamError"
#define A2DPD_SERVER_BUSY_ERROR			"com.access.Error.Busy"
#define A2DPD_SERVER_NOT_SUPPORTED_ERROR	"com.access.Error.NotSupported"
#define A2DPD_SERVER_ALREADY_EXISTS_ERROR	"com.access.Error.AlreadyExists"
#define A2DPD_SERVER_NOT_EXIST_ERROR		"com.access.Error.DoesNotExist"

static DBusConnection *dbus_conn = NULL;
static int ctl_socket = 0;
static struct pollfd dbus_pollfds[256];
static DBusWatch *dbus_watches[256];
static int dbus_watches_enabled[256];
static char g_cur_addr[20] = "";
static int g_state = DISCONNECTED;

int write_config_string(char *filename, char *section, char *key, char *value)
{
	DBG("%s:[%s] %s=%s", filename, section, key, value);
/*
	GError *error = NULL;
	GKeyFile *key_file = g_key_file_new();
	gchar *key_file_data = NULL;
	unsigned int key_file_len = 0;
	int result = EINVAL;

	if (key_file)
	{
		g_key_file_load_from_file(key_file, filename, G_KEY_FILE_KEEP_COMMENTS, &error);
		if (error)
		{
			DBG("Failed to load key file %s: %s", filename, error->message);
			g_error_free(error);
			error = NULL;
		}
		g_key_file_set_value(key_file, section, key, value);
		key_file_data = g_key_file_to_data(key_file, &key_file_len, &error);
		if (error)
		{
			DBG("Failed to convert keyfile %s to data: %s", filename, error->message);
			g_error_free(error);
			error = NULL;
		}
		g_key_file_free(key_file);
		if (key_file_data)
		{
			FILE *hFile = fopen(filename, "wt");

			if (hFile)
			{
				if ((int) (fwrite((void *) key_file_data, key_file_len, 1, hFile)) != 1)
				{
					DBG("Write failed (errno=%d:%s)", errno, strerror(errno));
					result = errno;
				}
				else
				{
					DBG("Write %s successful", filename);
					result = 0;
				}

				fclose(hFile);
			}
			else
			{
				DBG("fopen %s failed", filename);
				result = errno;
			}
		}
	}
	return result;
*/
	return -1;
}

static DBusHandlerResult a2dpd_dbus_Connect(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char c = 'c';
	DBG("Begin");
	write(ctl_socket, &c, sizeof(c));
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_Disconnect(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char c = 'd';
	DBG("Begin");
	write(ctl_socket, &c, sizeof(c));
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_AVRCPDisconnect(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char c = 'D';
	DBG("Begin");
	write(ctl_socket, &c, sizeof(c));
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_StreamStart(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char c = 's';
	DBG("Begin");
	write(ctl_socket, &c, sizeof(c));
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_StreamSuspend(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char c = 'p';
	DBG("Begin");
	write(ctl_socket, &c, sizeof(c));
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_Startup(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	// This dummy method is intended to be used with activation
	// Calling startup should startup the daemon thanks to dbus activation
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_Exit(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	kill(getpid(), SIGTERM);
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_Save(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char c = 'w';
	DBG("Begin");
	write(ctl_socket, &c, sizeof(c));
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_SetAddress(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char addr[32];
	char* lpszParam = NULL;
	char c = 't';
	DBG("Begin");
	if(dbus_message_get_args(message, error, DBUS_TYPE_STRING, &lpszParam, DBUS_TYPE_INVALID)) {
		memset(addr, 0, sizeof(addr));
		strncpy(addr, lpszParam, sizeof(addr));
		write(ctl_socket, &c, sizeof(c));
		write(ctl_socket, &addr, sizeof(addr));
	}
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_AutoConnect(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char* lpszParam = NULL;
	char c0 = 'a';
	char c1 = 'a';
	DBG("Begin");

	if(dbus_message_get_args(message, error, DBUS_TYPE_STRING, &lpszParam, DBUS_TYPE_INVALID)) {
		DBG("Param %s", lpszParam);
		write(ctl_socket, &c0, sizeof(c0));
		if(!strcasecmp(lpszParam, "swap")) {
			c1 = 'a';
		} else if(!strcasecmp(lpszParam, "0")) {
			c1 = '0';
		} else {
			c1 = '1';
		}
		write(ctl_socket, &c1, sizeof(c1));
	}
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_SetDebug(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char* lpszParam = NULL;
	DBG("Begin");

	if(dbus_message_get_args(message, error, DBUS_TYPE_STRING, &lpszParam, DBUS_TYPE_INVALID)) {
		g_bdebug = atoi(lpszParam);
		DBG("Param \"%s\" => %d", lpszParam, g_bdebug);
	}
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_SetOutput(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char* lpszParam = NULL;
	DBG("Begin");

	if(dbus_message_get_args(message, error, DBUS_TYPE_STRING, &lpszParam, DBUS_TYPE_INVALID)) {
		DBG("Param %s", lpszParam);
		FILE* fdout = NULL;
		if(g_fdout!=NULL) {
			fdout = g_fdout;
			// Make sure no trace is using the handle
			g_fdout = NULL;
			// Close handle
			fclose(fdout);
		}
		fdout = fopen(lpszParam, "wt");
		if(fdout != NULL) {
			g_fdout = fdout;
		}
	}
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_SetVolume(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char* lpszParam = NULL;
	char c = 'v';
	int param = 0;
	
	DBG("Begin");

	if(dbus_message_get_args(message, error, DBUS_TYPE_STRING, &lpszParam, DBUS_TYPE_INVALID)) {
		param = atoi(lpszParam);
		DBG("Param %s => %d", lpszParam, param);
		write(ctl_socket, &c, sizeof(c));
		write(ctl_socket, &param, sizeof(param));
	}
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_SetFlags(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char* lpszParam = NULL;
	char c = 'f';
	int param = 0;

	DBG("Begin");

	if(dbus_message_get_args(message, error, DBUS_TYPE_STRING, &lpszParam, DBUS_TYPE_INVALID)) {
		param = atoi(lpszParam);
		DBG("Param %s => %d", lpszParam, param);
		write(ctl_socket, &c, sizeof(c));
		write(ctl_socket, &param, sizeof(param));
	}
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_SetReReadConfig(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char* lpszParam = NULL;
	char c = 'r';
	int param = 0;

	DBG("Begin");

	if(dbus_message_get_args(message, error, DBUS_TYPE_STRING, &lpszParam, DBUS_TYPE_INVALID)) {
		param = atoi(lpszParam);
		DBG("Param %s => %d", lpszParam, param);
		write(ctl_socket, &c, sizeof(c));
		write(ctl_socket, &param, sizeof(param));
	}
	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_GetAddress(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char* cur_addr = (char*)g_cur_addr;

	DBG("Begin %s", cur_addr);

	// Append current address to reply
	if (dbus_message_append_args (reply,
				DBUS_TYPE_STRING, &cur_addr,
				DBUS_TYPE_INVALID)) {
	}

	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult a2dpd_dbus_GetState(DBusMessage * message, DBusMessage * reply, DBusError* error, void *user_data)
{
	char* cur_state = (g_state==DISCONNECTED)?"Disconnected":(g_state==CONNECTING)?"Connecting":"Connected";

	DBG("Begin %s", cur_state);

	// Append current address to reply
	if (dbus_message_append_args (reply,
				DBUS_TYPE_STRING, &cur_state,
				DBUS_TYPE_INVALID)) {
	}

	DBG("OK");
	return DBUS_HANDLER_RESULT_HANDLED;
}

void a2dpd_signal_command(char* dir, char* cmd)
{
	// Print a trace
	DBG("%s %s", cmd, dir);

	// Send signal if connected
	if (dbus_conn != NULL) {
		DBusMessage *signal = dbus_message_new_signal (A2DPD_SERVICE_PATH, A2DPD_SERVER_INTERFACE_NAME, "A2Command");
		if(signal) {
			if (dbus_message_append_args (signal,
						DBUS_TYPE_STRING, &dir,
						DBUS_TYPE_STRING, &cmd,
						DBUS_TYPE_INVALID)) {
				// In case this gets fired off after we've disconnected.
				if (dbus_connection_get_is_connected (dbus_conn)) {
					dbus_connection_send (dbus_conn, signal, NULL);
				}
			}
			dbus_message_unref(signal);
		}
	}
}

void a2dpd_signal_state(int state, char* bdaddr)
{
	static int s_state = -1;

	if(state != s_state) {
		g_state = state;
		char* signal_name = (state==DISCONNECTED)?"Disconnected":(state==CONNECTING)?"Connecting":"Connected";
		DBG("%s %s", signal_name, bdaddr);

		// Send signal if connected
		if (dbus_conn != NULL) {
			DBusMessage *signal = dbus_message_new_signal (A2DPD_SERVICE_PATH, A2DPD_SERVER_INTERFACE_NAME, "A2StateChange");
			if(signal) {
				if (dbus_message_append_args (signal,
							DBUS_TYPE_STRING, &signal_name,
							DBUS_TYPE_STRING, &bdaddr,
							DBUS_TYPE_INVALID)) {
					// In case this gets fired off after we've disconnected.
					if (dbus_connection_get_is_connected (dbus_conn)) {
						dbus_connection_send (dbus_conn, signal, NULL);
						dbus_connection_flush(dbus_conn);
					}
				}
				dbus_message_unref(signal);
			}
		}

		// Print a trace
		s_state = state;
	}
}

void a2dpd_signal_address_changed(char* cur_addr)
{
	// Print a trace
	DBG("%s", cur_addr);

	strncpy(g_cur_addr, cur_addr, sizeof(g_cur_addr));
	g_cur_addr[sizeof(g_cur_addr)-1] = '\0';

	// Send signal if connected
	if (dbus_conn != NULL) {
		DBusMessage *signal = dbus_message_new_signal (A2DPD_SERVICE_PATH, A2DPD_SERVER_INTERFACE_NAME, "A2AddressChanged");
		if(signal) {
			if (dbus_message_append_args (signal,
						DBUS_TYPE_STRING, &cur_addr,
						DBUS_TYPE_INVALID)) {
				// In case this gets fired off after we've disconnected.
				if (dbus_connection_get_is_connected (dbus_conn)) {
					dbus_connection_send (dbus_conn, signal, NULL);
				}
			}
			dbus_message_unref(signal);
		}
	}
}

static DBusHandlerResult a2dpd_handler_func(DBusConnection * connection, DBusMessage * message, void *user_data)
{
	DBusHandlerResult result = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	DBusError dbusError;
	DBusMessage *reply = dbus_message_new_method_return(message);

	if (reply) {
		dbus_error_init(&dbusError);

		DBG("");

		// Methods
		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "Connect"))
			result = a2dpd_dbus_Connect(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "Disconnect"))
			result = a2dpd_dbus_Disconnect(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "StreamStart"))
			result = a2dpd_dbus_StreamStart(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "StreamSuspend"))
			result = a2dpd_dbus_StreamSuspend(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "SetAddress"))
			result = a2dpd_dbus_SetAddress(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "AutoConnect"))
			result = a2dpd_dbus_AutoConnect(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "SetDebug"))
			result = a2dpd_dbus_SetDebug(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "SetOutput"))
			result = a2dpd_dbus_SetOutput(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "SetVolume"))
			result = a2dpd_dbus_SetVolume(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "SetFlags"))
			result = a2dpd_dbus_SetFlags(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "SetReReadConfig"))
			result = a2dpd_dbus_SetReReadConfig(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "Startup"))
			result = a2dpd_dbus_Startup(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "Exit"))
			result = a2dpd_dbus_Exit(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "Save"))
			result = a2dpd_dbus_Save(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "AVRCPDisconnect"))
			result = a2dpd_dbus_AVRCPDisconnect(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "GetAddress"))
			result = a2dpd_dbus_GetAddress(message, reply, &dbusError, user_data);

		if (dbus_message_is_method_call(message, A2DPD_SERVER_INTERFACE_NAME, "GetState"))
			result = a2dpd_dbus_GetState(message, reply, &dbusError, user_data);

		// Signals
		if (dbus_message_is_signal(message, DBUS_INTERFACE_LOCAL, "Disconnected"))
		{
			DBG("Disconnected from BUS");
			//result = a2dpd_dbus_Exit(message, reply, &dbusError, user_data);
			result = DBUS_HANDLER_RESULT_HANDLED;
		}

		// Send answers
		if(!dbus_connection_send(connection, reply, NULL)) {
			DBG("Failed to send reply");
			result = DBUS_HANDLER_RESULT_NEED_MEMORY;
		}
		dbus_message_unref(reply);

		dbus_error_free(&dbusError);
	} else {
		DBG("Failed to allocate reply");
		result = DBUS_HANDLER_RESULT_NEED_MEMORY;
	}

	return result;
}

uint32_t add_dbus_watch(DBusWatch *watch, void *data)
{
	// From: Claudio Takahasi
	uint32_t  cond   =  0;
	int i;
	uint32_t result = 0;
	int fd = dbus_watch_get_fd(watch);
	int flags = dbus_watch_get_flags(watch);

	if (flags & DBUS_WATCH_READABLE)
		cond |= POLLIN;
	if (flags & DBUS_WATCH_WRITABLE)
		cond |= POLLOUT;

	for(i=0; i<ARRAY_SIZE(dbus_pollfds); i++) {
		if(dbus_pollfds[i].fd == 0) {
			dbus_pollfds[i].fd = fd;
			dbus_pollfds[i].events = cond;
			dbus_pollfds[i].revents = 0;
			dbus_watches[i] = watch;
			dbus_watches_enabled[i] = dbus_watch_get_enabled(watch);
			dbus_watch_set_data(watch, (void*)i, NULL);
			result = 1;
			DBG("Added watch %d %p %s", i, watch, dbus_watches_enabled[i]?"enabled":"disabled");
			break;
		}
	}

	if(result == 0) {
		DBG("Failed to add watch %d %p", i, watch);
	}

	return result;
}

void remove_dbus_watch(DBusWatch *watch, void *data)
{
	int i = (int) data;
	DBG("Removing watch %d %p", i, dbus_watches[i]);

	dbus_pollfds[i].fd = 0;
	dbus_pollfds[i].events = 0;
	dbus_pollfds[i].revents = 0;
	dbus_watches[i] = NULL;
	dbus_watches_enabled[i] = 0;
}

void watch_dbus_toggled(DBusWatch *watch, void *data)
{
	int i = (int) data;
	// Remember enabled state
	dbus_watches_enabled[i] = dbus_watch_get_enabled(watch);
}

// Main fonction for dbus fds
void pollfd_cb_dbuswatches(struct pollfd* pollfds, DBusWatch* watch, void* param2)
{
	if (pollfds->revents) {
		uint32_t flags = 0;

		if (pollfds->revents & POLLIN)
			flags |= DBUS_WATCH_READABLE;
		if (pollfds->revents & POLLOUT)
			flags |= DBUS_WATCH_WRITABLE;
		if (pollfds->revents & POLLHUP)
			flags |= DBUS_WATCH_HANGUP;
		if (pollfds->revents & POLLERR)
			flags |= DBUS_WATCH_ERROR;
		if (pollfds->revents & POLLNVAL)
			flags |= DBUS_WATCH_ERROR;

		dbus_watch_handle(watch, flags);

		dbus_connection_ref(dbus_conn);

		/* Dispatch messages */
		while (dbus_connection_dispatch(dbus_conn) == DBUS_DISPATCH_DATA_REMAINS);

		dbus_connection_unref(dbus_conn);
	}
}

void a2dpd_signal_add_fd_to_poll(struct pollinfo* pollinfos)
{
	int i;
	for(i=0; i<ARRAY_SIZE(dbus_pollfds); i++) {
		// Do not poll disabled watches
		if(dbus_pollfds[i].fd != 0 && dbus_watches_enabled[i]) {
			add_fd_to_poll(pollinfos, dbus_pollfds[i].fd, dbus_pollfds[i].events, -1, (fnpollfd_cb)pollfd_cb_dbuswatches, dbus_watches[i], NULL);
		}
	}
}

void a2dpd_signal_init(int session_bus)
{
	DBusError dbusError;
	static DBusObjectPathVTable dbus_vtable_a2dpd_server = { NULL, a2dpd_handler_func, NULL, NULL, NULL, NULL };
	dbus_error_init(&dbusError);

	memset(dbus_pollfds, 0, sizeof(dbus_pollfds));
	g_state = DISCONNECTED;
	g_cur_addr[0] = '\0';

	DBG("Getting on DBUS");
	dbus_conn = dbus_bus_get((session_bus)?DBUS_BUS_SESSION:DBUS_BUS_SYSTEM, &dbusError);
	if(dbus_conn != NULL) {
		DBG("Installing watch");
		dbus_connection_set_watch_functions( dbus_conn, add_dbus_watch, remove_dbus_watch, watch_dbus_toggled, NULL, NULL);

		DBG("Registering object path: "A2DPD_SERVICE_PATH);
		if (dbus_connection_register_object_path(dbus_conn, A2DPD_SERVICE_PATH, &dbus_vtable_a2dpd_server, NULL)) {
			DBG("Acquiring service: " A2DPD_SERVICE_NAME);
			if(dbus_bus_request_name(dbus_conn, A2DPD_SERVICE_NAME, 0, &dbusError)) {
				if (!dbus_error_is_set(&dbusError)) {
					DBG("OK");
				}
			}
		}
	} else {
		DBG("Failed to get on DBUS");
	}
	dbus_error_free(&dbusError);
	DBG("OK");
}

void a2dpd_signal_set_socket(int a2dpd_ctl_socket)
{
	DBG("Signal socket set to %d", a2dpd_ctl_socket);
	ctl_socket = a2dpd_ctl_socket;
}

void a2dpd_signal_kill()
{
	DBG("OK");
}
