/*
*
*  A2DPD - Bluetooth A2DP daemon for Linux
*
*  Copyright (C) 2006-2007  Frédéric DALLEAU <frederic.dalleau@palmsource.com>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <malloc.h>
#include <signal.h>
#include <time.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/select.h>
#include <sys/poll.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>
#include <bluetooth/l2cap.h>
#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>

#include "a2dpd_output_a2dp.h"
#include "a2dpd_tools.h"
#include "a2dpd_protocol.h"
#include "a2dpd_uinput.h"
#include "a2dpd_dbus.h"
#include "a2dpd_avrcp.h"
#include "a2dpd_ipc.h"
#include "../avrcp.h"

#define MAXCLIENTS 16

typedef struct avrcp_data_t {
	char sCmdPlay[512];
	char sCmdPause[512];
	char sCmdStop[512];
	char sCmdPrev[512];
	char sCmdNext[512];
	char sCmdNew[512];
	int sockfd;
	int new_fd[MAXCLIENTS];
} avrcp_data;

static void init_response(struct avctp_header *header)
{
	header->ipid = 0;
	header->cr = AVCTP_RESPONSE_FRAME;
	header->packet_type = PACKET_TYPE_SINGLE;
}

int avrcp_handle_message(LPAVRCP lpAVRCP, int sockfd)
{
	char lpFrame[A2DPMAXIMUMTRANSFERUNITSIZE];
	int iReceived = recv(sockfd, lpFrame, sizeof(lpFrame), 0);
	if (iReceived > 0) {
		struct avc_frame frame = *((struct avc_frame *) lpFrame);

		if (frame.ctype == CTYPE_CONTROL && frame.opcode == OP_PASSTHROUGH) {
			switch (frame.operand0) {
			#define CASEPRINT_DN(x) case (x): DBG("%s DOWN", #x); a2dpd_signal_command("down", #x); break;
			#define CASEPRINT_UP(x) case (x+128): DBG("%s UP", #x); a2dpd_signal_command("up", #x); break;
			#define CASEPRINT(x)  CASEPRINT_DN(x) CASEPRINT_UP(x)
			#define CASEDOWN(x) case (x): DBG("%s DOWN", #x); a2dpd_signal_command("down", #x); 
			#define CASEUP(x) case (x+128): DBG("%s UP", #x); a2dpd_signal_command("up", #x); 
			CASEPRINT(avrcp_select);
			CASEPRINT(avrcp_up);
			CASEPRINT(avrcp_down);
			CASEPRINT(avrcp_left);
			CASEPRINT(avrcp_right);
			CASEPRINT(avrcp_right_up);
			CASEPRINT(avrcp_right_down);
			CASEPRINT(avrcp_left_up);
			CASEPRINT(avrcp_left_down);
			CASEPRINT(avrcp_root_menu);
			CASEPRINT(avrcp_setup_menu);
			CASEPRINT(avrcp_contents_menu);
			CASEPRINT(avrcp_favourite_menu);
			CASEPRINT(avrcp_exit);
			CASEPRINT(avrcp_0);
			CASEPRINT(avrcp_1);
			CASEPRINT(avrcp_2);
			CASEPRINT(avrcp_3);
			CASEPRINT(avrcp_4);
			CASEPRINT(avrcp_5);
			CASEPRINT(avrcp_6);
			CASEPRINT(avrcp_7);
			CASEPRINT(avrcp_8);
			CASEPRINT(avrcp_9);
			CASEPRINT(avrcp_dot);
			CASEPRINT(avrcp_enter);
			CASEPRINT(avrcp_clear);
			CASEPRINT(avrcp_channel_up);
			CASEPRINT(avrcp_channel_down);
			CASEPRINT(avrcp_sound_select);
			CASEPRINT(avrcp_input_select);
			CASEPRINT(avrcp_display_information);
			CASEPRINT(avrcp_help);
			CASEPRINT(avrcp_page_up);
			CASEPRINT(avrcp_page_down);
			CASEPRINT(avrcp_power);
			CASEPRINT(avrcp_volume_up);
			CASEPRINT(avrcp_volume_down);
			CASEPRINT(avrcp_mute);
			CASEPRINT_UP(avrcp_play);
			CASEDOWN(avrcp_play)
				DBG("[play] %s", lpAVRCP->sCmdPlay);
				if (lpAVRCP->sCmdPlay[0])
					async_run_process(lpAVRCP->sCmdPlay, 0);
				else
					send_key(A2DPD_UINPUT_PLAY);
				break;
			CASEPRINT_UP(avrcp_stop);
			CASEDOWN(avrcp_stop)
				DBG("[stop] %s", lpAVRCP->sCmdStop);
				if (lpAVRCP->sCmdStop[0])
					async_run_process(lpAVRCP->sCmdStop, 0);
				else
					send_key(A2DPD_UINPUT_STOP);
				break;
			CASEPRINT_UP(avrcp_pause);
			CASEDOWN(avrcp_pause)
				DBG("[pause] %s", lpAVRCP->sCmdPause);
				if (lpAVRCP->sCmdPause[0])
					async_run_process(lpAVRCP->sCmdPause, 0);
				else
					send_key(A2DPD_UINPUT_PAUSE);
				break;
			CASEPRINT(avrcp_record);
			CASEPRINT_UP(avrcp_rewind);
			CASEDOWN(avrcp_rewind)
				DBG("[previous] %s", lpAVRCP->sCmdPrev);
				if (lpAVRCP->sCmdPrev[0])
					async_run_process(lpAVRCP->sCmdPrev, 0);
				else
					send_key(A2DPD_UINPUT_PREV);
				break;
			CASEPRINT(avrcp_fast_forward);
			CASEPRINT(avrcp_eject);
			CASEPRINT_UP(avrcp_forward);
			CASEDOWN(avrcp_forward)
				DBG("[next] %s", lpAVRCP->sCmdNext);
				if (lpAVRCP->sCmdNext[0])
					async_run_process(lpAVRCP->sCmdNext, 0);
				else
					send_key(A2DPD_UINPUT_NEXT);
				break;
			CASEPRINT_UP(avrcp_backward);
			CASEDOWN(avrcp_backward)
				DBG("[previous] %s", lpAVRCP->sCmdPrev);
				if (lpAVRCP->sCmdPrev[0])
					async_run_process(lpAVRCP->sCmdPrev, 0);
				else
					send_key(A2DPD_UINPUT_PREV);
				break;
			CASEPRINT(avrcp_angle);
			CASEPRINT(avrcp_subpicture);
			CASEPRINT(avrcp_f1);
			CASEPRINT(avrcp_f2);
			CASEPRINT(avrcp_f3);
			CASEPRINT(avrcp_f4);
			CASEPRINT(avrcp_f5);
			CASEPRINT(avrcp_vendor_unique);
			default:
				DBG("received passthrough %d bytes: operand %d, operand %d", iReceived, frame.operand0, frame.operand1);
				//dump_packet(&frame, iReceived);
			}
		
			init_response(&frame.header);
			frame.ctype = CTYPE_ACCEPTED;
			write(sockfd, &frame, iReceived);
		} else if (frame.ctype == CTYPE_STATUS && frame.opcode == OP_UNITINFO) {
			DBG("received UNITINFO %d bytes:", iReceived);
			init_response(&frame.header);
			frame.ctype = CTYPE_STABLE;
			// Keep following fields from request
			//frame.subunit_type = 0x1F;
			//frame.subunit_id = 0x7;
			frame.operand0 = 0x7;
			frame.operand1 = 0x48;
			memset(((char*)&frame)+sizeof(struct avc_frame), 0xFF, iReceived-sizeof(struct avc_frame));
			write(sockfd, &frame, iReceived);
		} else if (frame.ctype == CTYPE_STATUS && frame.opcode == OP_SUBUNITINFO) {
			DBG("received SUBUNITINFO %d bytes:", iReceived);
			init_response(&frame.header);
			frame.ctype = CTYPE_STABLE;
			// Keep following fields from request
			//frame.subunit_type = 0x1F;
			//frame.subunit_id = 0x7;
			//frame.operand0 = 0x7;
			// Only customize subunit type
			frame.operand1 = 0x48;

			memset(((char*)&frame)+sizeof(struct avc_frame), 0xFF, iReceived-sizeof(struct avc_frame));
			write(sockfd, &frame, iReceived);
		} else {
			DBG("Only control and status ctype command is implemented!");
			DBG("received %d bytes:", iReceived);
		}
	} else {
		if (errno != EAGAIN)
			DBG("AVRCP Receive failed");
	}

	return iReceived;
}

void pollfd_cb_avrcpfd(struct pollfd* pollfds, LPAVRCP lpAVRCP, void* param2)
{
	char szRemote[64];
	uint16_t iMTU;

	// Handle incoming socket connections
	if(pollfds->revents & POLLIN) {
		int new_fd = a2dp_wait_connection(pollfds->fd, szRemote, sizeof(szRemote), &iMTU);
		if(new_fd>0) {
			int i;
			// Accept a new client
			for(i=0; i<MAXCLIENTS; i++) {
				if(lpAVRCP->new_fd[i]<=0) {
					DBG("socket %d: Connection from %s, mtu=%d accepted at index %d", new_fd, szRemote, iMTU, i);
					lpAVRCP->new_fd[i] = new_fd;
					break;
				}
			}
			if(i>=MAXCLIENTS) {
				// Reject a client if someone is already connected
				DBG("socket %d: Connection from %s, mtu=%d rejected", new_fd, szRemote, iMTU);
				close_socket(&new_fd);
			}
		}
	}
}

void pollfd_cb_newfd(struct pollfd* pollfds, LPAVRCP lpAVRCP, void* param2)
{
	// Handle incoming socket connections
	if(pollfds->revents & POLLIN) {
		avrcp_handle_message(lpAVRCP, pollfds->fd);
	}

	// Handle incoming socket connections
	if(pollfds->revents & POLLERR) {
		close_socket(&(lpAVRCP->new_fd[(int)param2]));
	}
}

void avrcp_add_fd_to_poll(LPAVRCP lpAVRCP, struct pollinfo* pollinfos)
{
	int i;
	if(lpAVRCP->sockfd>=0)
		add_fd_to_poll(pollinfos, lpAVRCP->sockfd, POLLIN, -1, (fnpollfd_cb)pollfd_cb_avrcpfd, lpAVRCP, NULL);

	for(i=0; i<MAXCLIENTS; i++) {
		if(lpAVRCP->new_fd[i]>0) {
			add_fd_to_poll(pollinfos, lpAVRCP->new_fd[i], POLLIN, -1, (fnpollfd_cb)pollfd_cb_newfd, lpAVRCP, (void*)i);
		}
	}
}

void avrcp_disconnect(LPAVRCP lpAVRCP)
{
	int i;
	// Close only incoming sockets, used by certification process
	for(i=0; i<MAXCLIENTS; i++) {
		if(lpAVRCP->new_fd[i]>0) {
			DBG("Closing socket %d", lpAVRCP->new_fd[i]);
			close_socket(&(lpAVRCP->new_fd[i]));
		}
	}
}

void avrcp_new_client(LPAVRCP lpAVRCP)
{
	a2dpd_signal_command("down", "new");
	a2dpd_signal_command("up", "new");
	if (lpAVRCP->sCmdNew[0]) {
		async_run_process(lpAVRCP->sCmdNew, 0);
	}
}

LPAVRCP avrcp_new(char* filename)
{
	LPAVRCP lpAVRCP = malloc(sizeof(struct avrcp_data_t));
	if(lpAVRCP) {
		memset(lpAVRCP, 0, sizeof(*lpAVRCP));
		init_uinput();

		// Read config strings
		read_config_string(filename, "a2dpd", "cmdplay", lpAVRCP->sCmdPlay, sizeof(lpAVRCP->sCmdPlay), "");
		read_config_string(filename, "a2dpd", "cmdpause", lpAVRCP->sCmdPause, sizeof(lpAVRCP->sCmdPause), "");
		read_config_string(filename, "a2dpd", "cmdstop", lpAVRCP->sCmdStop, sizeof(lpAVRCP->sCmdStop), "");
		read_config_string(filename, "a2dpd", "cmdprev", lpAVRCP->sCmdPrev, sizeof(lpAVRCP->sCmdPrev), "");
		read_config_string(filename, "a2dpd", "cmdnext", lpAVRCP->sCmdNext, sizeof(lpAVRCP->sCmdNext), "");
		read_config_string(filename, "a2dpd", "cmdnew", lpAVRCP->sCmdNew, sizeof(lpAVRCP->sCmdNew), "");

		// Start listening socket
		lpAVRCP->sockfd = a2dp_make_listen_socket(23);
		DBG("Listening for AVRCP on socket %d", lpAVRCP->sockfd);
	}
	DBG("%p", lpAVRCP);
	return lpAVRCP;
}

void avrcp_destroy(LPAVRCP lpAVRCP)
{
	int i;
	if(lpAVRCP) {
		close_socket(&lpAVRCP->sockfd);
		// Close only incoming sockets, used by certification process
		for(i=0; i<MAXCLIENTS; i++) {
			if(lpAVRCP->new_fd[i]>0) {
				DBG("Closing socket %d", lpAVRCP->new_fd[i]);
				close_socket(&(lpAVRCP->new_fd[i]));
			}
		}
		free(lpAVRCP);
		kill_uinput();
	}
	DBG("%p", lpAVRCP);
}

