/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2008  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#include "dialog.h"
#include "discover.h"

static const gchar *authors[] = {
	"Marcel Holtmann <marcel@holtmann.org>",
	NULL
};

void show_error_dialog(const gchar *message)
{
	GtkWidget *dialog;

	dialog = gtk_message_dialog_new_with_markup(NULL, GTK_DIALOG_MODAL,
				GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, message);

	gtk_dialog_run(GTK_DIALOG(dialog));

	gtk_widget_destroy(dialog);
}

static void open_uri(GtkWindow *parent, const char *uri)
{
	GtkWidget *dialog;
	GdkScreen *screen;
	GError *error = NULL;
	gchar *cmdline;

	screen = gtk_window_get_screen(parent);

	cmdline = g_strconcat("xdg-open ", uri, NULL);

	if (gdk_spawn_command_line_on_screen(screen,
						cmdline, &error) == FALSE) {
		dialog = gtk_message_dialog_new(parent,
			GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_ERROR,
					GTK_BUTTONS_CLOSE, error->message);
		gtk_dialog_run(GTK_DIALOG(dialog));
		gtk_widget_destroy(dialog);
		g_error_free(error);
	}

	g_free(cmdline);
}

static void about_url_hook(GtkAboutDialog *dialog,
					const gchar *url, gpointer data)
{
	open_uri(GTK_WINDOW(dialog), url);
}

static void about_email_hook(GtkAboutDialog *dialog,
					const gchar *email, gpointer data)
{
	char *uri;

	uri = g_strconcat ("mailto:", email, NULL);
	open_uri(GTK_WINDOW(dialog), uri);
	g_free(uri);
}

void show_about_dialog(void)
{
	GtkWidget *dialog;

	dialog = gtk_about_dialog_new();

	gtk_window_set_position(GTK_WINDOW(dialog),
					GTK_WIN_POS_CENTER_ON_PARENT);

	gtk_about_dialog_set_name(GTK_ABOUT_DIALOG(dialog),
						_("Bluetooth Analyzer"));

	gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(dialog), VERSION);

	gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(dialog),
			"Copyright \xc2\xa9 2005-2008 Marcel Holtmann");

	gtk_about_dialog_set_comments(GTK_ABOUT_DIALOG(dialog),
				_("Bluetooth protocol analyzer"));

	gtk_about_dialog_set_logo_icon_name(GTK_ABOUT_DIALOG(dialog),
							"bluetooth");

	gtk_about_dialog_set_url_hook(about_url_hook, NULL, NULL);

	gtk_about_dialog_set_email_hook(about_email_hook, NULL, NULL);

	gtk_about_dialog_set_website(GTK_ABOUT_DIALOG(dialog),
						"http://www.bluez.org");

	gtk_about_dialog_set_website_label(GTK_ABOUT_DIALOG(dialog),
							"www.bluez.org");

	gtk_about_dialog_set_authors(GTK_ABOUT_DIALOG(dialog), authors);

	gtk_about_dialog_set_translator_credits(GTK_ABOUT_DIALOG(dialog),
						_("translator-credits"));

	gtk_dialog_run(GTK_DIALOG(dialog));

	gtk_widget_destroy(dialog);
}

static void add_filters(GtkFileChooser *chooser)
{
	GtkFileFilter *filter;

	filter = gtk_file_filter_new();
	gtk_file_filter_set_name(filter, _("All Files"));
	gtk_file_filter_add_pattern(filter, "*");
	gtk_file_chooser_add_filter(chooser, filter);

	filter = gtk_file_filter_new();
	gtk_file_filter_set_name(filter, _("Supported Files"));
	gtk_file_filter_add_pattern(filter, "*.hcidump");
	gtk_file_filter_add_pattern(filter, "*.btsnoop");
	gtk_file_filter_add_pattern(filter, "*.log");
	gtk_file_filter_add_pattern(filter, "*.pklg");
	gtk_file_chooser_add_filter(chooser, filter);
	gtk_file_chooser_set_filter(chooser, filter);

	filter = gtk_file_filter_new();
	gtk_file_filter_set_name(filter, _("Frontline BTSnoop Files"));
	gtk_file_filter_add_pattern(filter, "*.btsnoop");
	gtk_file_filter_add_pattern(filter, "*.log");
	gtk_file_chooser_add_filter(chooser, filter);

	filter = gtk_file_filter_new();
	gtk_file_filter_set_name(filter, _("Apple Packet Logger Files"));
	gtk_file_filter_add_pattern(filter, "*.pklg");
	gtk_file_chooser_add_filter(chooser, filter);
}

static GtkTreeModel *create_filetype_model(void)
{
	GtkListStore *store;
	GtkTreeIter iter;

	store = gtk_list_store_new(3, G_TYPE_STRING,
					G_TYPE_STRING, G_TYPE_STRING);

	gtk_list_store_insert_with_values(store, &iter, -1,
					0, NULL,
					1, _("By Extension"),
					2, NULL, -1);

	gtk_list_store_insert_with_values(store, &iter, -1,
					0, "btsnoop",
					1, _("Frontline BTSnoop Format"),
					2, "log,btsnoop", -1);

	gtk_list_store_insert_with_values(store, &iter, -1,
					0, NULL,
					1, _("Apple Packet Logger Format"),
					2, "pklg", -1);

	return GTK_TREE_MODEL(store);
}

static void update_filename(GtkFileChooser *chooser, gchar *extensions)
{
	gchar *filename, *basename;
	gchar *lastdot, **extlist;
	GString *text;
	int i;

	filename = gtk_file_chooser_get_filename(chooser);

	extlist = g_strsplit(extensions, ",", 0);
	for (i = 0; i < g_strv_length(extlist); i++)
		if (g_str_has_suffix(filename, *(extlist + i)) == TRUE) {
			g_free(filename);
			g_strfreev(extlist);
			return;
		}

	basename = g_filename_display_basename(filename);

	g_free(filename);

	lastdot = g_strrstr(basename, ".");
	if (lastdot == NULL) {
		g_free(basename);
		g_strfreev(extlist);
		return;
	}

	text = g_string_new(basename);

	g_string_truncate(text, lastdot - basename);

	g_string_append(text, ".");
	g_string_append(text, *extlist);

	g_free(basename);
	g_strfreev(extlist);

	gtk_file_chooser_set_current_name(chooser, text->str);

	g_string_free(text, TRUE);
}

static void changed_callback(GtkTreeSelection *selection,
				GtkTreePath *start_path, GtkTreePath *end_path)
{
	GtkWidget *expander;
	GtkWidget *chooser;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *label, *name, *extlist = NULL;

	if (gtk_tree_selection_get_selected(selection, &model, &iter) == FALSE)
		return;

	gtk_tree_model_get(model, &iter, 1, &name, 2, &extlist, -1);

	label = g_strdup_printf("%s (%s)",
			_("Select File _Type"), name);

	g_free(name);

	if (extlist != NULL) {
		chooser = g_object_get_data(G_OBJECT(selection), "chooser");
		update_filename(GTK_FILE_CHOOSER(chooser), extlist);
		g_free(extlist);
	}

	expander = g_object_get_data(G_OBJECT(selection), "expander");

	gtk_expander_set_label(GTK_EXPANDER(expander), label);

	g_free(label);
}

static GtkWidget *create_filetype_widget(GtkWidget *dialog, GtkWidget *parent)
{
	GtkWidget *scrolled;
	GtkWidget *tree;
	GtkTreeModel *model;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	GtkTreeSelection *selection;
	GtkTreeIter iter;

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled),
								GTK_SHADOW_IN);

	tree = gtk_tree_view_new();
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tree), TRUE);
	gtk_tree_view_set_enable_search(GTK_TREE_VIEW(tree), FALSE);
	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(tree), TRUE);
	gtk_widget_set_size_request(tree, -1, 120);
	gtk_container_add(GTK_CONTAINER(scrolled), tree);

	renderer = gtk_cell_renderer_text_new();
	g_object_set(renderer, "yalign", 0.0, NULL);
	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(tree),
				-1, _("File Type"), renderer, "text", 1, NULL);

	renderer = gtk_cell_renderer_text_new();
	g_object_set(renderer, "yalign", 0.0, NULL);
	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(tree),
				-1, _("Extensions"), renderer, "text", 2, NULL);

	column = gtk_tree_view_get_column(GTK_TREE_VIEW(tree), 0);
	gtk_tree_view_column_set_expand(column, TRUE);

	model = create_filetype_model();
	gtk_tree_view_set_model(GTK_TREE_VIEW(tree), model);
	g_object_unref(model);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);

	g_object_set_data(G_OBJECT(selection), "chooser", dialog);
	g_object_set_data(G_OBJECT(selection), "expander", parent);

	g_signal_connect(G_OBJECT(selection), "changed",
				G_CALLBACK(changed_callback), NULL);

	if (gtk_tree_model_get_iter_first(model, &iter) == TRUE)
		gtk_tree_selection_select_iter(selection, &iter);

	gtk_widget_show_all(scrolled);

	return scrolled;
}

gchar *show_save_dialog(const gchar *currentname)
{
	GtkWidget *dialog;
	GtkWidget *expander;
	GtkWidget *child;
	gchar *filename = NULL;
	gboolean valid = FALSE;

	dialog = gtk_file_chooser_dialog_new(_("Save File"), NULL,
				GTK_FILE_CHOOSER_ACTION_SAVE,
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, NULL);

	gtk_file_chooser_set_local_only(GTK_FILE_CHOOSER(dialog), TRUE);

	gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog),
									TRUE);

	if (currentname != NULL)
		valid = gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dialog),
								currentname);

	if (valid == FALSE)
		gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog),
							"protocol-trace.pklg");

	add_filters(GTK_FILE_CHOOSER(dialog));

	expander = gtk_expander_new_with_mnemonic(NULL);
	gtk_expander_set_spacing(GTK_EXPANDER(expander), 6);
	gtk_widget_show(expander);

	child = create_filetype_widget(dialog, expander);
	gtk_container_add(GTK_CONTAINER(expander), child);

	gtk_file_chooser_set_extra_widget(GTK_FILE_CHOOSER(dialog), expander);

	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
		filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));

	gtk_widget_destroy(dialog);

	return filename;
}

gchar *show_open_dialog(void)
{
	GtkWidget *dialog;
	gchar *filename = NULL;

	dialog = gtk_file_chooser_dialog_new(_("Open File"), NULL,
				GTK_FILE_CHOOSER_ACTION_OPEN,
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);

	gtk_file_chooser_set_local_only(GTK_FILE_CHOOSER(dialog), TRUE);

	gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialog), FALSE);

	add_filters(GTK_FILE_CHOOSER(dialog));

	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
		filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));

	gtk_widget_destroy(dialog);

	return filename;
}

static void local_callback(GtkWidget *button, gpointer user_data)
{
	GtkWidget *dialog = user_data;
	gchar *hostname;

	hostname = g_strdup("localhost");
	g_object_set_data(G_OBJECT(dialog), "hostname", hostname);
}

static void remote_callback(GtkWidget *button, gpointer user_data)
{
	GtkWidget *dialog = user_data;
	GtkWidget *entry;
	gchar *hostname;
	gboolean active;

	entry = g_object_get_data(G_OBJECT(dialog), "entry");

	active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));

	hostname = gtk_combo_box_get_active_text(GTK_COMBO_BOX(entry));
	g_object_set_data(G_OBJECT(dialog), "hostname", hostname);

	gtk_widget_set_sensitive(entry, active);
}

static void hostname_callback(GtkWidget *editable, gpointer user_data)
{
	GtkWidget *dialog = user_data;
	gchar *hostname;

	hostname = gtk_combo_box_get_active_text(GTK_COMBO_BOX(editable));
	g_object_set_data(G_OBJECT(dialog), "hostname", hostname);
}

static gchar *last_hostname = NULL;

static gboolean separator_callback(GtkTreeModel *model,
					GtkTreeIter *iter, gpointer user_data)
{
	gchar *value;
	gboolean result;

	gtk_tree_model_get(model, iter, 0, &value, -1);

	result = (value == NULL) ? TRUE : FALSE;

	g_free(value);

	return result;
}

static GtkWidget *create_host_selector(GtkWidget *dialog, GtkTreeModel *model)
{
	GtkWidget *vbox;
	GtkWidget *entry;
	GtkWidget *button;
	GSList *group = NULL;

	vbox = gtk_vbox_new(FALSE, 6);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 12);

	button = gtk_radio_button_new_with_label(group,
					_("Local connection"));
	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	g_signal_connect(G_OBJECT(button), "toggled",
				G_CALLBACK(local_callback), dialog);

	button = gtk_radio_button_new_with_label(group,
					_("Remote connection"));
	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	g_signal_connect(G_OBJECT(button), "toggled",
				G_CALLBACK(remote_callback), dialog);

	entry = gtk_combo_box_entry_new();
	gtk_combo_box_set_model(GTK_COMBO_BOX(entry), model);
	gtk_combo_box_set_row_separator_func(GTK_COMBO_BOX(entry),
					separator_callback, NULL, NULL);
	gtk_combo_box_entry_set_text_column(GTK_COMBO_BOX_ENTRY(entry), 0);
	gtk_widget_set_sensitive(entry, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);

	g_signal_connect(G_OBJECT(entry), "changed",
				G_CALLBACK(hostname_callback), dialog);

	g_object_set_data(G_OBJECT(dialog), "entry", entry);

	g_object_set_data(G_OBJECT(dialog), "hostname", "localhost");

	gtk_widget_show_all(vbox);

	return vbox;
}

gchar *show_host_dialog(void)
{
	GtkWidget *dialog;
	GtkWidget *child;
	GtkTreeModel *model;
	gchar *hostname = NULL;

	dialog = gtk_dialog_new_with_buttons(_("Open Import"), NULL,
				GTK_DIALOG_NO_SEPARATOR,
				GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
				GTK_STOCK_CONNECT, GTK_RESPONSE_ACCEPT, NULL);

	gtk_dialog_set_default_response(GTK_DIALOG(dialog),
						GTK_RESPONSE_ACCEPT);

	model = create_discover(last_hostname);

	child = create_host_selector(dialog, model);

	gtk_container_set_border_width(GTK_CONTAINER(dialog), 6);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), child);

	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {
		hostname = g_object_get_data(G_OBJECT(dialog), "hostname");
		if (hostname != NULL &&
				g_ascii_strcasecmp(hostname, "") &&
				g_ascii_strcasecmp(hostname, "localhost")) {
			g_free(last_hostname);
			last_hostname = g_strdup(hostname);
		}
	}

	gtk_widget_destroy(dialog);

	destroy_discover(model);

	return g_strdup(hostname);
}

gchar *show_close_dialog(void)
{
	GtkWidget *dialog;
	gint response;
	gchar *text;

	text = g_strdup_printf("<big><b>%s</b></big>\n\n%s",
				_("Save the protocol trace before closing?"),
				_("If you don't save, "
				"all information will be permanently lost."));

	dialog = gtk_message_dialog_new_with_markup(NULL, GTK_DIALOG_MODAL,
				GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE, text);

	g_free(text);

	gtk_dialog_add_button(GTK_DIALOG(dialog), _("Close _without Saving"),
							GTK_RESPONSE_CLOSE);

	gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL,
							GTK_RESPONSE_CANCEL);

	gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_SAVE,
							GTK_RESPONSE_OK);

	response = gtk_dialog_run(GTK_DIALOG(dialog));

	gtk_widget_destroy(dialog);

	if (response == GTK_RESPONSE_CANCEL ||
				response == GTK_RESPONSE_DELETE_EVENT)
		return NULL;

	if (response == GTK_RESPONSE_OK)
		return g_strdup("save");

	return g_strdup("close");
}

gboolean show_quit_dialog(guint unsaved)
{
	GtkWidget *dialog;
	gint response;
	gchar *text;

	text = g_strdup_printf("<big><b>%s %d %s</b></big>\n\n%s",
			unsaved == 1 ? _("There is") : _("There are"), unsaved,
			unsaved == 1 ? _("unsaved protocol trace.") :
						_("unsaved protocol traces."),
			_("If you quit now, all information will be lost."));

	dialog = gtk_message_dialog_new_with_markup(NULL, GTK_DIALOG_MODAL,
				GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE, text);

	g_free(text);

	gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL,
							GTK_RESPONSE_CANCEL);

	gtk_dialog_add_button(GTK_DIALOG(dialog), _("_Discard Changes"),
							GTK_RESPONSE_CLOSE);

	response = gtk_dialog_run(GTK_DIALOG(dialog));

	gtk_widget_destroy(dialog);

	if (response == GTK_RESPONSE_CANCEL ||
				response == GTK_RESPONSE_DELETE_EVENT)
		return FALSE;

	return TRUE;
}
