#ifndef BMP_LOGGER_HPP
#define BMP_LOGGER_HPP

#include <map>
#include <fstream>
#include <string>
#include <glibmm/ustring.h>
#include <glibmm/thread.h>

#include <boost/shared_ptr.hpp>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xmlreader.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>


#define LOG_ALL_LEVELS                (GLogLevelFlags (G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION))
#define BMP_LOGGER_DEFAULT_LOG_LEVEL  (GLogLevelFlags (G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_INFO))

/* default log file max size: 512kb */
#define BMP_LOGGER_FILE_MAX_SIZE      (std::size_t (1) << 19)

namespace Bmp
{
  class Log
  {
      public:

        Log (Glib::ustring const &domain,
             const std::string   &filename,
             GLogLevelFlags       flags);

        ~Log ();

        void
        log_to_file (const char     *domain,
                     GLogLevelFlags  flags,
                     const char     *message);

      private:

        Log (const Log &original);
        const Log & operator = (const Log &original);

        std::ofstream   logfile;
        Glib::Mutex     file_mutex;
        Glib::ustring   domain;
        std::string     filename;
        unsigned int    id;

        static void
        log_to_file_wrap (const char     *domain,
                          GLogLevelFlags  flags,
                          const char     *message,
                          void           *data);
  };

  class Logger
  {
      public:

        Logger ();

        void
        add_log (Glib::ustring const &domain,
                 const std::string   &filename,
                 GLogLevelFlags       flags);

      private:

        Logger (const Logger &original);
        const Logger & operator = (const Logger &original);

        std::map<Glib::ustring, boost::shared_ptr<Bmp::Log> > logs;
  };

} // Bmp

#endif // BMP_LOGGER_HPP
