//-*- Mode: C++; indent-tabs-mode: nil; -*-
//
//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2 as 
//  published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#include <boost/shared_ptr.hpp>

#include <glibmm.h>
#include <glib.h>
#include <fam.h>
#include <utility>

#include "monitor.hh"

namespace Bmp
{
  namespace FileMonitor
  {
    Monitor::Watch::Watch (std::string const&   monitor_name,
                           std::string const&   path_name,
                           FamFunc const       &func_added,
                           FamFunc const       &func_removed) : monitor_name  (monitor_name),
                                                                path_name     (path_name),
                                                                func_added    (func_added),
                                                                func_removed  (func_removed)
    {
        FAMOpen (&fc);
        thread = Glib::Thread::create (sigc::mem_fun (this, &Watch::watch_thread), true);
        termination.lock ();
        terminate = false;
    }

    Monitor::Watch::~Watch ()
    {
        terminate_watch ();
        FAMClose (&fc);
    }

    void
    Monitor::Watch::terminate_watch ()
    {
        termination.unlock ();
        thread->join ();
        termination.unlock ();
    }

    void
    Monitor::Watch::watch_thread ()
    {
        FAMRequest fr;
        FAMMonitorDirectory (&fc, path_name.c_str(), &fr, 0);

        while (true)
        {
            Glib::Thread *collector = Glib::Thread::create (sigc::mem_fun (this, &Watch::monitor_thread), true);
            collector->join();

            for (EventList::const_iterator i = events.begin () ; i != events.end (); ++i)
            {
                switch (i->second)
                {
                    case FAMCreated:
                        func_added (i->first);
                        break;

                    case FAMDeleted:
                        func_removed (i->first);
                        break;
                    default: break;
                }
            }

            events.clear();

            if (terminate)
              {
                FAMCancelMonitor (&fc, &fr);
                return;
              }
        }
    }

    void
    Monitor::Watch::monitor_thread ()
    {
        FAMEvent fe;
        bool ev = false;

        while (true)
        {
            Glib::usleep (50000);
            switch (FAMPending (&fc))
            {
              case 1:
              {
                FAMNextEvent (&fc, &fe);
                switch (fe.code)
                {
                  case FAMCreated:
                  case FAMDeleted:
                  case FAMMoved:
                  {
                    events.push_back (std::make_pair (Glib::build_filename (path_name, fe.filename), fe.code));
                    ev = true;
                    break;
                  }
                  default: break;
                }
              }
            }

            terminate = termination.trylock();

            if (terminate || ev)
              break;
        }
    }

    Monitor::Monitor ()
    {
        FAMConnection fc;

        if (FAMOpen (&fc) == -1)
          {
            monitor_running = false;
            g_message ("Running: Monitor (FAM daemon not running: Not accepting watches!)");
          }
        else
          {
            monitor_running = true;
            g_message ("Running: Monitor");
            FAMClose (&fc);
          }
    }

    Monitor::~Monitor ()
    {
      watches.clear();
    }

    bool
    Monitor::add_watch (std::string const& monitor_name,
                        std::string const& path_name,
                        FamFunc const& func_added,
                        FamFunc const& func_removed)
    {
        if (!monitor_running)
          {
            g_message (G_STRLOC
                        ": Not accepting monitor '%s' for path '%s': FAM daemon not running.",
                        monitor_name.c_str(),
                        path_name.c_str());
            return false;
          }

        if (watches.find (monitor_name) != watches.end())
          {
            g_message (G_STRLOC ": Monitor with name '%s' exists", monitor_name.c_str());
            return false;
          }

        watches[monitor_name] = TWatchPtr (new Bmp::FileMonitor::Monitor::Watch (monitor_name, path_name, func_added, func_removed));
        return true;
    }

    bool
    Monitor::stop_watch (std::string const& monitor_name)
    {
        TWatchPtr watch; 

        watch = watches.find(monitor_name)->second;
        if (!watch)
          return false;

        watch->terminate_watch();
        watches.erase (monitor_name);
        return true;
    }

  } //FileMonitor
} //Bmp
