//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <boost/format.hpp>
#include <boost/algorithm/string.hpp>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xmlreader.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include <glib/gstdio.h>
#include <glibmm.h>
#include <glibmm/i18n.h>
#include <glibmm/markup.h>
#include <iostream>
#include <fstream>

#include "main.hh"
#include "paths.hh"
#include "util.hh"
#include "util-string.hh"
#include "uri++.hh"
#include "podcast.hh"
#include "podcast-utils.hh"
#include "podcast-libxml2-sax.hh"
#include "x_vfs.hh"
#include "minisoup.hh"
using namespace Glib;
using namespace Gtk;

namespace
{
  const char * A_URL = "url";
  const char * A_LENGTH = "length";
  const char * A_IS_PERMA_LINK = "isPermaLink";
  const char * A_TYPE = "type";

  static boost::format time_f ("%llu");

  std::string
  get_uuid (Bmp::PodcastBackend::Podcast const& cast)
  {
    return Bmp::Util::md5_hex_string (cast.uri.c_str(), strlen(cast.uri.c_str())); 
  }
}

namespace Bmp
{
  namespace PodcastBackend
  {
    PodcastManager::PodcastManager ()
    {
      std::string filename (build_filename (BMP_PATH_USER_DIR, "feedlist.opml"));
      if (file_test (filename, FILE_TEST_EXISTS))
      {
        try {
            OPMLParser parser (m_casts, *this);
            Markup::ParseContext context (parser);
            std::string data (file_get_contents (filename)); 
            context.parse (data);
            context.end_parse ();
            if (!parser.check_sanity ())
            {
              throw Bmp::PodcastBackend::ParsingError();
            }
          }
        catch (MarkupError & cxe)
          {
            g_critical ("%s: Catch exception: %s", G_STRLOC, cxe.what().c_str());
          }
        catch (std::exception & cxe)
          {
            g_critical ("%s: Catch exception: %s", G_STRLOC, cxe.what());
          }
      }
    }

    void
    PodcastManager::podcast_overlay_save (Podcast const& cast)
    {
      xmlDocPtr doc = xmlNewDoc (BAD_CAST "1.0");
      xmlNodePtr ovrl = xmlNewNode (NULL, BAD_CAST "overlay");
      xmlSetProp (ovrl, BAD_CAST "version", BAD_CAST "1.0"); 

      xmlDocSetRootElement (doc, ovrl);

      for (PodcastItemsMap::const_iterator i = cast.items.begin() ; i != cast.items.end() ; ++i)
      {
        PodcastItem const& item (i->second);

        xmlNodePtr o = xmlNewNode (NULL, BAD_CAST "cast-item");
        xmlAddChild (ovrl, o);

        xmlSetProp (o, BAD_CAST "item-uid",
                       BAD_CAST item.uid_value.c_str()); 

        xmlSetProp (o, BAD_CAST "played",
                       BAD_CAST (item.played ? "1" : "0"));

        xmlSetProp (o, BAD_CAST "downloaded",
                       BAD_CAST (item.downloaded ? "1" : "0"));

        xmlSetProp (o, BAD_CAST "filename",
                       BAD_CAST (item.downloaded ? (filename_to_utf8 (item.filename).c_str()) : ""));
      }

      xmlKeepBlanksDefault (0);
      xmlChar * data;
      int size;
      xmlDocDumpFormatMemoryEnc (doc, &data, &size, "UTF-8", 1);

      std::ofstream o (cast_overlay_filename (cast).c_str());
      o << data;
      o.close ();

      xmlFreeDoc (doc);
      g_free (data);
    }

    void
    PodcastManager::save_overlays ()
    {
      for (PodcastMap::const_iterator i = m_casts.begin() ; i != m_casts.end() ; ++i)
      {
        Podcast const& cast (i->second);
        podcast_overlay_save (cast);
      }
    }

    void
    PodcastManager::save_state ()
    {
      save_opml (build_filename (BMP_PATH_USER_DIR, "feedlist.opml"));
      save_overlays ();
    }

    void
    PodcastManager::save_opml (std::string const& filename)
    {
      xmlDocPtr doc = xmlNewDoc (BAD_CAST "1.0");
        
      xmlNodePtr opml = xmlNewNode (NULL, BAD_CAST "opml");
      xmlNodePtr head = xmlNewNode (NULL, BAD_CAST "head");
      xmlNodePtr body = xmlNewNode (NULL, BAD_CAST "body");

      xmlSetProp (opml, BAD_CAST "version", BAD_CAST "1.0"); 
      xmlNewTextChild (head, NULL, BAD_CAST "title", BAD_CAST "BMP Podcast List");
      xmlDocSetRootElement (doc, opml);

      xmlAddChild (opml, head);
      xmlAddChild (opml, body);

      for (PodcastMap::const_iterator i = m_casts.begin() ; i != m_casts.end() ; ++i)
      {
        Podcast const& cast (i->second);
        xmlNodePtr o = xmlNewNode (NULL, BAD_CAST "outline");
        xmlAddChild (body, o);

        xmlSetProp (o, BAD_CAST "text",
                       BAD_CAST  cast.title.c_str());

        xmlSetProp (o, BAD_CAST "title",
                       BAD_CAST  cast.title.c_str());

        xmlSetProp (o, BAD_CAST "type",  
                       BAD_CAST "rss");

        xmlSetProp (o, BAD_CAST "description", 
                       BAD_CAST  cast.description.c_str());

        xmlSetProp (o, BAD_CAST "htmlUrl", 
                       BAD_CAST  cast.link.c_str()); 

        xmlSetProp (o, BAD_CAST "xmlUrl", 
                       BAD_CAST  cast.uri.c_str()); 

        xmlSetProp (o, BAD_CAST "updateInterval", 
                       BAD_CAST (time_f % cast.ttl).str().c_str()); 

        xmlSetProp (o, BAD_CAST "id", 
                       BAD_CAST  cast.cast_uuid.c_str()); 

        xmlSetProp (o, BAD_CAST "lastPollTime", 
                       BAD_CAST  (time_f % cast.last_poll_time).str().c_str()); 

        xmlSetProp (o, BAD_CAST "lastFaviconPollTime", 
                       BAD_CAST "0"); 

        xmlSetProp (o, BAD_CAST "sortColumn", 
                       BAD_CAST "time"); 
      }

      xmlKeepBlanksDefault (0);
      xmlChar * data;
      int size;
      xmlDocDumpFormatMemoryEnc (doc, &data, &size, "UTF-8", 1);

      std::ofstream o (filename.c_str());
      o << data;
      o.close ();

      xmlFreeDoc (doc);
      g_free (data);
    }

    PodcastManager::~PodcastManager ()
    {
      save_state ();
    }

    void
    PodcastManager::podcast_delete (ustring const& uri)
    {
      Podcast const& cast (podcast_fetch (uri));
      g_unlink (cast_filename (cast).c_str());
      g_unlink (cast_overlay_filename (cast).c_str());
      if (!cast.image_url.empty())
      {
        g_unlink (cast_image_filename (cast).c_str());
      }
      m_casts.erase (uri);
    }

    void
    PodcastManager::podcast_get_list (PodcastList & list)
    {
      for (PodcastMap::const_iterator i = m_casts.begin() ; i != m_casts.end() ; ++i)
      {
        list.push_back (i->second.uri);
      }
    }

    Podcast const&
    PodcastManager::podcast_fetch (ustring const& uri) const
    {
      return m_casts.find (uri)->second;
    }

    void
    PodcastManager::podcast_load (Podcast & cast)
    {
      PodcastOverlayItemsMap overlays;

      try{
        if (file_test (cast_overlay_filename (cast), FILE_TEST_EXISTS))
        {
          try{
              PodcastOverlayParseContext context (cast, overlays);
              PodcastOverlayParser p (context);
              Markup::ParseContext c (p);

              c.parse (file_get_contents (cast_overlay_filename (cast)));
              c.end_parse ();

              if (!p.check_sanity ())
              {
                static boost::format message (_("Error parsing overlay for item %s"));
                throw PodcastBackend::ParsingError((message % cast.cast_uuid.c_str()).str());
              }
            }
         catch (ConvertError & cxe)
            {
              throw PodcastBackend::ParsingError(cxe.what()); 
            }
         catch (MarkupError & cxe)
            {
              throw PodcastBackend::ParsingError(cxe.what()); 
            }
        }

        std::string filename = cast_filename (cast);
        if (!file_test (filename, FILE_TEST_EXISTS))
        {
          podcast_cache (cast.uri, cast.cast_uuid);
        }
        else
        {
          if (podcast_rss2_parse (overlays, cast, file_get_contents (filename)) < 0)
          {
            static boost::format err (_("Error parsing RSS from item %s"));
            g_warning ("%s: %s", G_STRLOC, (err % cast.cast_uuid.c_str()).str().c_str()); 
            throw PodcastBackend::ParsingError((err % cast.cast_uuid.c_str()).str());
          }
        }
      }
     catch (URI::ParseError & cxe)
        {
          throw PodcastBackend::PodcastInvalidError(cxe.what()); 
        }
     catch (VFS::Exception & cxe)
        {
          throw PodcastBackend::NetworkError(cxe.what()); 
        }
     catch (ConvertError & cxe)
        {
          throw PodcastBackend::ParsingError(cxe.what()); 
        }
    }

    void
    PodcastManager::podcast_update (ustring const& uri)
    {
      Podcast cast (m_casts.find (uri)->second);

      g_unlink (cast_filename (cast).c_str());
      if (!cast.image_url.empty())
      {
        std::string filename (cast_image_filename (cast));
        g_unlink (filename.c_str());
      }

      try {
        podcast_overlay_save (cast);
        podcast_cache (cast.uri, cast.cast_uuid);
        }
      catch (...)
        {
          m_casts.erase (m_casts.find (uri));
          m_casts.insert (std::make_pair (uri, cast));
          throw;
        }
    }

    void
    PodcastManager::podcast_cache (ustring const& uri, std::string const& uuid_)
    {
      Podcast cast;
      cast.uri = uri;
      cast.last_poll_time = time (NULL);

      PodcastOverlayItemsMap overlays;
      if (uuid_.empty())
      {
        if (m_casts.find (uri) != m_casts.end())
        {
          g_warning ("%s: Warning, podcasts '%s' exists", G_STRLOC, uri.c_str());
          Podcast const& cast = podcast_fetch (uri);
          podcast_cache (uri, cast.cast_uuid);
          return;
        }
        cast.cast_uuid = get_uuid (cast);
      }
      else
      {
        m_casts.erase (uri);
        cast.cast_uuid = uuid_;
        if (file_test (cast_overlay_filename (cast), FILE_TEST_EXISTS))
        {
          try{
              PodcastOverlayParseContext context (cast, overlays);
              PodcastOverlayParser p (context);
              Markup::ParseContext c (p);
              c.parse (file_get_contents (cast_overlay_filename (cast)));
              c.end_parse ();
              if (!p.check_sanity ())
              {
                static boost::format overlay_error_f ("Error Parsing Podcast Overlay %s"); 
                std::string err ((overlay_error_f % cast.cast_uuid).str());
                g_warning ("%s: %s", G_STRLOC, err.c_str());
                throw PodcastBackend::ParsingError (err);
              }
            }
         catch (ConvertError & cxe)
            {
              throw PodcastBackend::ParsingError(cxe.what()); 
            }
         catch (MarkupError & cxe)
            {
              throw PodcastBackend::ParsingError(cxe.what()); 
            }
          }
      }

      std::string data; 

      try{

          Bmp::URI u;
          try{
            u = Bmp::URI (uri, true);
          }
          catch (URI::ParseError & cxe)
          {
            g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "RSS: Invalid URI '%s': %s", uri.c_str(), cxe.what()); 
            throw PodcastBackend::InvalidUriError(cxe.what()); 
          }

          Soup::RequestSyncRefP request = Soup::RequestSync::create (ustring (u));
          guint code = request->run ();

          if (code != 200)
          {
            throw PodcastBackend::ParsingError(_("Bad HTTP Server Reply"));
          }
            
          data = request->get_data (); 
          if (data.empty())
            throw PodcastBackend::ParsingError(_("Empty data in Reply"));

          if (podcast_rss2_parse (overlays, cast, data) < 0)
          {
            static boost::format rss_parse_error_f (_("Error parsing RSS from URI %s"));
            std::string _err ((rss_parse_error_f % uri.c_str()).str());
            g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "RSS: %s", _err.c_str());
            throw Bmp::PodcastBackend::ParsingError(_err);
          }
        }
     catch (URI::ParseError & cxe)
        {
          g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "RSS: Invalid URI '%s': %s", uri.c_str(), cxe.what()); 
          throw PodcastBackend::PodcastInvalidError(cxe.what()); 
        }
     catch (ConvertError & cxe)
        {
          g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "RSS: Convert Error: %s", cxe.what().c_str()); 
          throw PodcastBackend::ParsingError(cxe.what()); 
        }
      catch (std::exception & cxe)
        {
          g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "RSS: Stdexception: %s", cxe.what()); 
          throw PodcastBackend::ParsingError(cxe.what()); 
        }

      std::ofstream o (cast_filename (cast).c_str()); 
      o << data.c_str();  
      o.close ();

      if (!cast.image_url.empty())
      {
        g_log (G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Getting image from '%s'", cast.image_url.c_str());
        RefPtr<Gdk::Pixbuf> cast_image = Util::get_image_from_uri (cast.image_url);
        if (cast_image)
          cast_image->save (cast_image_filename (cast), "png");
        else
          cast.image_url = ustring();
      }
      m_casts.insert (std::make_pair (uri, cast));
    }

    void
    PodcastManager::podcast_item_change (ustring     const& uri,
                                         ustring     const& uid,
                                         PodcastItem const& item)
    {
      Podcast & cast (m_casts.find (uri)->second);
      PodcastItemsMap::iterator i (cast.items.find (uid));
      if (i != cast.items.end())
      {
        cast.items.erase (i);
        cast.items.insert (std::make_pair (uid, item));
      }
    }

    //////////////////////////////////////////////////////////////////////////////////////////////

#define STATE(e) ((m_state & e) != 0)
#define SET_STATE(e) ((m_state |= e))
#define CLEAR_STATE(e) ((m_state &= ~ e))

    bool
    OPMLParser::check_sanity  ()
    {
      if (m_state)
      {
        g_warning (G_STRLOC ": State should be 0, but is %d", m_state);
        return false;
      }
      return true;
    }

    OPMLParser::OPMLParser (PodcastMap & casts, PodcastManager & manager)
    : m_casts   (casts)
    , m_manager (manager)
    , m_state   (0)
    {
    }

    OPMLParser::~OPMLParser () 
    {
    }
 
    void
    OPMLParser::on_start_element  (Markup::ParseContext  & context,
                                   ustring          const& name,
                                   AttributeMap     const& attributes)
	  {
      if (name == "opml") 
      {
        SET_STATE(E_OPML);
        return;
      }

      if (name == "head") 
      {
        SET_STATE(E_HEAD);
        return;
      }

      if (name == "body") 
      {
        SET_STATE(E_BODY);
        return;
      }

      if (name == "outline") 
      {
        SET_STATE(E_OUTLINE);

        if (attributes.find ("xmlUrl") != attributes.end())
        {
          try{
              Podcast cast (attributes.find ("id")->second);
              cast.uri = attributes.find ("xmlUrl")->second; 
              m_manager.podcast_load (cast);
              if (attributes.find ("lastPollTime") != attributes.end())
              {
                cast.last_poll_time = strtoull (attributes.find ("lastPollTime")->second.c_str(), NULL, 10);
              }
              m_casts.insert (std::make_pair (cast.uri, cast)); 
            }
          catch (Bmp::PodcastBackend::ParsingError & cxe)
            {}
        }
        return;
      }
    }

    void
    OPMLParser::on_end_element    (Markup::ParseContext    & context,
                                   ustring            const& name)
	  {
      if (name == "opml") 
      {
        CLEAR_STATE(E_OPML);
        return;
      }

      if (name == "head") 
      {
        CLEAR_STATE(E_HEAD);
        return;
      }

      if (name == "body") 
      {
        CLEAR_STATE(E_BODY);
        return;
      }

      if (name == "outline") 
      {
        CLEAR_STATE(E_OUTLINE);
        return;
      }
	  }

    void
    OPMLParser::on_text       (Markup::ParseContext    & context,
                               ustring            const& text)
    {
    }

    void
    OPMLParser::on_passtrough (Markup::ParseContext  & context,
                               ustring          const& text) {}

    void
    OPMLParser::on_error      (Markup::ParseContext  & context,
                               MarkupError      const& error) {}


    //////////////////////////////////////////////////////////////////////////////////////////////

    bool
    PodcastOverlayParser::check_sanity  ()
    {
      if (m_state)
      {
        g_warning (G_STRLOC ": State should be 0, but is %d", m_state);
        return false;
      }
      return true;
    }

    PodcastOverlayParser::PodcastOverlayParser (PodcastOverlayParseContext & context)
    : m_context (context),
      m_state   (0)
    {
    }

    PodcastOverlayParser::~PodcastOverlayParser () 
    {
    }
 
    void
    PodcastOverlayParser::on_start_element  (Markup::ParseContext& context,
                                             ustring const& name,
                                             AttributeMap const& attributes)
	  {
      if (name == "overlay")
      {
        SET_STATE(E_BMP_CAST_OVERLAY);
        return;
      }

      if (name == "cast-item")
      {
        SET_STATE(E_CAST_ITEM);
        PodcastOverlayItem item;

        // Fill item with overlay attributes
        int _i = 0;

        _i = g_ascii_strtoull (attributes.find ("played")->second.c_str(), NULL, 10);
        item.played = ((_i > 0) ? true : false); 

        _i = g_ascii_strtoull (attributes.find ("downloaded")->second.c_str(), NULL, 10);
        item.downloaded =  ((_i > 0) ? true : false); 

        if (item.downloaded)
        {
          item.filename = filename_from_utf8 (attributes.find ("filename")->second);
          item.downloaded = file_test (item.filename, FILE_TEST_EXISTS);
        } 
   
        m_context.m_overlays.insert (std::make_pair (attributes.find ("item-uid")->second, item));
        return;
      }
    }

    void
    PodcastOverlayParser::on_end_element    (Markup::ParseContext& context,
                                             ustring const& name)
	  {
      if (name == "overlay")
      {
        CLEAR_STATE(E_BMP_CAST_OVERLAY);
        return;
      }

      if (name == "cast-item")
      {
        CLEAR_STATE(E_CAST_ITEM);
        return;
      }
	  }

    void
    PodcastOverlayParser::on_text       (Markup::ParseContext& context,
                                         ustring const& text) {}

    void
    PodcastOverlayParser::on_passtrough (Markup::ParseContext& context,
                                         ustring const& text) {}

    void
    PodcastOverlayParser::on_error      (Markup::ParseContext& context,
                                         MarkupError const& error) {}
  }
}
