//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_SHELL_HH
#define BMP_SHELL_HH

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <map>
#include <string>

#include <glib/gtypes.h>
#include <glib/ghash.h>
#include <glibmm/main.h>
#include <glibmm/ustring.h>
#include <gdkmm/pixbuf.h>
#include <gdkmm/pixbufanimation.h>
#include <gtkmm/box.h>
#include <gtkmm/label.h>
#include <gtkmm/scale.h>
#include <gtkmm/notebook.h>
#include <gtkmm/statusbar.h>
#include <gtkmm/toggleaction.h>
#include <gtkmm/uimanager.h>
#include <gtkmm/window.h>
#include <libglademm/xml.h>
#include <sigc++/connection.h>
#include <sigc++/signal.h>

#include <boost/shared_ptr.hpp>
#include <boost/optional.hpp>
#include "mcs/mcs.h"

// BMP Widgets
#include "widgets/bmp_tray_icon.h"
#include "widgets/bmp_status_icon.h"
#include "widgets/button.hh"
#include "widgets/popup.hh"

// BMP Types
#include "bmp/base-types.hh"
#include "bmp/ui-types.hh"

// BMP Dialogs
#include "dialog-about.hh"
#include "dialog-equalizer.hh"
#include "dialog-simple-progress.hh"
#include "dialog-track-details.hh"

#include "playbacksource.hh"
#include "preferences.hh"
#include "vfs.hh"

#include "lastfm-types.hh"

#include "bookmark-manager.hh"

using namespace Bmp::LastFM;

namespace Bmp
{
  class BlackBar;
  class Scroll;

  enum PlaybackSourceID
  {
    SOURCE_NONE         = -1,
    SOURCE_LIBRARY      =  0,
    SOURCE_PLAYLIST     =  1,
    SOURCE_RADIO        =  2,
    SOURCE_LASTFM       =  3,
    SOURCE_PODCASTS     =  4,
    SOURCE_CDDA         =  5,
    SOURCE_JAMENDO      =  6,

    N_SOURCES           =  7,
  };

  typedef sigc::signal<void, int> SignalSourceSelected;

  class SourceSelection
    : public Gtk::VBox
  {
    private:
      SignalSourceSelected  source_selected_;
    public:

      SourceSelection (BaseObjectType                        *  obj,
                       Glib::RefPtr<Gnome::Glade::Xml>  const&  xml);
      virtual ~SourceSelection () {};

      void
      source_select   (PlaybackSourceID source);

      void
      source_activate (PlaybackSourceID source);

      SignalSourceSelected&
      source_selected ()
      {
        return source_selected_;
      }

      PlaybackSourceID      m_active_source;
      PlaybackSourceID      m_selected_source;

   protected:

      virtual bool
      on_expose_event (GdkEventExpose * event);

    private:

      typedef std::map<PlaybackSourceID, SourceButton*> ButtonsMap;
      typedef std::map<PlaybackSourceID, unsigned int>  SourceNotebookTabMap;

      void  source_button_clicked (PlaybackSourceID source);

      ButtonsMap            m_source_selection_buttons_map;
      SourceNotebookTabMap  m_notebook_tab_map;
  };

  class PlayerShell
    : public Gtk::Window
  {
    protected:

      virtual bool on_delete_event (GdkEventAny *event);
      virtual bool on_key_press_event (GdkEventKey*);
      virtual bool on_window_state_event (GdkEventWindowState *event);

    public:

      class Player;

    private:

      Player * m_player_dbus_obj; // MPRIS /Player object

      typedef boost::shared_ptr<PlaybackSource> SourceSP;

      SourceSelection           * m_selection;
      SourceSP                    m_sources[N_SOURCES];
      PlaybackSource::Caps        m_source_caps[N_SOURCES];
      PlaybackSource::Flags       m_source_flags[N_SOURCES];
      Glib::Mutex                 m_source_state_lock;
      Glib::Mutex                 m_metadata_mutex;
      Glib::Mutex                 m_playback_mutex;
      TrackMetadata               m_metadata;
      opt_stdstring               m_asin;

      GHashTable * get_metadata_hash_table ();

      /////////////// SCROBBLING
      void  shell_lastfm_scrobble_current ();
      void  shell_lastfm_now_playing ();

      /////////////// PLAYBACK CONTROLS
      void  play ();
      void  pause ();
      void  prev ();
      void  next ();
      void  stop ();

      /////////////// PLAYBACK ENGINE HANDLERS 
      void  on_play_status_changed ();
      void  on_play_position (int position);
      void  on_play_buffering (double buffered);
      void  on_play_error (Glib::ustring const& element, Glib::ustring const& uri,
              Glib::ustring const& debug, GError * error, GstElement const* src);
      void  on_play_eos ();

      /////////////// SEEKING
      void  on_seek_range_changed ();
      bool  on_seek_event (GdkEvent *event);
      void  reset_seek_range ();

      /////////////// SOURCE HANDLING 
      void  connect_source_signals        (PlaybackSourceID source);
      bool  source_restore_context        (GdkEventButton*);
      void  source_changed                (int n);
      void  on_source_caps                (PlaybackSource::Caps  caps,  PlaybackSourceID source);
      void  on_source_flags               (PlaybackSource::Flags flags, PlaybackSourceID source);
      void  on_source_segment             (PlaybackSourceID source);
      void  on_source_stop                (PlaybackSourceID source);
      void  on_source_next                (PlaybackSourceID source);
      void  on_source_track_metadata (TrackMetadata const& metadata,
              PlaybackSourceID source, BmpMetadataParseFlags flags = PARSE_FLAGS_ALL);
      void  on_source_message_set (Glib::ustring const& message);
      void  on_source_message_clear ();
      void  on_source_play_request (PlaybackSourceID source);

      /////////////// METADATA HANDLING
      void  on_gst_metadata (BmpGstMetadataField field);
      bool  metadata_update_when_idle (BmpMetadataParseFlags flags);
      void  reparse_metadata (BmpMetadataParseFlags flags = PARSE_FLAGS_ALL);
      void  set_current_source_image ();

      /////////////// SIMPLE PROGRESS
      void  progress_start (int n_items, const char * title);
      void  progress_step ();
      void  progress_end (bool wait_for_close);

      /////////////// BOOKMARKS API
      BookmarkManager m_bookmark_manager;
      void  on_bookmark_loaded (Bookmark const& bookmark);
      void  on_bookmark_create ();
      void  on_bookmark_activate (std::string const& url);
      void  bookmark_append (Bookmark const& bookmark, uint64_t bookmark_id);

      /////////////// STATUSICON/POPUP
      static void create_popup  (GtkPlug *plug, gpointer userdata);
      static void status_icon_click (BmpStatusIcon *icon, gpointer data);
      static void status_icon_scroll_up (BmpStatusIcon *icon, gpointer data);
      static void status_icon_scroll_down (BmpStatusIcon *icon, gpointer data);
      static void status_icon_popup_menu (BmpStatusIcon *icon, guint arg1, guint arg2, gpointer data);
      static gboolean status_icon_enter (BmpStatusIcon *icon, GdkEventCrossing *event, gpointer data);
      static gboolean status_icon_leave (BmpStatusIcon *icon, GdkEventCrossing *event, gpointer data);
      bool  display_popup_delayed ();

      /////////////// DISPLAY AUXILLIARIES 
      void  display_preferences ();
      void  display_track_details ();
      void  display_lastfm_recommend ();
      void  display_eq ();
      void  on_toggle_statusbar ();
      void  on_toggle_compact ();
      void  on_toggle_menubar ();
      void  status_clear_message ();
      bool  on_black_bar_event (GdkEvent* /*ev*/);

      ////////////// LAST.FM DEPENDENT STUFF
      void  on_last_fm_radio_connected ();
      void  on_last_fm_radio_disconnected ();

      void  on_scrobbler_connected ();
      void  on_scrobbler_disconnected ();

      /////////////// MISC
      void  on_volume_changed (MCS_CB_DEFAULT_SIGNATURE);
      void  play_artist_on_lastfm ();

      Glib::RefPtr<Gnome::Glade::Xml>     m_ref_xml;
      Glib::RefPtr<Gtk::ActionGroup>      m_actions;
      Glib::RefPtr<Gdk::Pixbuf>           m_source_icons[N_SOURCES];
      Glib::RefPtr<Gdk::Pixbuf>           m_source_icons_small[N_SOURCES];
      Glib::RefPtr<Gdk::Pixbuf>           m_source_icons_med[N_SOURCES];
      Glib::RefPtr<Gdk::Pixbuf>           yy_stopped;
      Glib::RefPtr<Gdk::Pixbuf>           yy_seeking;
      Glib::RefPtr<Gdk::PixbufAnimation>  yy_playing;
      Glib::RefPtr<Gdk::PixbufAnimation>  yy_paused;

      Glib::RefPtr<Gtk::UIManager>        m_ui_manager;
      guint                               m_ui_merge_id, m_ui_merge_id_context, m_ui_merge_id_tray;

      Glib::RefPtr<Gdk::Pixbuf> m_status_icon_image_default;
      Glib::RefPtr<Gdk::Pixbuf> m_status_icon_image_paused;
      Glib::RefPtr<Gdk::Pixbuf> m_status_icon_image_playing;

      Gtk::Image*         m_yy_image;
      Gtk::Label*         m_time_label;
      Gtk::HScale*        m_seek_range;
      Scroll*             m_volume;
      Popup*              m_popup;
      BmpStatusIcon*      m_status_icon;
      SimpleProgress*     m_simple_progress;
      Gtk::Widget*        m_menu_bar;
      AboutDialog*        m_about;
      Preferences*        m_prefs;
      Equalizer*          m_equalizer;
      Gtk::Notebook*      m_notebook_main;
      BlackBar*           m_blackBar;
      TrackDetails*       m_track_details;
      Gtk::Statusbar*     m_statusbar;
  
      guint64 m_n_status_messages;

      guint64 m_n_progress_items;
      guint64 m_nth_progress_item;

      bool  m_visible;
      bool  m_spm_paused;

      enum SeekDirection
      {
        M_SEEK_FORWARD,
        M_SEEK_BACKWARD
      };
      SeekDirection m_seek_direction;
      bool m_seeking;

      sigc::connection delay_popup;

    public:
#include "exception.hh"
      EXCEPTION(InvalidSourceError)
      EXCEPTION(NoMetadataError)

      PlayerShell (BaseObjectType*                        cobject,
                   Glib::RefPtr<Gnome::Glade::Xml> const& xml);
      static PlayerShell* create ();
      void init ();
      void raise ();
      virtual ~PlayerShell ();
      unsigned int status_push_message (Glib::ustring const& message);
  };
}
#endif
