//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm/i18n.h>
#include <glib/gstdio.h>
#include <gtkmm.h>
#include <gdk/gdkkeysyms.h>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <sstream>
#include <fstream>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>
#include <boost/shared_ptr.hpp>

#include <mcs/mcs.h>

// BMP Musicbrainz
#include "musicbrainz/mbxml-v2.hh"
#include "mb-tagger.hh"

// BMP Audio
#include "audio/audio.hh"

// BMP Widgets
#include "widgets/taskdialog.hh"

// BMP Misc
#include "dialog-export.hh"
#include "dialog-progress.hh"
#include "dialog-simple-progress.hh"

#include "ui-tools.hh"

#include "debug.hh"
#include "main.hh"
#include "paths.hh"
#include "stock.hh"

#include "uri++.hh"
#include "util.hh"
#include "util-file.hh"

#include "x_amazon.hh"
#ifdef HAVE_HAL
#  include "x_hal.hh"
#endif //HAVE_HAL
#include "x_library.hh"
#include "x_mcsbind.hh"
#include "x_play.hh"
#include "x_vfs.hh"

#include "ui-part-library.hh"

#define KEY_PRESS_MAX_TIME 0.1
#define MIN_SEARCH_CHARS 3 

using namespace std;
using namespace boost;
using boost::algorithm::trim;

using namespace Gtk;
using namespace Gdk;
using namespace Glib;

using namespace Bmp::DB;
using namespace Bmp::Util;
using namespace Bmp::VFS;
using namespace Bmp::Audio;
using namespace Bmp::MusicBrainzXml;

#define ACTION_RETAG_TRACKS               "library-action-retag-tracks"
#define ACTION_PLAYLIST_EXPORT            "library-action-playlist-export"
#define ACTION_HISTORY_CLEAR              "library-action-history-clear"
#define ACTION_DELETE_FILES               "library-action-delete-files"

#define ACTION_SEARCH_SEARCH_ALL              "action-search-search-all"
#define ACTION_SEARCH_SEARCH_ARTIST           "action-search-search-artist"
#define ACTION_SEARCH_SEARCH_ALBUM            "action-search-search-album"
#define ACTION_SEARCH_SEARCH_TITLE            "action-search-search-title"
#define ACTION_SEARCH_SEARCH_ALBUM_ARTIST     "action-search-search-album-artist"
#define ACTION_SEARCH_SEARCH_GENRE            "action-search-search-genre"

namespace
{
  const char * library_menu_playlist =
  "<ui>"
  ""
  "<menubar name='popup-library-playlist'>"
  ""
  "   <menu action='dummy' name='menu-library-playlist'>"
  "     <menuitem action='" ACTION_RETAG_TRACKS "'/>"
  "       <separator name='sep560'/>"
  "     <menuitem action='" ACTION_HISTORY_CLEAR "'/>"
  "       <separator name='sep561'/>"
  "     <menuitem action='" ACTION_DELETE_FILES "'/>"
  "     <menuitem action='" ACTION_PLAYLIST_EXPORT "'/>"
  "   </menu>"
  ""
  "</menubar>"
  ""
  "</ui>";

  const char * library_search_popup =
  "<ui>"
  ""
  "<menubar name='popup-library-search'>"
  ""
  "   <menu action='dummy' name='menu-library-search'>"
  "     <menuitem action='" ACTION_SEARCH_SEARCH_ALL "'/>"
  "       <separator name='sep600'/>"
  "     <menuitem action='" ACTION_SEARCH_SEARCH_ARTIST "'/>"
  "     <menuitem action='" ACTION_SEARCH_SEARCH_ALBUM "'/>"
  "     <menuitem action='" ACTION_SEARCH_SEARCH_TITLE "'/>"
  "     <menuitem action='" ACTION_SEARCH_SEARCH_ALBUM_ARTIST "'/>"
  "     <menuitem action='" ACTION_SEARCH_SEARCH_GENRE "'/>"
  "   </menu>"
  ""
  "</menubar>"
  ""
  "</ui>";

  typedef std::vector < TargetEntry > DNDEntries;

  static boost::format int_f ("%d");
  static boost::format uint64_f ("%llu");
  static boost::format progress_f ("%d / %d");

  void
  menu_item_set_markup (RefPtr<UIManager> uimanager,
                        ustring const&  menupath,
                        ustring const&  markup)
  {
    Bin * bin = 0;
    bin = dynamic_cast <Bin *> (uimanager->get_widget (menupath));

    if( bin )
      reinterpret_cast <Gtk::Label *> (bin->get_child())->set_markup (markup);
    else
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "Widget with path '%s' not found or not a Gtk::Bin", menupath.c_str());
  }

  enum DnDTypes
  {
    DND_URI_TARGETS = 0,
    DND_ALBUM       = 1,
    DND_TRACK       = 2,
  };

  //// Various functors for for_each

  using namespace Bmp;

  template <class T>
  class TrackApply
  {
    public:

      TrackApply (T const& column, Track const& track) : m_track (track), m_column (column) {} 

      void
      operator()(TreeIter const& i)
      {
        (*i)[m_column] = m_track;
      }

    private:
    
      Track const& m_track;
      T const& m_column;
  };  

  template <class T>
  class PathCollect
  {
    public:

      PathCollect ( Glib::RefPtr<Gtk::TreeModel> const& model,
                    T const& column,
                    TrackV & tracks)
      : m_model   (model)
      , m_tracks  (tracks)
      , m_column  (column)
      {} 

      void
      operator()(Gtk::TreePath const& p)
      {
        m_tracks.push_back ((*m_model->get_iter (p))[m_column]);
      }

    private:
        
      Glib::RefPtr<Gtk::TreeModel> const& m_model; 
      TrackV & m_tracks;
      T const& m_column;
  };  

  class ReferenceCollect
  {
    public:

      ReferenceCollect (Glib::RefPtr<Gtk::TreeModel> const& model,
                        Bmp::ReferenceV & references)
      : m_model       (model)
      , m_references  (references)
      {} 

      void
      operator()(Gtk::TreePath const& p)
      {
        m_references.push_back (TreeRowReference (m_model, p));
      }

    private:
        
      Glib::RefPtr<Gtk::TreeModel> const& m_model; 
      Bmp::ReferenceV & m_references;
  };  

  std::string
  get_date_string_markup (std::string const& in)
  {
    int y = 0, m = 0, d = 0;

    const char * z (in.c_str());

    if( strlen (z) == 4 )
      sscanf (z, "%04d", &y);
    else
#if 0
    if( strlen (z) == 6 )
      sscanf (z, "%04d%02d", &y, &m);
    else
    if( strlen (z) == 7 )
      sscanf (z, "%04d-%02d", &y, &m);
    else
#endif
    if( strlen (z) == 8 )
      sscanf (z, "%04d%02d%02d", &y, &m, &d);
    else
    if( (strlen (z) == 10) ||  (strlen (z) == 19) )
      sscanf (z, "%04d-%02d-%02d", &y, &m, &d);

    char result[1024];

    struct tm * _tm = g_new0 (struct tm, 1);

    if (y) _tm->tm_year = (y - 1900);
    if (m) _tm->tm_mon  = m - 1;
    if (d) _tm->tm_mday = d;

    /*
    if( y && m && d )
    {
      char ys[256];
      char ms[256];
      char ds[256];
      strftime (ys, 255, "%Y", _tm);
      strftime (ms, 255, "%B", _tm);
      strftime (ds, 255, "%d", _tm);
      g_free (_tm);
      std::string r = (boost::format ("%s <small>(%s %s)</small>")
                          % std::string (ys)
                          % std::string (ds)
                          % Markup::escape_text (locale_to_utf8 (ustring (ms))).c_str()).str();
      return r;
    }
    else
    if( y && m )
    {
      char ys[256];
      char ms[256];
      strftime (ys, 255, "%Y", _tm);
      strftime (ms, 255, "%B", _tm);
      g_free (_tm);
      std::string r = (boost::format ("%s <small>(%s)</small>")
                          % std::string (ys)
                          % Markup::escape_text (locale_to_utf8 (ustring (ms))).c_str()).str();
      return r;
    }
    else
    */
    if( y )
    {
      char ys[256];
      strftime (ys, 255, "(%Y)", _tm);
      g_free (_tm);
      return std::string (ys); 
    }

    return std::string();
  }

  template <class T>
  inline T
  clamp (T min, T max, T val)
  {
    if( (val >= min) && (val <= max) )
    {
      return val;
    }
    else if( val < min )
    {
      return min;
    }
    else
    {
      return max;
    }
  }

  enum Renderer
  {
    R_TEXT,
    R_PIXBUF,
    R_TOGGLE,
    R_SURFACE
  };
}

namespace Bmp
{
      ViewArtists::ViewArtists (BaseObjectType                 *  obj,
                                RefPtr<Gnome::Glade::Xml> const&  xml)
      : TreeView (obj)
      {
        TreeViewColumn * column = manage (new Gtk::TreeViewColumn ());
        column->set_resizable (false);
        column->set_expand (false);

        {
          Gtk::CellRendererPixbuf * cell = manage (new CellRendererPixbuf());
          cell->property_xpad() = 4;
          column->pack_start (*cell, false);
          column->set_cell_data_func (*cell, (sigc::bind (sigc::mem_fun (*this, &Bmp::ViewArtists::cell_data_func), 0)));
        }

        {
          Gtk::CellRendererText * cell = manage (new CellRendererText());
          cell->property_xpad() = 0;
          cell->property_ellipsize () = Pango::ELLIPSIZE_END;
          column->pack_start (*cell, true);
          column->set_cell_data_func (*cell, (sigc::bind (sigc::mem_fun (*this, &Bmp::ViewArtists::cell_data_func), 1)));
        }

        append_column (*column);

        m_store = TreeStore::create (m_artist_cr);
        m_store->set_default_sort_func (sigc::mem_fun (*this, &Bmp::ViewArtists::default_sort_func));
        m_store->set_sort_column (Gtk::TreeSortable::DEFAULT_SORT_COLUMN_ID, Gtk::SORT_ASCENDING);

        get_selection()->set_mode (SELECTION_SINGLE);
        conn_changed = get_selection()->signal_changed().connect (sigc::mem_fun (*this, &Bmp::ViewArtists::on_selection_changed));
      }

      void
      ViewArtists::on_row_activated (Gtk::TreeModel::Path const&, Gtk::TreeViewColumn*)
      {
        signal_activated_.emit ();
      }

      void
      ViewArtists::cell_data_func (CellRenderer * basecell, TreeModel::iterator const& iter, int cell)
      {
        CellRendererText * cell_t = 0;
        CellRendererPixbuf * cell_p = 0;

        switch (cell)
        {
          case 0:
            cell_p = dynamic_cast <CellRendererPixbuf*>(basecell);
            break;

          case 1:
            cell_t = dynamic_cast <CellRendererText*>(basecell);
            break;
        }

        if( iter == m_store->children().begin() )
        {
          switch (cell)
          {
            case 0:
              cell_p->property_pixbuf() = render_icon (Gtk::StockID (BMP_STOCK_ARTIST_ALL),
                    Gtk::ICON_SIZE_SMALL_TOOLBAR)->scale_simple (16, 16, Gdk::INTERP_BILINEAR);
              break;

            case 1:
              if( m_UidIterMap.size() )
              {
                uint64_t n_entries = uint64_t (gtk_tree_model_iter_n_children (GTK_TREE_MODEL (m_store->gobj()), NULL)-1);
                cell_t->property_markup() = ((boost::format (_("<b>%llu %s</b>"))
                        % n_entries 
                        % ((n_entries != 1) ? _("Artists") : _("Artist"))
                        ).str());
              }
              else
              {
                cell_t->property_markup() = _("<b>Nothing Found</b>"); 
              }
              break;
          }
          return;
        }

        NodeType node = (*iter)[m_artist_cr.type];
        switch (node)
        {
          case NODE_BRANCH:
          {
            switch (cell) 
            {
              case 0:
                cell_p->property_pixbuf() = render_icon (Gtk::StockID (BMP_STOCK_ARTIST_MULTIPLE),
                Gtk::ICON_SIZE_SMALL_TOOLBAR)->scale_simple (16, 16, Gdk::INTERP_BILINEAR);
                return;

              case 1:
                cell_t->property_text() = (*iter)[m_artist_cr.value];
                return;
            }
            break;
          }

          case NODE_LEAF:
          {
            Bmp::AlbumArtist const& artist = (*iter)[m_artist_cr.artist];
            switch (cell)
            {
              case 0:
              {
                cell_p->property_pixbuf() = render_icon ((artist.is_mb_album_artist ? Gtk::StockID (BMP_STOCK_ARTIST_MB) 
                : Gtk::StockID (BMP_STOCK_ARTIST)), Gtk::ICON_SIZE_SMALL_TOOLBAR)->scale_simple (16, 16, Gdk::INTERP_BILINEAR);
                break;
              }

              case 1:
              {
                cell_t->property_text() = ustring (artist);
                break;
              }
            }
            break;
          }
        }
      }

      int
      ViewArtists::default_sort_func (Gtk::TreeModel::iterator const& iter_a,
                                      Gtk::TreeModel::iterator const& iter_b)
      {
        // NOTE: for some reason, using brackets doesn't work:
        //   ustring str_a (Bmp::AlbumArtist ((*iter_a)[m_artist_cr.artist]));
        //   ustring str_b (Bmp::AlbumArtist ((*iter_b)[m_artist_cr.artist]));
        // gcc bug? or do the statements have some ambiguity I don't know about?
        // - Des

        //ustring str_a = Bmp::AlbumArtist ((*iter_a)[m_artist_cr.artist]);
        //ustring str_b = Bmp::AlbumArtist ((*iter_b)[m_artist_cr.artist]);
        
        if( iter_a ==  m_store->children().begin() )
        {
          return -1; // anything else (iter_b) is always "smaller"
        }

        if( iter_b ==  m_store->children().begin() )
        {
          return 1; // anything else (iter_a) is always "larger"
        }
        
        ustring str_a = (*iter_a)[m_artist_cr.value];
        ustring str_b = (*iter_b)[m_artist_cr.value];
        return str_a.lowercase().compare (str_b.lowercase());
      }

      void
      ViewArtists::clear ()
      {
        unset_model ();
        m_store->clear ();

        m_UidIterMap.clear ();
        m_index_map.clear();

        TreeIter i = m_store->append ();
        (*i)[m_artist_cr.type] = NODE_LEAF; // not really, but we handle all artist this way

        set_model (m_store);
        on_selection_changed ();
      }

      void 
      ViewArtists::put_artist (Bmp::AlbumArtist const& artist)
      {
        std::string trimmed = ustring (artist);
        boost::algorithm::trim (trimmed);

        if( trimmed.empty() )
        {
          return;
        }
        
        ustring         a = Util::utf8_string_normalize (trimmed); 
        ustring         x = a.casefold (); 
        IndexMapIter    n = m_index_map.find (x);
        TreeIter        i;

        if( n != m_index_map.end() )
        {
          TreeIter const& i_toplevel = n->second;

          NodeType node = (*i_toplevel)[m_artist_cr.type];

          (*i_toplevel)[m_artist_cr.type] = NODE_BRANCH; 
          (*i_toplevel)[m_artist_cr.value] = a; 

          if( node != NODE_BRANCH )
          {
            // Clone this node as one child row
            UID uid (Bmp::AlbumArtist ((*i_toplevel)[m_artist_cr.artist]).bmpx_album_artist_id);
            UidIterMapIter imi = m_UidIterMap.find (uid);
        
            if( imi != m_UidIterMap.end() )
            {
              m_UidIterMap.erase (imi);
            }

            i = m_store->append (i_toplevel->children());
            (*i)[m_artist_cr.value] = ustring ((*i_toplevel)[m_artist_cr.value]);
            (*i)[m_artist_cr.artist] = Bmp::AlbumArtist ((*i_toplevel)[m_artist_cr.artist]);
            (*i)[m_artist_cr.uid] = UID ((*i_toplevel)[m_artist_cr.uid]);
            (*i)[m_artist_cr.type] = NODE_LEAF; 
            m_UidIterMap.insert (std::make_pair (uid, i));
          }

          i = m_store->append (i_toplevel->children());
        }
        else
        {
          if( m_UidIterMap.find (artist.bmpx_album_artist_id) == m_UidIterMap.end() )
          {
            i = m_store->append ();
            m_index_map.insert (IndexMapPair (x, i)); 
          }
          else
          {
            return;
          } 
        }

        (*i)[m_artist_cr.value] = trimmed; 
        (*i)[m_artist_cr.artist] = artist;
        (*i)[m_artist_cr.uid] = artist.bmpx_album_artist_id;
        (*i)[m_artist_cr.type] = NODE_LEAF; 
        m_UidIterMap.insert (std::make_pair (artist.bmpx_album_artist_id, i));
      }

      void
      ViewArtists::display ()
      {
        if( !m_update_mutex.trylock() )
          return;

        conn_changed.block (true);

        // Clear the Model
        unset_model ();
        m_store->clear ();

        // Clear state (NOT the shared state)
        m_UidIterMap.clear ();
        m_index_map.clear();

        // Create the "All" node
        TreeIter i = m_store->append ();
        (*i)[m_artist_cr.type] = NODE_LEAF; // not really, but we handle all artist this way

        m_state->lock ();
        bool empty = m_state->r_album_artist_j.empty();
        m_state->unlock ();

        // Process shared state
        if( empty )
        {
          RowV r = library->get_artists();
          for (RowV::const_iterator i = r.begin(); i != r.end(); ++i)
          {
            put_artist (*i);
          }
        }
        else
        {
          m_state->lock ();
          RowV rows = m_state->r_album_artist_j;
          m_state->unlock ();

          for (RowV::const_iterator a = rows.begin(); a != rows.end(); ++a)
          {
            if( m_state->m_reset.trylock() )
            {
              if( m_state->m_querying )
              {
                m_store->clear ();
                m_UidIterMap.clear ();
                m_index_map.clear ();
                m_state->m_reset.unlock();
                return;
              }
              m_state->m_reset.unlock();
            }

            AlbumArtist artist (library->get_artist (boost::get <uint64_t> (a->find ("album_artist_j")->second))[0]);
            put_artist (artist);
          }
        }

        set_model (m_store);
        columns_autosize ();

        conn_changed.block (false);
        get_selection()->select (m_store->children().begin());

        m_update_mutex.unlock ();
      }

      void
      ViewArtists::on_selection_changed ()
      {
        if( get_selection()->count_selected_rows() == 0 )
        {
          signal_artist_cleared_.emit ();
        }
        else
        if( get_selection()->get_selected() == m_store->children().begin() )
        {
          uint64_t n_entries = uint64_t (gtk_tree_model_iter_n_children (GTK_TREE_MODEL (m_store->gobj()), NULL)-1);
          signal_artist_all_selected_.emit (n_entries);
        }
        else
        {
          UidSet        s;
          TreeIter      i = get_selection()->get_selected();
          NodeType      type = (*i)[m_artist_cr.type];

          switch (type)
          {
            case NODE_LEAF:
            {
              AlbumArtist aa ((*i)[m_artist_cr.artist]);
              s.insert (aa.bmpx_album_artist_id);
              break;
            }

            case NODE_BRANCH:
            {
              Gtk::TreeModel::Row row = *i;
              for (TreeNodeChildren::const_iterator n = row.children().begin(); n != row.children().end(); ++n)
              {
                s.insert (UID ((*n)[m_artist_cr.uid]));
              }
              break;
            }
          }
          signal_artist_selected_.emit (s, (*i)[m_artist_cr.value]);
        }
      }
}

namespace Bmp
{
    UidList
    ViewAlbums::get_albums ()
    {
      return m_selected_uids;
    }

    ViewAlbums::ViewAlbums (BaseObjectType                 *  obj,
                            RefPtr<Gnome::Glade::Xml> const&  xml)
    : TreeView            (obj)
    , m_ref_xml           (xml)
    , m_all_artists       (0)
    {
      m_cover = Util::cairo_image_surface_from_pixbuf (Pixbuf::create_from_file (build_filename
            (BMP_IMAGE_DIR,BMP_COVER_IMAGE_DEFAULT))->scale_simple (52, 52, Gdk::INTERP_BILINEAR));

      TreeViewColumn * column = 0; 
      {
        column = manage (new TreeViewColumn ());
        CellRendererCairoSurface * cell = manage (new CellRendererCairoSurface());
        cell->property_xalign() = 1.0; 
        cell->property_yalign() = 0.0;
        cell->property_xpad() = 0;
        cell->property_ypad() = 2;
        cell->set_fixed_size (60, 56);

        column->pack_start (*cell, false);
        column->set_resizable (false);
        column->set_expand (false);
        column->set_sizing (TREE_VIEW_COLUMN_FIXED);
        column->set_fixed_width (60);
        column->add_attribute (*cell, "surface", m_album_cr.cover);
        append_column (*column);
      }

      {
        column = manage (new TreeViewColumn (_("Title")));
        CellRendererText * cell = manage (new CellRendererText());
        cell = manage (new CellRendererText());
        cell->property_xalign() = 0.0;
        cell->property_yalign() = 0.5;
        cell->property_xpad() = 4;
        cell->property_ypad() = 2;
        cell->property_ellipsize () = Pango::ELLIPSIZE_END;
        column->pack_start (*cell, true);
        column->set_resizable (true);
        column->set_expand (false);
        column->set_min_width (150);
        column->set_cell_data_func (*cell, (sigc::mem_fun (*this, &Bmp::ViewAlbums::album_cell_data_func)));
        append_column (*column);
      }

      m_store = ListStore::create (m_album_cr);
      set_model (m_store);
      get_selection()->set_mode (SELECTION_MULTIPLE);
      get_selection()->set_select_function (sigc::mem_fun (*this, &Bmp::ViewAlbums::slot_select));
      set_search_column (m_album_cr.searchKey);
    }

    void
    ViewAlbums::album_cell_data_func (Gtk::CellRenderer * basecell, Gtk::TreeModel::iterator const& iter)
    {
      CellRendererText * cell = dynamic_cast <CellRendererText *>(basecell);

      Album const& a = (*iter)[m_album_cr.album];
      AlbumArtist aa (a.m_row);

      static char * const format_f ("<b>%s</b>\n%s %s");

      std::string album;
      if( a.album )
      {
        album = a.album.get(); 
        trim (album);
      }

      cell->property_markup() = (sql_uprintf (format_f, (Markup::escape_text (ustring (aa)).c_str()),
                                                        (Markup::escape_text (ustring (album)).c_str()),
                                                        (a.mb_release_date ? get_date_string_markup (a.mb_release_date.get()).c_str()
                                                                           : "")));
    }

    void
    ViewAlbums::clear ()
    {
      m_uid_album_map.clear();
      m_selected_uids.clear ();
      m_store->clear ();
    }

    void
    ViewAlbums::change ()
    {
      signal_changed_.emit ();
    }

    bool
    ViewAlbums::slot_select (Glib::RefPtr <Gtk::TreeModel> const& model, Gtk::TreePath const& path, bool was_selected)
    {
      TreeIter iter (m_store->get_iter (path));
      UID uid ((*iter)[m_album_cr.uid]);
  
      UidList _tmp = m_selected_uids;

      if( was_selected )
      {
        UidList::iterator i = m_selected_uids.begin();
        for ( ; i != m_selected_uids.end(); ++i)
        {
          if ( *i == uid) break;
        }

        if( i != m_selected_uids.end() )
        {
          m_selected_uids.erase (i);
        }
        else
        {
          g_message ("%s: System claims Path %d was selected, but it's not on our list!", G_STRLOC, path.get_indices().data()[0]);
        }
      }
      else
      {
        m_selected_uids.push_back (uid);
      }

      if( _tmp != m_selected_uids )
      {
        signal_changed_.emit ();
      }

      return true;
    }

    void
    ViewAlbums::on_artist_cleared ()
    {
      m_state->lock ();
      m_state->m_new_artist = 1;
      m_state->unlock ();

      clear ();
      signal_changed_.emit ();
    }

    void
    ViewAlbums::on_artist_all_selected (uint64_t n_artists)
    {
      m_state->lock ();
      m_state->m_new_artist = 1;
      m_state->unlock ();
 
      m_all_artists = 1;
      unset_model ();

      clear ();
      uint64_t n_albums = 0;

      if( m_state->r_album_artist_j.empty() )
      {
        RowV r = library->get_albums ();
        for (RowV::const_iterator i = r.begin(); i != r.end(); ++i)
        {
          insert (*i);
          ++n_albums;
        }
      }
      else
      {
        m_state->lock ();
        RowV rows = m_state->r_album_j;
        m_state->unlock ();

        for (RowV::const_iterator a = rows.begin(); a != rows.end(); ++a)
        {
          if( m_state->m_reset.trylock() )
          {
            if( m_state->m_querying )
            {
              clear ();
              m_state->m_reset.unlock();
              return;
            }
            m_state->m_reset.unlock();
          }

          Row::const_iterator i = a->find ("album_j");
          if( i != a->end() )
          {
            RowV r = library->get_album (boost::get <uint64_t> (i->second));
            if( r.size() )
            {
              Album album (r[0]);
              insert (album);
              ++n_albums;
            }
          }
        }
      }

      set_model (m_store);
      columns_autosize ();
      signal_changed_.emit ();
    }
 
    void
    ViewAlbums::on_artist_selected (UidSet const& x, ustring const& name)
    {
      m_state->lock ();
      m_state->m_new_artist = 1;
      bool empty = m_state->i_attrs.empty();
      m_state->unlock ();

      m_all_artists = 0;
      unset_model ();
      clear ();

      uint64_t n_albums = 0;

      for (UidSet::const_iterator xx = x.begin(); xx != x.end(); ++xx)
      {
        if( m_state->m_reset.trylock() )
        {
          if( m_state->m_querying )
          {
            clear ();
            m_state->m_reset.unlock();
            return;
          }
          m_state->m_reset.unlock();
        }

        if( !empty )
        {
          Query q;
          q.set_prefix (" DISTINCT ");
          q.set_suffix (/*" GROUP BY mb_album_artist_sort_name, mb_album_artist, mb_release_date, album*/" ORDER BY ifnull(mb_album_artist_sort_name, mb_album_artist), ifnull(mb_release_date, album) ");

          for (AttributeVV::const_iterator i = m_state->i_attrs.begin(); i != m_state->i_attrs.end(); ++i)
          {
            AttributeV a = *i;
            q.add_attr_v (a, CHAIN_OR);
            q.seq_chain_append (CHAIN_AND);
          }

          AttributeV a;
          a.push_back (Attribute (EXACT, "album_artist_j", boost::get<uint64_t>(*xx))); 
          q.add_attr_v (a);
          q.seq_chain_append (CHAIN_AND);
          q.set_columns (" album_j ");
          
          RowV r;
          library->query (q, r, true);

          for (RowV::const_iterator a = r.begin(); a != r.end(); ++a)
          {
            Row::const_iterator i = a->find ("album_j");
            if( i != a->end() )
            {
              RowV r = library->get_album (boost::get <uint64_t> (i->second));
              if( r.size() )
              {
                Album album (r[0]);
                insert (album);
                ++n_albums;
              }
            }
          }
        }
        else
        {
          RowV r = library->get_albums_by_artist (*xx);
          for (RowV::const_iterator a = r.begin(); a != r.end(); ++a)
          {
            insert (*a);
            ++n_albums;
          }
        } 
      }

      set_model (m_store);
      columns_autosize ();
      signal_changed_.emit ();
    }

    void
    ViewAlbums::insert (Bmp::Album const& album)
    {
      if( !album.album )
      {
        g_warning ("%s: No album title set for album id %llu", G_STRLOC, album.bmpx_album_id);
        return;
      }

      TreeModel::iterator iter (m_store->append());
      m_uid_album_map.insert (std::make_pair (album.bmpx_album_id, iter));

      (*iter)[m_album_cr.uid] = album.bmpx_album_id;
      (*iter)[m_album_cr.album] = album;
      (*iter)[m_album_cr.searchKey] = album.album.get(); 

      UidList uid_list;
      uid_list.push_back (album.bmpx_album_id);
      get_cover (uid_list);
    }

    void
    ViewAlbums::get_cover (UidList const& uid_list)
    {
      for (UidList::const_iterator i_uid = uid_list.begin(); i_uid != uid_list.end(); ++i_uid)
      {
        UidIterMap::iterator i (m_uid_album_map.find (*i_uid));
        if( i != m_uid_album_map.end() )
        {
          TreeModel::iterator const& iter (i->second);
          Album const& album = (*iter)[m_album_cr.album];

          if( album.asin )
          {
            try{
              ::Cairo::RefPtr< ::Cairo::ImageSurface> surface = amazon->fetch (album.asin.get(), COVER_SIZE_ALBUM_LIST, true);
              cairo_image_surface_border (surface, 1.);
              (*iter)[m_album_cr.cover] = surface; 
              return;
            }
            catch (...) {}
          }
          (*iter)[m_album_cr.cover] = m_cover;
        }
        else
          g_warning ("%s: Iter for uid '%llu' was not found in the map!", G_STRLOC, *i_uid);
      }
    }

    void
    ViewAlbums::on_row_activated (Gtk::TreeModel::Path const& __attribute__ ((unused)), Gtk::TreeViewColumn* __attribute__ ((unused)) )
    {
      signal_activated_.emit ();
    }
}

namespace
{
    char const * gradient[] =
    {
      NULL, /* dummy for 0 rating */
      "#EDB315",
      "#10FF10",
      "#FF1010"
    };

    enum Column
    {
      COLUMN_PLAYING,
      COLUMN_NEW_ITEM,
/*
      COLUMN_LIBRARY_TRACK,
*/
      COLUMN_ACTIVE,
      COLUMN_TRACK,
      COLUMN_TITLE,
      COLUMN_TIME,
      COLUMN_ALBUM,
      COLUMN_ARTIST,
      COLUMN_GENRE,
      COLUMN_BITRATE,
      COLUMN_SAMPLERATE,
      COLUMN_RATING,
      COLUMN_COUNT,
/*
      COLUMN_UID,
      COLUMN_UID_OK,
*/
    };
}

namespace Bmp
{
    ViewPlaylist::ViewPlaylist (BaseObjectType                 *  obj,
                                RefPtr<Gnome::Glade::Xml> const&  xml)

      : TreeView            (obj)
      , m_ref_xml           (xml)
      , m_localUid          (0)
      , m_playing           (0)
      , m_bmpx_track_id     (0)
    {
      m_pb_playing = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_STOCK, "play.png"));

      struct {
          char const* title;
          double      xalign;
          Renderer    r;
          int         column;
      } cells[] = {
          { " ",              0.5,   R_PIXBUF,   COLUMN_PLAYING },
          { " ",              0.5,   R_TOGGLE,   COLUMN_ACTIVE  },
          { N_("Title"),      0.0,   R_TEXT,     COLUMN_TITLE   },
          { N_("Artist"),     0.0,   R_TEXT,     COLUMN_ARTIST  },
          { N_("Length"),     1.0,   R_TEXT,     COLUMN_TIME    },
          { N_("Album"),      0.0,   R_TEXT,     COLUMN_ALBUM   },
          { N_("Track"),      1.0,   R_TEXT,     COLUMN_TRACK   },
          { N_("Genre"),      0.0,   R_TEXT,     COLUMN_GENRE   },
      };

      for (unsigned int n = 0 ; n < G_N_ELEMENTS (cells); ++n)
      {
        TreeView::Column * c = manage (new TreeView::Column (_(cells[n].title)));
        CellRenderer     * r = 0;

        switch (cells[n].r)
        {
          case R_SURFACE:
            r = manage (new CellRendererCairoSurface());
            r->property_xpad() = 0; 
            c->set_fixed_width (64);
            c->set_sizing (Gtk::TREE_VIEW_COLUMN_FIXED);
            c->pack_end (*r);
            break;

          case R_PIXBUF:
            r = manage (new CellRendererPixbuf());
            r->property_xalign() = cells[n].xalign;
            c->pack_end (*r);
            c->set_min_width (24);
            break;

          case R_TEXT:
            r = manage (new CellRendererText());
            r->property_xalign() = cells[n].xalign;
            c->pack_end (*r);
            c->set_resizable (true);
            break;

          case R_TOGGLE:
            CellRendererToggle * _r = manage (new CellRendererToggle());
            c->pack_end (*_r, false);
            c->set_resizable (false);
            _r->signal_toggled().connect (sigc::mem_fun (*this, &Bmp::ViewPlaylist::cell_play_track_toggled));
            r = _r;
            break;
        }

        append_column (*c);
        c->set_cell_data_func (*r, (sigc::bind (sigc::mem_fun (*this, &Bmp::ViewPlaylist::cell_data_func), cells[n].column, cells[n].r)));
      }

#ifdef HAVE_HAL
      // Connect Bmp::HAL
      if( hal->is_initialized() )
      {
        hal->signal_volume_added().connect
          (sigc::mem_fun (*this, &Bmp::ViewPlaylist::hal_volume_add));
        hal->signal_volume_removed().connect
          (sigc::mem_fun (*this, &Bmp::ViewPlaylist::hal_volume_del));
      }
#endif //HAVE_HAL

      get_selection()->set_select_function
        (sigc::mem_fun (*this, &Bmp::ViewPlaylist::slot_select));

      ::library->signal_track_modified().connect
        (sigc::mem_fun (*this, &Bmp::ViewPlaylist::on_library_track_modified));

      m_store = Gtk::ListStore::create (m_track_cr);
      clear_current_iter ();
      set_model (m_store);
      get_selection()->set_mode (SELECTION_MULTIPLE);
      set_search_column (m_track_cr.searchKey);
    }

    void
    ViewPlaylist::on_row_activated (Gtk::TreeModel::Path const& path, Gtk::TreeViewColumn* G_GNUC_UNUSED )
    {
      TreeIter iter (m_store->get_iter(path));

      if ((*iter)[m_track_cr.present]) 
      {
        signal_activated_.emit ();
      }
    }

    void
    ViewPlaylist::append_album (uint64_t bmpx_album_id)
    {
      Query q;
      q.set_prefix (" DISTINCT ");
      q.set_suffix (" ORDER BY ifnull(tracknumber, title) ");

      for (AttributeVV::const_iterator i = m_state->i_attrs.begin(); i != m_state->i_attrs.end(); ++i)
      {
        AttributeV a = *i;
        q.add_attr_v (a, CHAIN_OR);
        q.seq_chain_append (CHAIN_AND);
      }

      AttributeV a;
      a.push_back (Attribute (EXACT, "album_j", bmpx_album_id)); 
      q.add_attr_v (a);
      q.seq_chain_append (CHAIN_AND);
      q.set_columns (" * ");
      
      RowV r;
      library->query (q, r);

      for (RowV::const_iterator t = r.begin(); t != r.end(); ++t)
      {
        put_track_at_iter (*t);
      }
      columns_autosize ();
    }

    void
    ViewPlaylist::display ()
    {
      clear ();
    
      m_state->lock ();
      TrackV tracks = m_state->r_tracks;
      m_state->unlock ();

      for (TrackV::const_iterator i = tracks.begin(); i != tracks.end(); ++i)
      {
        if (m_state->m_reset.trylock()) 
        {
          if( m_state->m_querying )
          {
            clear ();
            m_state->m_reset.unlock ();
            return;
          }
        }

        put_track_at_iter (*i);
      }
      columns_autosize ();
      signal_recheck_caps_.emit ();
    }

    void
    ViewPlaylist::put_track_at_iter (Track const& track)
    {
      TreeIter i = m_store->append();
      put_track_at_iter (track, i);
    }
 
    void
    ViewPlaylist::put_track_at_iter (Track const& track, Gtk::TreeModel::iterator & iter)
    {
      (*iter)[m_track_cr.track] = track;
      (*iter)[m_track_cr.uid] = track.bmpx_track_id;
      (*iter)[m_track_cr.localUid] = m_localUid;
      (*iter)[m_track_cr.searchKey] = track.title ? track.title.get() : std::string();
      (*iter)[m_track_cr.playTrack] = track.active.get();

      bool present = file_test (filename_from_uri (track.location.get()), FILE_TEST_EXISTS);
#ifdef HAVE_HAL
      (*iter)[m_track_cr.present] = present && (hal->volume_is_mounted (HAL::VolumeKey (track.volume_udi.get(), track.device_udi.get())));
#else
      (*iter)[m_track_cr.present] = present; 
#endif //HAVE_HAL

      if( m_playing && m_bmpx_track_id == track.bmpx_track_id )
      {
        assign_current_iter (iter);
      }

      m_localUidIterMap.insert (std::make_pair (m_localUid, iter));

      UidIterSetMap::iterator i (m_UidIterMap.find (track.bmpx_track_id));
      if( i == m_UidIterMap.end() )
      {
        IterSet x;
        x.insert (iter);
        m_UidIterMap.insert (std::make_pair (track.bmpx_track_id, x));
      }
      else
      {
        IterSet & x (i->second);
        x.insert (iter);
      }
      m_localUid++;
    }

    bool
    ViewPlaylist::slot_select (RefPtr < Gtk::TreeModel > const& model, Gtk::TreeModel::Path const& path, bool was_selected)
    {
      return bool ((*m_store->get_iter (path))[m_track_cr.present]);
    }

    void
    ViewPlaylist::cell_play_track_toggled (ustring const& path)
    {
      if( bool ((*m_store->get_iter (path))[m_track_cr.present]) )
      {
            TreeIter iter (m_store->get_iter (path));
            Track const& track = (*iter)[m_track_cr.track];
            ::library->track_set_active (track.bmpx_track_id, ! track.active.get()); 
      }
    }

    void
    ViewPlaylist::cell_data_func (CellRenderer * basecell, TreeModel::iterator const& iter, int column, int renderer)
    {
      basecell->property_sensitive() = bool ((*iter)[m_track_cr.present]);

      CellRendererText          * cell_t = 0;
      CellRendererPixbuf        * cell_p = 0;
      CellRendererCairoSurface  * cell_s = 0;
      CellRendererToggle        * cell_g = 0;

      Track const& track = (*iter)[m_track_cr.track];

      UID uid      = (*iter)[m_track_cr.uid];
      UID localUid = (*iter)[m_track_cr.localUid];

      switch (renderer)
      {
        case R_TEXT:
          cell_t = dynamic_cast <CellRendererText *>    (basecell);
          break;

        case R_PIXBUF:
          cell_p = dynamic_cast <CellRendererPixbuf *>  (basecell);
          break;

        case R_SURFACE:
          cell_s = dynamic_cast <CellRendererCairoSurface *>  (basecell);
          break;

        case R_TOGGLE:
          cell_g = dynamic_cast <CellRendererToggle *>  (basecell);
          break;
      }

      switch (column)
      {
        case COLUMN_PLAYING:
        {
          if( m_current_iter && (m_current_iter.get() == iter) )
            cell_p->property_pixbuf() = m_pb_playing;
          else
            cell_p->property_pixbuf() = RefPtr <Gdk::Pixbuf> (0);

          break;
        }

        case COLUMN_TRACK:
        {
          if( track.tracknumber && (track.tracknumber.get() != 0) )
            cell_t->property_text() = (uint64_f % track.tracknumber.get()).str();
          else
            cell_t->property_text() = "";
          break;
        }

        case COLUMN_ACTIVE:
        {
          cell_g->property_active() = track.active.get(); 
          break;
        }

        case COLUMN_RATING:
        {
          int rating = int (track.rating ? track.rating.get() : 0);

          ::Cairo::RefPtr< ::Cairo::ImageSurface> surface = ::Cairo::ImageSurface::create (::Cairo::FORMAT_ARGB32, 64, 16);
          ::Cairo::RefPtr< ::Cairo::Context> cr = ::Cairo::Context::create (surface); 
          cr->set_operator (::Cairo::OPERATOR_CLEAR);
          cr->paint ();

          if ((rating > 0) && (rating <= 3)) // don't overflow the gradient array
          {
            Gdk::Color color (gradient[rating]);
            cr->set_operator (::Cairo::OPERATOR_SOURCE);
            cr->set_source_rgba (color.get_red_p(), color.get_green_p(), color.get_blue_p(), 1.);
            cr->rectangle (0, 0, 64, 16);
            cr->fill ();
          }

          cell_s->property_surface() = surface;
        
          break;
        }

        /*case COLUMN_LIBRARY_TRACK:
        {
          if( track.new_item )
            cell_p->property_pixbuf() = render_icon (Gtk::StockID (BMP_STOCK_LIBRARY_TRACK), Gtk::ICON_SIZE_MENU);
          else
            cell_p->property_pixbuf() = RefPtr <Gdk::Pixbuf> (0);
          break;
        }*/

        case COLUMN_TITLE:
        {
          cell_t->property_text() = track.title
                                    ? track.title.get().c_str()           
                                    : "";
          break;
        }

        case COLUMN_ARTIST:
        {
          cell_t->property_text() = track.artist
                                    ? track.artist.get().c_str()
                                    : ""; 
          break;
        }

        case COLUMN_ALBUM:
        {
          cell_t->property_text() = track.album             ? track.album.get().c_str()           
                                                            : "";
          break;
        }

        case COLUMN_TIME:
        {
          if( track.duration )
          {
            uint64_t duration (track.duration.get());
            cell_t->property_text() = (boost::format ("%d:%02d") % (duration / 60) % (duration % 60)).str();
          }
          else
            cell_t->property_text() = "";
          break;
        }

        case COLUMN_GENRE:
        {
          cell_t->property_text() = track.genre       ? track.genre.get().c_str() 
                                                      : "";
          break;
        }

        case COLUMN_BITRATE:
        {
          cell_t->property_text() = track.bitrate     ? (uint64_f % track.bitrate.get()).str()    
                                                      : "";
          break;
        }

        case COLUMN_SAMPLERATE:
        {
          cell_t->property_text() = track.samplerate  ? (uint64_f % track.samplerate.get()).str() 
                                                      : "";
          break;
        }

        case COLUMN_COUNT:
        {
          cell_t->property_text() = track.count   ? (uint64_f % track.count.get()).str() 
                                                  : "";
          break;
        }
      }
    }

    // PlaybackSource
    bool
    ViewPlaylist::has_playing ()
    {
      return bool (m_current_iter);
    }

    void
    ViewPlaylist::set_first_iter ()
    {
      assign_current_iter (m_store->get_iter (TreePath (TreePath::size_type (1), TreePath::value_type (0))));
    }

    Track
    ViewPlaylist::get_track ()
    {
      return Track ((*m_current_iter.get())[m_track_cr.track]);
    }

    ustring
    ViewPlaylist::get_uri ()
    {
      return Track ((*m_current_iter.get())[m_track_cr.track]).location.get();
    }

    void
    ViewPlaylist::notify_stop ()
    {
      clear_current_iter ();
    }

    bool
    ViewPlaylist::notify_play ()
    {
      PathV paths (get_selection()->get_selected_rows ());

      if( paths.size() > 1 )
      { 
        return false;
      }
      else
      if( paths.size() < 1 )
      {
        if( m_current_iter )
        {
          return true;
        }

        // Find the first playable row
        TreePath path = TreePath (TreePath::size_type (1), TreePath::value_type (0));
        TreeIter iter;

        bool has_track = false;       
        bool play = false;

        do {
          if( path.get_indices().data()[0] == (m_store->children().size()-1) )
            break;
          iter = m_store->get_iter (path); 
          Track const& track = (*iter)[m_track_cr.track];
          bool q1 (track.active.get());
          bool q2 ((*iter)[m_track_cr.present]);
          play = (q1 && q2); 
          has_track = (path.get_indices().data()[0] < (m_store->children().size()-1));
          path.next ();
          }
        while (!play && has_track);
  
        if( play && has_track )
        {
          m_history.set (UID ((*iter)[m_track_cr.localUid]));
          assign_current_iter (iter);
          return true;
        }
        else
          return false;
      }
      else
      {
        m_history.set (UID ((*m_store->get_iter (paths[0]))[m_track_cr.localUid]));
        assign_current_iter (m_store->get_iter (paths[0]));
        return true;
      }
    }

    bool
    ViewPlaylist::check_play ()
    {
      if( !m_store->children().size() )
        return false;

      if( get_selection()->count_selected_rows() == 1 )
      {
        TreeIter iter = m_store->get_iter (PathV (get_selection()->get_selected_rows())[0]);
        Track const& track = (*iter)[m_track_cr.track];
        bool q1 (track.active.get());
        bool q2 ((*iter)[m_track_cr.present]);
        return (q1 && q2);
      }
      else
      {
        // Find the first playable row
        TreePath path = TreePath (TreePath::size_type (1), TreePath::value_type (0));
        TreeIter iter;

        bool last = false;       
        bool play = false;

        do {
          if( path.get_indices().data()[0] == (m_store->children().size()-1) )
            break;
          iter = m_store->get_iter (path); 
          Track const& track = (*iter)[m_track_cr.track];
          bool q1 (track.active.get());
          bool q2 ((*iter)[m_track_cr.present]);
          play = (q1 && q2); 
          last = (path.get_indices().data()[0] == (m_store->children().size()-1));
          path.next ();
          }
        while (!play && !last);
        return play;
      }
    }

    void
    ViewPlaylist::assign_current_iter (Gtk::TreeModel::iterator const& iter)
    {
      m_current_iter  = iter;
      m_bmpx_track_id = Track ((*iter)[m_track_cr.track]).bmpx_track_id;

      g_assert (bool (iter));

      TreeModel::Path path1, path2, path3 (iter);
      if( get_visible_range (path1, path2) )
      {
        if( (path3 < path1) || (path3 > path2) )
        {
          scroll_to_row (path3, 0.5);
        }
      }

      queue_draw ();
    }

    bool
    ViewPlaylist::notify_next ()
    {
      TreeModel::Children const& children (m_store->children());
      TreeModel::Children::size_type size (children.size());

      /* There is no next either way if the datastore is empty */
      if( !size )
      {
        return false;
      }

      /* First let's see if the history has a next item (this should not be the case if the store was cleared previously, which
       * is not unimportant, but implict here
       */
      if( m_history.have_next() )
      {
        UID uid (m_history.get_next());
        assign_current_iter (m_localUidIterMap.find (uid)->second);
        return true;
      }

      if( mcs->key_get <bool> ("bmp", "shuffle") )
      {
        TreeNodeChildren::size_type n1 = 0; 
        TreeNodeChildren::size_type n2 = (m_store->children().size()-1);

        // Find a random iter until history doesn't have it
        Glib::Rand rand;
        std::set<UID> UidKeeper;
        UID uid = 0;
        while (n1 != n2)
        {
          guint32 index = rand.get_int_range (0, n2+1); 
          TreeIter iter = m_store->get_iter (TreePath (TreePath::size_type (1), TreePath::value_type (index)));
          uid = ((*iter)[m_track_cr.localUid]);

          if (UidKeeper.find (uid) != UidKeeper.end()) 
          {
            continue;
          }
          else
          {
            UidKeeper.insert (uid);
            Track const& track = (*iter)[m_track_cr.track];
            bool q1 (track.active.get());
            bool q2 ((*iter)[m_track_cr.present]);
            if( !(q1 && q2) )
            {
              ++n1;
              continue;
            }

            if( m_history.has_uid (uid) )
            {
              ++n1; 
              continue;
            }
          }
          break;
        }

        if( (uid == 0) || (n1 == n2) )
        {
          // History is exhausted
          if( mcs->key_get <bool> ("bmp", "repeat") )
          {
            if( !m_history.empty() )
            {
              m_history.rewind ();
              UID uid;
              if( m_history.get (uid) )
              {
                UidIterMap::iterator i = m_localUidIterMap.find (uid);
                if( i != m_localUidIterMap.end() )
                {
                  assign_current_iter (i->second);
                  return true;
                }
                else
                  g_warning ("%s: History is not empty but uid %llu is not present in current view", G_STRLOC, uid);
              }
              else
                g_warning ("%s: History claims to be not empty, but can not deliver an uid (this is EXTREMELY strange!)", G_STRLOC);
            }
          }
          m_history.clear();
          return false;
        }
        else
        {
          UidIterMap::iterator i = m_localUidIterMap.find (uid);
          if( i != m_localUidIterMap.end() )
          {
            m_history.append (uid);
            assign_current_iter (i->second);
            return true;
          }
          else
          {
            g_warning ("%s: History is not empty but uid %llu is not present in current view", G_STRLOC, uid);
            return false;
          }
        }
      }

      TreeIter  iter;
      TreePath  path;
      bool      advance = 0;

      /* There is no current iter set, which should be only the case if the store is empty,
       * or it's not, but the current track is not present in the datastore 
       */
      if( !m_current_iter )
      {
        if( mcs->key_get <bool> ("bmp", "repeat") )
        {
          m_history.clear();
          path = TreePath (TreePath::size_type (1), TreePath::value_type (0));
          iter = m_store->get_iter (path);
          advance = 0;
        }
        else
        {
          /* No current iter, no repeat = nada */
          return false;
        }
      }
      else
      {
        /* I can has current iter? KTHX */
        iter = m_current_iter.get();
        path = m_store->get_path (iter);
        advance = 1;

        if( (path.get_indices().data()[0] == (m_store->children().size()-1) )
            &&  mcs->key_get<bool>("bmp","repeat"))
        {
          if( !m_history.empty() )
          {
            m_history.rewind ();
            UID uid;
            if( m_history.get (uid) )
            {
              UidIterMap::iterator i = m_localUidIterMap.find (uid);
              if( i != m_localUidIterMap.end() )
              {
                assign_current_iter (i->second);
                return true;
              }
              else
                g_warning ("%s: History is not empty but uid %llu is not present in current view", G_STRLOC, uid);
            }
            else
              g_warning ("%s: History claims to be not empty, but can not deliver an uid (this is EXTREMELY strange!)", G_STRLOC);
          }

          path = TreePath (TreePath::size_type (1), TreePath::value_type (0));
          iter = m_store->get_iter (path);
          advance = 0;
        }
      }

      do{
          if( advance )
            path.next ();
          else
            advance = 1;
  
          if (path.get_indices().data()[0] < m_store->children().size()) 
          {
            iter = m_store->get_iter (path);
            Track const& track = (*iter)[m_track_cr.track];
            bool q1 (track.active.get());
            bool q2 ((*iter)[m_track_cr.present]);
            if( q1 && q2 )
            {
              m_history.append (UID ((*iter)[m_track_cr.localUid]));
              assign_current_iter (iter);
              return true;
            }
          }
          else
            break;
        }
      while (1);

      clear_current_iter ();
      return false;

    }

    bool
    ViewPlaylist::notify_prev ()
    {
      TreeModel::Children const& children (m_store->children());
      TreeModel::Children::size_type size (children.size());

      if( !size )
      {
        return false;
      }

      if( m_history.have_prev() )
      {
        UID uid (m_history.get_prev());
        assign_current_iter (m_localUidIterMap.find (uid)->second);
        return true;
      }

      TreeIter iter = m_current_iter.get();
      TreePath path = m_store->get_path (iter);

      do{
          path.prev ();
          if( path.get_indices().data()[0] >= 0 )
          {
            iter = m_store->get_iter (path);
            Track const& track = (*iter)[m_track_cr.track];
            bool q1 (track.active.get());
            bool q2 ((*iter)[m_track_cr.present]);
            if( q1 && q2 )
            {
              m_history.prepend (UID ((*iter)[m_track_cr.localUid]));
              assign_current_iter (iter);
              return true;
            }
          }
          else
            break;
        }
      while (1);

      clear_current_iter ();
      return false;
    }

    void
    ViewPlaylist::has_next_prev (bool & next, bool & prev)
    {
      TreeModel::Children const& children (m_store->children());
      TreeModel::Children::size_type size = children.size();

      next = false;
      prev = false;

      if( !size )
        return;

      if( !m_history.boundary() )
      {
        next = m_history.have_next();
        prev = m_history.have_prev();
        return;
      }

      TreePath path;
      TreeIter iter;

      if( m_current_iter && m_current_iter.get() )
      {
        path = TreePath (m_store->get_path (m_current_iter.get()));
        if (path.get_indices().data()[0] > 0) 
        do{
            path.prev ();
            if (path.get_indices().data()[0] >= 0) 
            {
              iter = m_store->get_iter (path); 
              Track const& track = (*iter)[m_track_cr.track];
              bool q1 (track.active.get());
              bool q2 ((*iter)[m_track_cr.present]);
              prev = (q1 && q2); 
            }
          }
        while (!prev && (path.get_indices().data()[0] > 0));
      }

      if( mcs->key_get<bool>("bmp","repeat") || mcs->key_get<bool>("bmp","shuffle") )
      {
        next = true;
      }
      else if( m_current_iter )
      {
        path = TreePath (m_store->get_path (m_current_iter.get()));
        if( path.get_indices().data()[0] < (m_store->children().size()-1) )
        do{
            path.next ();
            if( path.get_indices().data()[0] < m_store->children().size() )
            {
              iter = m_store->get_iter (path); 
              Track const& track = (*iter)[m_track_cr.track];
              bool q1 (track.active.get());
              bool q2 ((*iter)[m_track_cr.present]);
              next = (q1 && q2); 
            }
            else
              break;
          }
        while (!next);
      }
    }

    void
    ViewPlaylist::clear_current_iter ()
    {
      if( m_current_iter && m_store->children().size() )
      {
        // Weird construct to notify treeview that the row has changed
        TreeIter iter = m_current_iter.get();
        m_current_iter.reset ();
        if( iter )
        {
          m_store->row_changed (m_store->get_path (iter), iter);
        }
      }
      else
      {
        m_current_iter.reset ();
      }
      signal_recheck_caps_.emit ();
    }

    void
    ViewPlaylist::on_library_track_modified (Track const& track)
    {
      UidIterSetMap::iterator m (m_UidIterMap.find (track.bmpx_track_id));
      if( m != m_UidIterMap.end() )
      {
        IterSet& x = m->second;
        std::for_each (x.begin(), x.end(), TrackApply<ColumnTrack> (m_track_cr.track, track));
      }

      // Check the current state index
      m_state->lock ();
      UidIndexMapIter i (m_state->r_tracks_index.find (track.bmpx_track_id));
      if( i != m_state->r_tracks_index.end() )
      {
        m_state->r_tracks[std::distance (m_state->r_tracks_index.begin(), i)] = track;
      }
      m_state->unlock ();
    }

    void
    ViewPlaylist::clear ()
    {
      m_UidIterMap.clear();
      m_localUidIterMap.clear();
      m_store->clear();
      m_history.clear ();
      clear_current_iter ();
    }

    void
    ViewPlaylist::on_tracks_retag ()
    {
      TrackV  v,
              m;

      PathV p = get_selection()->get_selected_rows ();
      std::for_each (p.begin(), p.end(), PathCollect<ColumnTrack> (m_store, m_track_cr.track, v));

      boost::shared_ptr<MusicBrainzTagger> tagger (MusicBrainzTagger::Create ());
      int response = tagger->run (v, m);
      tagger->hide ();

      // FIXME: Don't do this here in the playlist's code
      if( response == Gtk::RESPONSE_OK )
      {
        library->modify (m, TrackModFlags (TRACK_MOD_RETAG | TRACK_MOD_UPDATE_DB));
        library->vacuum_tables ();
        signal_updated_.emit ();
      }
    }

    void
    ViewPlaylist::on_playlist_export ()
    {
      PathV   p = get_selection()->get_selected_rows ();
      TrackV  v;
      std::for_each (p.begin(), p.end(), PathCollect<ColumnTrack> (m_store, m_track_cr.track, v));

      ExportDialog * dialog = ExportDialog::create ();
      dialog->run (v);
      delete dialog;
    }

    void
    ViewPlaylist::on_delete_files ()
    {
      PathV p = get_selection()->get_selected_rows();
      ReferenceV r;
      std::for_each (p.begin(), p.end(), ReferenceCollect (m_store, r));
      ReferenceVIter i;

      if( m_current_iter )
      {
        for (i = r.begin() ; i != r.end(); ++i) 
        {
          TreeIter iter = m_store->get_iter (i->get_path());
          if( m_current_iter.get() == iter )
          {
            MessageDialog dialog (_("You can not delete the currently playing track.\n"
                                    "Stop playback, or exclude it from the selection of\n"
                                    "tracks to be deleted."), false, MESSAGE_ERROR, BUTTONS_OK, true);
            dialog.set_title (_("Can't Delete current Track - BMP"));
            dialog.run ();
            return;
          }
        }
      }

      MessageDialog * dialog = new MessageDialog (_("Are you <b>sure</b> you want to delete the selected tracks <b>permanently</b>?"),
        true, MESSAGE_QUESTION, BUTTONS_YES_NO, true);
      dialog->set_title (_("Delete Tracks - BMP"));
      int response = dialog->run ();
      delete dialog;

      if( response == GTK_RESPONSE_YES )
      {
        for (i = r.begin() ; !r.empty() ; )
        {
          TreeIter iter = m_store->get_iter (i->get_path());
          Track track ((*iter)[m_track_cr.track]);
          if( g_unlink (filename_from_uri (track.location.get()).c_str()) == 0 )
          {
            library->remove (track);
            m_store->erase (iter);
          }
          i = r.erase (i);
        }

        library->vacuum_tables ();
        signal_updated_.emit ();
      }
    }

    void
    ViewPlaylist::set_ui_manager (RefPtr<Gtk::UIManager> const& ui_manager)
    {
      m_ui_manager = ui_manager;

      m_actions = Gtk::ActionGroup::create ("Actions_UiPartLibrary-ViewPlaylist");

      m_actions->add  (Gtk::Action::create (ACTION_RETAG_TRACKS,
                                            Gtk::Stock::EDIT,
                                            _("Re-Tag Tracks")),
                                            sigc::mem_fun (*this, &ViewPlaylist::on_tracks_retag));

      m_actions->add  (Gtk::Action::create (ACTION_PLAYLIST_EXPORT,
                                            Gtk::Stock::SAVE_AS,
                                            _("Export Selected")),
                                            sigc::mem_fun (*this, &ViewPlaylist::on_playlist_export));

      m_actions->add  (Gtk::Action::create (ACTION_HISTORY_CLEAR,
                                            Gtk::StockID (GTK_STOCK_CLEAR),
                                            _("Clear Playback History")),
                                            sigc::mem_fun (m_history, &ViewPlaylist::History::clear));

      m_actions->add  (Gtk::Action::create (ACTION_DELETE_FILES,
                                            Gtk::StockID (GTK_STOCK_DELETE),
                                            _("Delete Files Physically")),
                                            sigc::mem_fun (*this, &ViewPlaylist::on_delete_files));

      m_ui_manager->insert_action_group (m_actions);
      m_ui_manager->add_ui_from_string (library_menu_playlist);
    }

    bool
    ViewPlaylist::on_event (GdkEvent * ev)
    {
      if( ev->type == GDK_BUTTON_PRESS )
      {
        GdkEventButton * event = reinterpret_cast <GdkEventButton *> (ev);
        if( event->button == 3 )
        {
          m_actions->get_action (ACTION_RETAG_TRACKS)->set_sensitive
            (get_selection()->count_selected_rows() != 0);

          m_actions->get_action (ACTION_HISTORY_CLEAR)->set_sensitive
            (!m_history.empty());

          m_actions->get_action (ACTION_PLAYLIST_EXPORT)->set_sensitive
            (get_selection()->count_selected_rows());

          Gtk::Menu * menu = dynamic_cast < Gtk::Menu* >
                                (Util::get_popup (m_ui_manager, "/popup-library-playlist/menu-library-playlist"));

          if (menu) // better safe than screwed
          {
            menu->popup (event->button, event->time);
          }
          return true;
        }
      }

      return false;
    }

    bool
    ViewPlaylist::on_motion_notify_event (GdkEventMotion * event)
    {
      TreeView::on_motion_notify_event (event);
      return false;
    }

    ViewPlaylist::~ViewPlaylist ()
    {}

#ifdef HAVE_HAL
    void
    ViewPlaylist::hal_volume_del  (HAL::Volume const& volume)
    {
      unselect_missing ();
      TreeNodeChildren nodes = m_store->children();
      for (TreeNodeChildren::iterator i = nodes.begin(); i != nodes.end(); ++i)
      {
        Track const& track = (*i)[m_track_cr.track];
        (*i)[m_track_cr.present] = file_test (filename_from_uri (track.location.get()), FILE_TEST_EXISTS); 
      }
    }

    void
    ViewPlaylist::hal_volume_add  (HAL::Volume const& volume)
    {
      TreeNodeChildren nodes = m_store->children();
      for (TreeNodeChildren::iterator i = nodes.begin(); i != nodes.end(); ++i)
      {
        Track const& track = (*i)[m_track_cr.track];
        (*i)[m_track_cr.present] = file_test (filename_from_uri (track.location.get()), FILE_TEST_EXISTS); 
      }
    }

    void
    ViewPlaylist::unselect_missing ()
    {
      PathV list (get_selection()->get_selected_rows());
      for (PathV::const_iterator i = list.begin(); i != list.end(); ++i)
      {
            Track const& track = (*m_store->get_iter (*i))[m_track_cr.track];

            if( !hal->volume_is_mounted (HAL::VolumeKey (track.volume_udi.get(), track.device_udi.get())) )
            {
                  get_selection()->unselect (*i);
            }
      }
    }
#endif //HAVE_HAL
}

namespace Bmp
{
  LibrarySearch::~LibrarySearch ()
  {
    if( m_state->m_reset.trylock() )
    {
      m_state->m_reset.unlock();
    }
    else
    {
      g_warning ("%s: m_reset was locked!", G_STRLOC);
      m_state->m_reset.unlock();
    }

    if( m_state->mLock1.trylock() )
    {
      m_state->mLock1.unlock();
    }
    else
    {
      g_warning ("%s: mLock1 was locked!", G_STRLOC);
      m_state->mLock1.unlock();
    }
  }

  LibrarySearch::LibrarySearch (RefPtr <Gnome::Glade::Xml>  const& xml,
                                RefPtr <Gtk::UIManager>     const& ui_manager)
    : m_ref_xml (xml)
    , m_current_attribute (-1)
    , m_dissociated (false)
    , m_change_block (false)
  {
    m_search_entry = manage (new Sexy::IconEntry());
    m_saved_color = m_search_entry->get_style()->get_text (Gtk::STATE_NORMAL);

    Label * label = manage (new Label());
    label->set_markup_with_mnemonic (_("_Search:"));
    label->set_mnemonic_widget (*m_search_entry);

    Gtk::Image * image = 0; 
    image = manage (new Gtk::Image (m_search_entry->render_icon (Gtk::StockID (BMP_STOCK_SEARCH), Gtk::ICON_SIZE_MENU)));
    m_search_entry->set_icon (Sexy::ICON_ENTRY_PRIMARY, image);
    image = manage (new Gtk::Image (m_search_entry->render_icon (Gtk::StockID (BMP_STOCK_ENTRY_CLEAR), Gtk::ICON_SIZE_MENU)));
    m_search_entry->set_icon (Sexy::ICON_ENTRY_SECONDARY, image);

    m_search_entry->signal_icon_pressed().connect
      (sigc::mem_fun (*this, &Bmp::LibrarySearch::on_search_entry_icon_pressed));
    m_search_entry->signal_changed().connect
      (sigc::mem_fun (*this, &Bmp::LibrarySearch::on_search_entry_changed));

    dynamic_cast<Gtk::HBox*>(m_ref_xml->get_widget ("library-hbox-filter"))->pack_start (*label, false, false);
    label->show();

    dynamic_cast<Gtk::HBox*>(m_ref_xml->get_widget ("library-hbox-filter"))->pack_start (*m_search_entry, true, true);
    m_search_entry->show();

    m_ui_manager  = ui_manager;
    m_actions     = Gtk::ActionGroup::create ("Actions_LibrarySearch");

    Gtk::RadioButtonGroup gr1;
    m_actions->add  (Gtk::RadioAction::create (gr1, ACTION_SEARCH_SEARCH_ALL,
                                              _("Everything")),
                                              (sigc::mem_fun (*this, &LibrarySearch::select_attribute)));
    RefPtr<Gtk::RadioAction>::cast_static (m_actions->get_action
      (ACTION_SEARCH_SEARCH_ALL))->property_value() = -1;

    m_actions->add  (Gtk::RadioAction::create (gr1, ACTION_SEARCH_SEARCH_ARTIST,
                                              _("Artist")),
                                              (sigc::mem_fun (*this, &LibrarySearch::select_attribute)));
    RefPtr<Gtk::RadioAction>::cast_static (m_actions->get_action
      (ACTION_SEARCH_SEARCH_ARTIST))->property_value() = ATTRIBUTE_ARTIST; 

    m_actions->add  (Gtk::RadioAction::create (gr1, ACTION_SEARCH_SEARCH_ALBUM,
                                              _("Album")),
                                              (sigc::mem_fun (*this, &LibrarySearch::select_attribute)));
    RefPtr<Gtk::RadioAction>::cast_static (m_actions->get_action
      (ACTION_SEARCH_SEARCH_ALBUM))->property_value() = ATTRIBUTE_ALBUM; 

    m_actions->add  (Gtk::RadioAction::create (gr1, ACTION_SEARCH_SEARCH_TITLE,
                                              _("Title")),
                                              (sigc::mem_fun (*this, &LibrarySearch::select_attribute)));
    RefPtr<Gtk::RadioAction>::cast_static (m_actions->get_action
      (ACTION_SEARCH_SEARCH_TITLE))->property_value() = ATTRIBUTE_TITLE; 

    m_actions->add  (Gtk::RadioAction::create (gr1, ACTION_SEARCH_SEARCH_ALBUM_ARTIST,
                                              _("Album Artist")),
                                              (sigc::mem_fun (*this, &LibrarySearch::select_attribute)));
    RefPtr<Gtk::RadioAction>::cast_static (m_actions->get_action
      (ACTION_SEARCH_SEARCH_ALBUM_ARTIST))->property_value() = ATTRIBUTE_MB_ALBUM_ARTIST;

    m_actions->add  (Gtk::RadioAction::create (gr1, ACTION_SEARCH_SEARCH_GENRE,
                                              _("Genre")),
                                              (sigc::mem_fun (*this, &LibrarySearch::select_attribute)));
    RefPtr<Gtk::RadioAction>::cast_static (m_actions->get_action
      (ACTION_SEARCH_SEARCH_GENRE))->property_value() = ATTRIBUTE_GENRE;

    RefPtr<Gtk::RadioAction>::cast_static (m_actions->get_action (ACTION_SEARCH_SEARCH_ALL))->set_active();

    m_actions->add  (Gtk::Action::create ("action-focus-search-entry",
                                          _("Focus Search Entry")),
                                          AccelKey ("F6"),
                                          (sigc::mem_fun (*m_search_entry, &Gtk::Widget::grab_focus)));


    m_ui_manager->insert_action_group (m_actions);
    m_ui_manager->add_ui_from_string (library_search_popup);
  }

  void
  LibrarySearch::select_attribute ()
  {
    static bool x = 0;
    if( !x )
    {
      x = 1;
      return;
    }

    x = 0;
    m_current_attribute = RefPtr<Gtk::RadioAction>::cast_static (m_actions->get_action (ACTION_SEARCH_SEARCH_ALL))->get_current_value();
    run_query ();
  }

  void
  LibrarySearch::on_search_entry_icon_pressed (Sexy::IconEntryPosition pos, int button)
  {
    if( pos == Sexy::ICON_ENTRY_PRIMARY )
    {
      if (button == 1) 
      {
        Gtk::Menu * menu = dynamic_cast <Gtk::Menu*> (Util::get_popup (m_ui_manager, "/popup-library-search/menu-library-search"));
        menu->popup (sigc::mem_fun (*this, &Bmp::LibrarySearch::menu_position), button, 0L); 
      }
      else
      if( button == 3 )
      {
        associate ();
        signal_search_restore_.emit ();
      }
    }
    else
    if( pos == Sexy::ICON_ENTRY_SECONDARY )
    {
      if( button == 1 )
      {
        m_search_entry->set_text (ustring());
        signal_search_clear_.emit ();
      }
    }
  }

  void
  LibrarySearch::dissociate ()
  {
    if( !m_dissociated && (m_search_entry->get_text().length() > MIN_SEARCH_CHARS) )
    {
      m_search_entry->modify_text (Gtk::STATE_NORMAL, m_search_entry->get_style()->get_text (Gtk::STATE_INSENSITIVE));
      m_dissociated = 1;
    }
  }

  void
  LibrarySearch::associate ()
  {
    m_search_entry->modify_text (Gtk::STATE_NORMAL, m_saved_color); 
    m_dissociated = 0;
  }

  void
  LibrarySearch::menu_position (int & x, int & y, bool & push_in)
  {
    Glib::RefPtr<Gdk::Window> window = dynamic_cast<Gtk::Window*>(m_ref_xml->get_widget ("main-ui"))->get_window();
    window->get_root_origin (x, y);

    x += m_search_entry->get_allocation().get_x();
    y += m_search_entry->get_allocation().get_y() + m_search_entry->get_allocation().get_height() + 2;
    push_in = true;
  }

  bool
  LibrarySearch::on_search_entry_timeout ()
  {
    if( entry_changed_timer.elapsed() >= KEY_PRESS_MAX_TIME )
    {
      entry_changed_timer.stop();
      entry_changed_timer.reset();
      run_query ();
      return false;
    }
    return true;
  }

  void
  LibrarySearch::on_search_entry_changed ()
  {
    sort_entry_changed_timeout_conn.disconnect();
    entry_changed_timer.stop();

    if( m_search_entry->get_text().length() < MIN_SEARCH_CHARS )
    {   
      m_state->clear ();
      signal_search_result_.emit ();
      return;
    }

    entry_changed_timer.reset();
    entry_changed_timer.start();
    sort_entry_changed_timeout_conn = signal_timeout().connect
      (sigc::mem_fun (*this, &Bmp::LibrarySearch::on_search_entry_timeout), int (1000. * KEY_PRESS_MAX_TIME));
  }

  bool
  LibrarySearch::run_query_result_idle ()
  {
    signal_search_result_.emit ();
    return false;
  }

  void
  LibrarySearch::run_query_thread ()
  {
    using boost::algorithm::trim;
    std::string text (m_search_entry->get_text());
    trim (text);

    if( text.size() >= MIN_SEARCH_CHARS )
    {
      char **strv = g_strsplit (text.c_str(), " ", -1);
      char **save = strv;

      m_state->lock ();
      m_state->clear_unlocked ();
      m_state->m_querying = 1;
      m_state->m_reset.unlock ();

      AttributeV  a;
      Query       q;
      while (*strv)
      {
        if( strlen (*strv) < 2 )
        {
          ++strv;
          continue; // we ignore substrings shorter in length than 2 characters
        }

        if( m_current_attribute == -1 )
        {
          a.push_back  (Attribute (FUZZY, get_attribute_info (ATTRIBUTE_ARTIST).id, std::string (*strv)));
          a.push_back  (Attribute (FUZZY, get_attribute_info (ATTRIBUTE_MB_ARTIST_SORTNAME).id, std::string (*strv)));
          a.push_back  (Attribute (FUZZY, get_attribute_info (ATTRIBUTE_MB_ALBUM_ARTIST).id, std::string (*strv)));
          a.push_back  (Attribute (FUZZY, get_attribute_info (ATTRIBUTE_MB_ALBUM_ARTIST_SORTNAME).id, std::string (*strv)));
          a.push_back  (Attribute (FUZZY, get_attribute_info (ATTRIBUTE_ALBUM).id, std::string (*strv)));
          a.push_back  (Attribute (FUZZY, get_attribute_info (ATTRIBUTE_TITLE).id, std::string (*strv)));
        }
        else
        {
          a.push_back  (Attribute (FUZZY, get_attribute_info (AttributeId (m_current_attribute)).id, std::string (*strv)));
        }

        q.add_attr_v (a, CHAIN_OR);
        q.seq_chain_append (CHAIN_AND);
        ++strv;
        m_state->i_attrs.push_back (a);
        a.clear ();
      }
      g_strfreev (save);

      q.set_suffix  (" ORDER BY ifnull(mb_album_artist_sort_name, mb_album_artist), ifnull(mb_release_date, album), tracknumber ");
      q.set_columns (" * ");
      RowV rows;
      library->query (q, rows, true); 

      // Build Index 
      TrackV::size_type n = 0;
      for (RowV::const_iterator i = rows.begin(); i != rows.end(); ++i)
      {
        Row const& r (*i);
        m_state->r_tracks.push_back (Track (r));
        m_state->r_tracks_index.insert (UidIndexPair (boost::get <uint64_t> (r.find ("id")->second), n++)); 
      }

      q.set_prefix  ( " DISTINCT ");
      q.set_columns ( " album_artist_j ");
      q.set_suffix  ( " ORDER BY ifnull(mb_album_artist_sort_name, mb_album_artist), ifnull(mb_release_date, album) ");
      library->query (q, m_state->r_album_artist_j, true);

      q.set_prefix  ( " DISTINCT ");
      q.set_columns ( " album_artist_j, album_j ");
      q.set_suffix  ( //" GROUP BY mb_album_artist, mb_album_artist_sort_name, mb_release_date, album "
                      " ORDER BY ifnull(mb_album_artist_sort_name, mb_album_artist), ifnull(mb_release_date, album) ");
      library->query (q, m_state->r_album_j, true);

      m_state->m_querying = 0;
      m_state->m_reset.lock ();
      m_state->unlock ();
      Glib::signal_idle().connect (sigc::mem_fun (*this, &Bmp::LibrarySearch::run_query_result_idle));
    }
  }

  void
  LibrarySearch::run_query ()
  {
    Glib::Thread::create (sigc::mem_fun (*this, &Bmp::LibrarySearch::run_query_thread), false);
  }
}

namespace
{
  using namespace Bmp;
  const PlaybackSource::Flags flags =
    PlaybackSource::Flags (PlaybackSource::F_ALWAYS_IMAGE_FRAME | PlaybackSource::F_HANDLE_LASTFM |
                           PlaybackSource::F_USES_REPEAT | PlaybackSource::F_USES_SHUFFLE);
}

namespace Bmp
{
  namespace UiPart
  {
      Library::Library (RefPtr<Gnome::Glade::Xml> const& xml,RefPtr<UIManager> ui_manager)

      : PlaybackSource          (_("Library/Playlist"), PlaybackSource::CAN_SEEK, flags)
      , Base                    (xml, ui_manager)
      , m_playing               (0)
      , m_queue_playback        (0)
      , m_queue_block_changes   (0)

      {
        m_ref_xml->get_widget ("notebook-library", m_library_notebook);
        dynamic_cast <Gtk::Image *>(m_ref_xml->get_widget ("i_throbber-1"))->set (BMP_IMAGE_DIR G_DIR_SEPARATOR_S BMP_THROBBER);

        m_library_notebook->set_current_page (1);

        m_ref_xml->get_widget_derived ("view_attrib", m_view_artists);
        m_view_artists->m_state = &m_state;

        m_ref_xml->get_widget_derived ("view_albums", m_view_albums);
        m_view_albums->m_state = &m_state;

        m_ref_xml->get_widget_derived ("library-view-playlist", m_view_playlist);
        m_view_playlist->m_state = &m_state;

  // Search

        m_library_search = new LibrarySearch (m_ref_xml, m_ui_manager);
        m_library_search->m_state = &m_state;

        m_library_search->signal_search_result()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Library::on_search_result));

        m_library_search->signal_search_clear()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Library::on_search_clear));

        m_library_search->signal_search_restore()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Library::on_search_restore));

  // Playlist

        m_view_playlist->set_ui_manager (m_ui_manager);
        m_view_playlist->get_selection()->signal_changed()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Library::on_playlist_selection_changed)); // could be merged with sig caps recheck ?
        m_view_playlist->get_model()->signal_rows_reordered()
          .connect (sigc::hide (sigc::hide (sigc::hide (sigc::mem_fun (*this, &Bmp::UiPart::Library::on_playlist_rows_reordered)))));
        m_view_playlist->signal_activated()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Library::on_playlist_activated));

        m_view_playlist->signal_recheck_caps()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Library::query_playlist_caps)); // something changed
        m_view_playlist->signal_updated()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Library::update)); // metadata editing

  // Albums

        m_view_albums->signal_activated()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Library::on_albums_activated));
        m_view_albums->signal_changed()
          .connect (sigc::mem_fun (*m_view_playlist, &Bmp::ViewPlaylist::clear));
        m_view_albums->signal_changed()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Library::on_albums_changed));

  // Artists

        m_view_artists->signal_artist_selected()
          .connect (sigc::mem_fun (*m_view_albums, &Bmp::ViewAlbums::on_artist_selected));
        m_view_artists->signal_artist_cleared()
          .connect (sigc::mem_fun (*m_view_albums, &Bmp::ViewAlbums::on_artist_cleared));
        m_view_artists->signal_artist_all_selected()
          .connect (sigc::mem_fun (*m_view_albums, &Bmp::ViewAlbums::on_artist_all_selected));

        m_view_artists->signal_activated()
          .connect (sigc::mem_fun (*this, &Bmp::UiPart::Library::on_artists_activated));

        m_view_artists->display ();

  // Gtk::Actions 

        m_actions = Gtk::ActionGroup::create ("Actions_UiPartLibrary");
        m_ui_manager->insert_action_group (m_actions);

        ::library->signal_modified().connect
          (sigc::mem_fun (*this, &Bmp::UiPart::Library::on_library_backend_modified));

        mcs->subscribe ("UiPart::Library", "bmp", "shuffle",
          sigc::mem_fun (*this, &Bmp::UiPart::Library::on_shuffle_repeat_toggled));

        mcs->subscribe ("UiPart::Library", "bmp", "repeat",
          sigc::mem_fun (*this, &Bmp::UiPart::Library::on_shuffle_repeat_toggled));

        m_library_notebook->set_current_page (0);
      }

      Library::~Library () {}
      guint Library::add_ui () { return 0; }

      void
      Library::update ()
      {
        m_library_notebook->set_current_page (1);

        send_metadata ();
        m_state.clear ();

        if( m_library_search->entry().get_text().length() >= MIN_SEARCH_CHARS )
          m_library_search->run_query ();
        else
          m_view_artists->display ();

        m_library_notebook->set_current_page (0);
      }

      void
      Library::on_albums_changed ()
      {
        m_state.lock ();
        UidList l = m_view_albums->get_albums();
        if( l.size() )
        {
          m_view_playlist->clear ();
          for (UidList::const_iterator i = l.begin(); i != l.end(); ++i)
          {
            m_view_playlist->append_album (*i); 
          }

          query_playlist_caps ();

          if( m_queue_playback )
          {
            m_queue_playback = false;
            m_view_playlist->set_first_iter ();
            s_playback_request_.emit();
          }
        }
        else
        if( !m_state.new_artist() )
        {
          m_state.unlock ();
          m_view_playlist->display ();
          m_library_search->associate ();
          return;
        }
        m_state.unlock ();
      }

      ///////////// SEARCH

      void
      Library::on_search_restore ()
      {
        if (m_state.r_tracks.empty()) // means we have a selection from Albums
        {
          m_view_albums->change ();
        }
        else
        {
          m_state.m_new_artist = 1;
          m_view_albums->get_selection()->unselect_all();
          m_view_playlist->display ();
          m_library_search->associate ();
        }
      }

      void
      Library::on_search_clear ()
      {
        m_state.clear ();
        m_view_artists->display ();
      }

      void
      Library::on_search_result ()
      {
        m_state.lock ();
        bool empty = m_state.r_tracks.empty();
        m_state.unlock ();

        if( empty )
        {
          m_view_artists->clear ();
          m_view_playlist->clear ();
        }
        else
        {
          m_view_artists->display (); 
          m_view_playlist->display ();
        }
      }

      ///////////// </SEARCH>

      ///////////// ACTIVATED

      void
      Library::on_playlist_activated ()
      {
        s_playback_request_.emit();
      }

      void
      Library::on_artists_activated ()
      {
        if( m_view_albums->get_selection()->count_selected_rows() )
        {
          stop ();
          m_queue_playback = true;
          m_view_albums->change ();
        }
      }

      void
      Library::on_albums_activated ()
      {
        stop ();
        m_queue_playback = true;
        m_view_albums->change ();
      }

      //// MISC (CAPS ETC)

      void
      Library::query_playlist_caps ()
      {
        if( m_playing )
        {
          bool next, prev;
          m_view_playlist->has_next_prev (next, prev);

          if( next )
            m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
          else
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

          if( prev )
            m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
          else
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
        }
        else
        {
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
        }
        s_caps_.emit (m_caps);
      }

      void
      Library::on_playlist_selection_changed ()
      {
        if( m_view_playlist->check_play() )
          m_caps = Caps (m_caps | PlaybackSource::CAN_PLAY);
        else
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);

        s_caps_.emit (m_caps);
      }

      void
      Library::on_playlist_rows_reordered ()
      {
        m_library_search->dissociate ();
      }

      /// Misc

      void
      Library::on_library_backend_modified ()
      {
        update ();
      }

      void
      Library::on_shuffle_repeat_toggled (MCS_CB_DEFAULT_SIGNATURE)
      {
        query_playlist_caps ();
      }

      /// PlaybackSource

      ustring
      Library::get_uri ()
      {
        return m_view_playlist->get_uri ();
      }

      GHashTable*
      Library::get_metadata ()
      {
        return library->get_metadata_hash_table_for_uri (m_view_playlist->get_uri());
      }

      void
      Library::send_metadata ()
      {
        if( m_view_playlist->has_playing() )
        {
          s_track_metadata_.emit (m_view_playlist->get_track ());
        }
      }

      bool
      Library::go_next ()
      {
        if( !m_view_playlist->notify_next () )
          return false;

        query_playlist_caps ();
        send_metadata ();
        return true;
      }

      bool
      Library::go_prev ()
      {
        if( !m_view_playlist->notify_prev () )
          return false;

        query_playlist_caps ();
        send_metadata ();
        return true;
      }

      void
      Library::stop ()
      {
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_PAUSE);
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_PROVIDE_METADATA);
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

        m_playing = 0;
        m_view_playlist->m_playing = 0;

        m_view_playlist->notify_stop ();
      }

      void
      Library::play ()
      {
        if( !m_view_playlist->notify_play () )
        {
          throw UnableToInitiatePlaybackError();
        }
      }

      void
      Library::play_post ()
      {
        m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);
        m_caps = Caps (m_caps | PlaybackSource::CAN_PROVIDE_METADATA);
        m_playing = 1;
        m_view_playlist->m_playing = 1;
        query_playlist_caps ();
        send_metadata ();
      }
  }
}
