//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <gtkmm.h>
#include <gtk/gtk.h>
#include <glibmm.h>
#include <glibmm/i18n.h>
#include <libglademm.h>

#include "network.hh"
#include "paths.hh"
#include "stock.hh"

#include "audio/audio.hh"

#include "x_play.hh"
#include "x_vfs.hh"

#include "ui-part-radio.hh"
#include "streams-shoutcast.hh"
#include "streams-icecast.hh"

#include "dialog-play-uri.hh"

using namespace Glib;
using namespace Gtk;
using namespace Bmp::VFS;

namespace
{
  enum Page
  {
    PAGE_SHOUTCAST,
    PAGE_ICECAST,
  };
}

#define RADIO_ACTION_PLAY_STREAM    "radio-action-play-stream"
#define RADIO_ACTION_SHOW_ICECAST   "radio-action-show-icecast"
#define RADIO_ACTION_SHOW_SHOUTCAST "radio-action-show-shoutcast"
#define RADIO_ACTION_UPDATE_LIST    "radio-action-update-list"

namespace
{
  const char * ui_string_radio =
  "<ui>"
  ""
  "<menubar name='MenuBarMain'>"
  "   <menu action='MenuUiPartRadio'>"
  "     <menuitem action='" RADIO_ACTION_PLAY_STREAM "'/>"
  "     <separator/>"
  "     <menuitem action='" RADIO_ACTION_UPDATE_LIST "'/>"
  "     <separator/>"
  "     <menuitem action='" RADIO_ACTION_SHOW_SHOUTCAST "'/>"
  "     <menuitem action='" RADIO_ACTION_SHOW_ICECAST "'/>"
  "   </menu>"
  "</menubar>"
  ""
  "</ui>";
}

namespace Bmp
{
  namespace UiPart
  {
    Radio::Radio (RefPtr<Gnome::Glade::Xml> const& xml,
                  RefPtr<Gtk::UIManager>    const& ui_manager)
    : PlaybackSource  (_("Radio"), PlaybackSource::NONE, F_HANDLE_STREAMINFO)
    , Base (xml, ui_manager)
    , BookmarkHandler ()
    {
      m_ref_xml->get_widget         ("notebook-radio",              m_notebook_radio);
      m_ref_xml->get_widget         ("notebook-shoutcast-streams",  m_notebook_shoutcast);
      m_ref_xml->get_widget         ("notebook-icecast",            m_notebook_icecast);
      m_ref_xml->get_widget_derived ("shoutcast-genres",            m_shoutcast_g);
      m_ref_xml->get_widget_derived ("shoutcast-streams",           m_shoutcast);
      m_ref_xml->get_widget_derived ("icecast-streams",             m_icecast);

      m_filter_entry = manage (new Sexy::IconEntry());
      m_filter_entry->add_clear_button ();
      m_filter_entry->show();

      Label * label = manage (new Label());
      label->set_markup_with_mnemonic (_("_Search:"));
      label->set_mnemonic_widget (*m_filter_entry);
      label->show();

      m_filter_entry->signal_changed().connect (
        sigc::mem_fun (*this, &Radio::on_filter_changed));
       
      dynamic_cast<Gtk::HBox*> (m_ref_xml->get_widget ("radio-hbox-filter"))->pack_start (*label, false, false);
      dynamic_cast<Gtk::HBox*> (m_ref_xml->get_widget ("radio-hbox-filter"))->pack_start (*m_filter_entry, true, true);

      dynamic_cast<Gtk::Image *> (m_ref_xml->get_widget ("throbber-icecast"))->set (build_filename (BMP_IMAGE_DIR,BMP_THROBBER));
      dynamic_cast<Gtk::Image *> (m_ref_xml->get_widget ("throbber-shoutcast"))->set (BMP_IMAGE_DIR G_DIR_SEPARATOR_S BMP_THROBBER);

      m_icecast->get_selection()->signal_changed().connect
        (sigc::mem_fun (*this, &Radio::on_icecast_selection_changed));
      m_shoutcast->get_selection()->signal_changed().connect
        (sigc::mem_fun (*this, &Radio::on_shoutcast_selection_changed));

      m_shoutcast_g->s_start().connect
        (sigc::bind (sigc::mem_fun (*m_notebook_shoutcast, &Gtk::Notebook::set_current_page), 1));
      m_shoutcast_g->s_stop().connect
        (sigc::bind (sigc::mem_fun (*m_notebook_shoutcast, &Gtk::Notebook::set_current_page), 0));

      m_shoutcast_g->s_start().connect
        (sigc::bind (sigc::mem_fun (*m_shoutcast_g, &Gtk::Widget::set_sensitive), false));
      m_shoutcast_g->s_stop().connect
        (sigc::bind (sigc::mem_fun (*m_shoutcast_g, &Gtk::Widget::set_sensitive), true));

      m_shoutcast_g->s_list_updated().connect
        (sigc::mem_fun (*this, &Radio::on_shoutcast_list_updated));

      m_icecast->signal_row_activated().connect
        (sigc::mem_fun (*this, &Radio::on_row_activated));
      m_shoutcast->signal_row_activated().connect
        (sigc::mem_fun (*this, &Radio::on_row_activated));

      m_actions = ActionGroup::create ("Actions_Radio");
      m_actions->add (Action::create ("MenuUiPartRadio", _("Radio")));

      m_actions->add (Action::create (RADIO_ACTION_PLAY_STREAM,
                                      Gtk::StockID (BMP_STOCK_LINK),
                                      _("Play Stream...")),
                                      (sigc::mem_fun (*this, &Bmp::UiPart::Radio::on_play_stream)));

      m_actions->add (Action::create (RADIO_ACTION_UPDATE_LIST,
                                      Gtk::StockID (GTK_STOCK_GO_FORWARD),
                                      _("Update List")),
                                      (sigc::mem_fun (*this, &Bmp::UiPart::Radio::on_update_list)));


      Gtk::RadioButtonGroup gr1;
      m_actions->add  (RadioAction::create (gr1, RADIO_ACTION_SHOW_SHOUTCAST,
                                                "Shoutcast"),
                                            (sigc::mem_fun (*this, &Radio::on_set_notebook_page)));
      m_actions->add  (RadioAction::create (gr1, RADIO_ACTION_SHOW_ICECAST,
                                                "Icecast"),
                                            (sigc::mem_fun (*this, &Radio::on_set_notebook_page)));

      RefPtr<Gtk::RadioAction>::cast_static (m_actions->get_action (RADIO_ACTION_SHOW_SHOUTCAST))->property_value() = 0; 
      RefPtr<Gtk::RadioAction>::cast_static (m_actions->get_action (RADIO_ACTION_SHOW_ICECAST))->property_value() = 1; 
      m_ui_manager->insert_action_group (m_actions);

      on_set_notebook_page ();
    }

    void
    Radio::on_set_notebook_page ()
    {
      int page = RefPtr<Gtk::RadioAction>::cast_static (m_actions->get_action (RADIO_ACTION_SHOW_SHOUTCAST))->get_current_value();
      m_notebook_radio->set_current_page (page);

      bool has_selected_rows = false;
      switch (page)
      {
            case PAGE_SHOUTCAST:
                m_shoutcast->filter (m_filter_entry->get_text());
                has_selected_rows = (m_icecast->get_selection ()->count_selected_rows() == 1);
                break;

            case PAGE_ICECAST:
                m_icecast->filter (m_filter_entry->get_text());
                has_selected_rows = (m_shoutcast->get_selection ()->count_selected_rows() == 1);
                break;
      }

      if( has_selected_rows )
            m_caps = Caps (m_caps |  PlaybackSource::CAN_PLAY);
      else
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
      s_caps_.emit (m_caps);
    }

    void
    Radio::on_update_list ()
    {
      if (m_notebook_radio->get_current_page () == 0)
      {
        m_shoutcast_g->refresh (true);
      }
      else if (m_notebook_radio->get_current_page () == 1)
      {
        m_notebook_icecast->set_current_page (1);
        m_icecast->refresh();
        m_icecast->filter (m_filter_entry->get_text());
        m_notebook_icecast->set_current_page (0);
      }
    }

    void
    Radio::on_play_stream ()
    {
      DialogPlayUri * d = DialogPlayUri::create();
      ustring uri;
      int response = d->run (uri);
      delete d;

      if( response == GTK_RESPONSE_OK )
      {
            Bmp::URI u (uri);
            Bmp::URI::Protocol p = u.get_protocol();

            if( p == URI::PROTOCOL_MMS || p == URI::PROTOCOL_MMSU || p == URI::PROTOCOL_MMST )
            {
                  m_current_uri   = uri; 
                  m_current_name  = ustring();
                  s_playback_request_.emit ();
                  return;
            }

            if( p == URI::PROTOCOL_HTTP )
            {
                  if( Audio::is_audio_file (uri) )
                  {
                        m_current_uri = uri; 
                  }
                  else
                  {
                        VFS::Handle handle = VFS::Handle (uri);
                        VUri list;

                        try{
                                vfs->read (handle, list);
                          }
                        catch (VFS::Exception & cxe)
                          { 
                                return;
                          }

                        if( list.empty() )
                                m_current_uri = uri; 
                        else
                                m_current_uri = list[0];
                  }

                  m_current_name  = ustring();
                  s_playback_request_.emit ();
            }
      }
    }

    void
    Radio::on_row_activated (Gtk::TreeModel::Path const& path,
                             Gtk::TreeViewColumn       * column)
    {
      m_current_uri = ustring ();
      s_playback_request_.emit ();
    }

    void
    Radio::on_filter_changed ()
    {
      switch (m_notebook_radio->get_current_page ())
      {
            case PAGE_SHOUTCAST:
                m_shoutcast->filter (m_filter_entry->get_text());
                break;

            case PAGE_ICECAST:
                m_icecast->filter (m_filter_entry->get_text());
                break;
      }
    }

    void
    Radio::on_shoutcast_selection_changed ()
    {
      if( m_shoutcast->get_selection ()->count_selected_rows() == 1 )
            m_caps = Caps (m_caps |  PlaybackSource::CAN_PLAY);
      else
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
      s_caps_.emit (m_caps);
    }

    void
    Radio::on_shoutcast_list_updated (Streams::StreamList const& list)
    {
      m_shoutcast->set_streams (list);
      m_shoutcast->filter (m_filter_entry->get_text());
      m_notebook_shoutcast->set_current_page (0);
      m_shoutcast_g->set_sensitive (1);
      m_shoutcast_g->queue_draw ();
    }

    void
    Radio::on_icecast_selection_changed ()
    {
      if( m_icecast->get_selection ()->count_selected_rows() == 1 )
            m_caps = Caps (m_caps |  PlaybackSource::CAN_PLAY);
      else
            m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
      s_caps_.emit (m_caps);
    }

    // PlaybackSource 
    Glib::ustring
    Radio::get_uri ()
    {
      return m_current_uri;
    }

    bool
    Radio::go_next ()
    {
      return false;
    }

    bool
    Radio::go_prev ()
    {
      return false;
    }

    void
    Radio::stop ()
    {
      m_current_uri   = ustring();
      m_current_name  = ustring();
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PAUSE);
      s_caps_.emit (m_caps);
    }

    void
    Radio::play ()
    {
      if( !m_current_uri.empty() )
      {
            return;
      }

      int page = m_notebook_radio->get_current_page ();
      if( page == PAGE_SHOUTCAST )
      {
            ustring uri;
            m_shoutcast->get_stream (m_current_name, uri);
            VFS::Handle handle = VFS::Handle (uri);
            VUri list;

            try{
                    vfs->read (handle, list);
              }
            catch (VFS::Exception & cxe)
              { 
                    throw UnableToInitiatePlaybackError (cxe.what ());
              }

            m_current_uri = list[0];
      }

      if( page == PAGE_ICECAST )
      {
            m_icecast->get_stream (m_current_name, m_current_uri);
      }
    }

    void
    Radio::play_post ()
    {
      TrackMetadata metadata;
      metadata.artist = m_current_name; 
      metadata.title  = m_current_uri;
      s_track_metadata_.emit (metadata);
      m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);
      m_caps = Caps (m_caps | PlaybackSource::CAN_BOOKMARK);
    }

    void
    Radio::restore_context ()
    {
    }

    guint
    Radio::add_ui ()
    {
      return m_ui_manager->add_ui_from_string  (ui_string_radio);
    }

    // BookmarkHandler
    bool 
    Radio::get_bookmark (ustring const& id, /* the id to be used as the hostname part, but otherwise ignored */
                  ustring & url,
                  ustring & title)
    {
      static boost::format url_f ("ibs://%s/?streamurl=%s&title=%s");
  
      if( m_current_uri.empty() )
        return false;

      url = (url_f % id % m_current_uri % Bmp::URI::escape_string (m_current_name)).str();
      title = m_current_name;

      return true;
    }

    bool 
    Radio::set_bookmark (ustring const& url)
    {
      Bmp::URI u (url);
      Bmp::URI::Query q;
      u.parse_query (q);
    
      Bmp::URI::Query::iterator i = q.find ("streamurl");
      if( i != q.end() )
      {
        m_current_name = Bmp::URI::unescape_string (q.find ("title")->second.second);
        m_current_uri = i->second.second;
        s_playback_request_.emit ();
        return true;
      }
      return false;
    }

  } // UiPart
} // Bmp
