/***************************************************************************
 *            async-job-manager.c
 *
 *  ven avr  7 14:39:35 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "async-job-manager.h"
 
static void bonfire_async_job_manager_class_init (BonfireAsyncJobManagerClass *klass);
static void bonfire_async_job_manager_init (BonfireAsyncJobManager *sp);
static void bonfire_async_job_manager_finalize (GObject *object);

struct BonfireAsyncJobManagerPrivate {
	GCond *wake_cond;
	GMutex *lock;

	GSList *waiting_jobs;
	GSList *active_jobs;
	GSList *results;

	int num_threads;
	int unused_threads;

	gint results_id;

	GHashTable *types;
	int type_num;

	gint cancel:1;
};

static BonfireAsyncJobManager *manager = NULL;

struct _BonfireAsyncJobType {
	GObject *obj;
	BonfireAsyncRunJob run;
	BonfireSyncGetResult results;
	BonfireAsyncDestroy destroy;
	BonfireAsyncCancelJob cancel;
};
typedef struct _BonfireAsyncJobType BonfireAsyncJobType;

struct _BonfireAsyncJobCtx {
	BonfireAsyncJobType *common;
	gpointer data;

	int cancel:1;
};
typedef struct _BonfireAsyncJobCtx BonfireAsyncJobCtx;

#define MANAGER_MAX_THREAD 2

static GObjectClass *parent_class = NULL;

GType
bonfire_async_job_manager_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireAsyncJobManagerClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_async_job_manager_class_init,
			NULL,
			NULL,
			sizeof (BonfireAsyncJobManager),
			0,
			(GInstanceInitFunc)bonfire_async_job_manager_init,
		};

		type = g_type_register_static (G_TYPE_OBJECT,
					       "BonfireAsyncJobManager",
					       &our_info,
					       0);
	}

	return type;
}

static void
bonfire_async_job_manager_class_init (BonfireAsyncJobManagerClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_async_job_manager_finalize;
}

static void
bonfire_async_job_manager_init (BonfireAsyncJobManager *obj)
{
	obj->priv = g_new0 (BonfireAsyncJobManagerPrivate, 1);

	obj->priv->wake_cond = g_cond_new ();
	obj->priv->lock = g_mutex_new ();
}

static void
bonfire_async_job_destroy (BonfireAsyncJobCtx *ctx)
{
	if (!ctx->common->destroy)
		return;

	ctx->common->destroy (ctx->data);
	g_free (ctx);
}

static void
bonfire_async_job_cancel (BonfireAsyncJobCtx *ctx)
{
	ctx->cancel = 1;

	if (!ctx->common->cancel)
		return;

	ctx->common->cancel (ctx->data);
}

static void
bonfire_async_job_manager_finalize (GObject *object)
{
	BonfireAsyncJobManager *cobj;

	cobj = BONFIRE_ASYNC_JOB_MANAGER (object);

	/* stop the threads first */
	g_mutex_lock (cobj->priv->lock);
	cobj->priv->cancel = TRUE;

	/* remove all the waiting jobs */
	g_slist_foreach (manager->priv->waiting_jobs,
			 (GFunc) bonfire_async_job_destroy,
			 NULL);
	g_slist_free (manager->priv->waiting_jobs);
	manager->priv->waiting_jobs = NULL;

	/* cancel all the active jobs */
	g_slist_foreach (manager->priv->active_jobs,
			 (GFunc) bonfire_async_job_cancel,
			 NULL);

	/* terminate all sleeping threads */
	g_cond_broadcast (cobj->priv->wake_cond);

	/* Now we wait for the active thread queue to be empty */
	while (cobj->priv->num_threads)
		g_cond_wait (cobj->priv->wake_cond, cobj->priv->lock);

	g_mutex_unlock (cobj->priv->lock);

	if (cobj->priv->wake_cond) {
		g_cond_free (cobj->priv->wake_cond);
		cobj->priv->wake_cond = NULL;
	}

	if (cobj->priv->lock) {
		g_mutex_free (cobj->priv->lock);
		cobj->priv->lock = NULL;
	}

	if (cobj->priv->results_id) {
		g_source_remove (cobj->priv->results_id);
		cobj->priv->results_id = 0;
	}

	/* remove all the waiting jobs */
	g_slist_foreach (manager->priv->results,
			 (GFunc) bonfire_async_job_destroy,
			 NULL);
	g_slist_free (manager->priv->results);
	manager->priv->results = NULL;

	if (cobj->priv->types) {
		g_hash_table_destroy (cobj->priv->types);
		cobj->priv->types = NULL;
	}

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);

	manager = NULL;
}

BonfireAsyncJobManager *
bonfire_async_job_manager_get_default ()
{
	if (!manager)
		manager = BONFIRE_ASYNC_JOB_MANAGER (g_object_new(BONFIRE_TYPE_ASYNC_JOB_MANAGER, NULL));
	else
		g_object_ref (manager);

	return manager;
}

gint
bonfire_async_job_manager_register_type (BonfireAsyncJobManager *manager,
					 GObject *object,
					 BonfireAsyncRunJob run,
					 BonfireSyncGetResult results,
					 BonfireAsyncDestroy destroy,
					 BonfireAsyncCancelJob cancel)
{
	BonfireAsyncJobType *type;

	g_return_val_if_fail (manager != 0, 0);

	manager->priv->type_num ++;
	if (manager->priv->type_num == G_MAXINT) {
		manager->priv->type_num = 1;

		while (g_hash_table_lookup (manager->priv->types, GINT_TO_POINTER (manager->priv->type_num))) {
			manager->priv->type_num ++;

			if (manager->priv->type_num == G_MAXINT) {
				g_warning ("ERROR: reached the max number of types\n");
				return 0;
			}
		}
	}
	
	type = g_new0 (BonfireAsyncJobType, 1);
	type->obj = object;
	type->run = run;
	type->results = results;
	type->destroy = destroy;

	if (!manager->priv->types)
		manager->priv->types = g_hash_table_new_full (g_direct_hash,
							      g_direct_equal,
							      NULL,
							      g_free);

	g_hash_table_insert (manager->priv->types,
			     GINT_TO_POINTER (manager->priv->type_num),
			     type);

	return manager->priv->type_num;
}

void
bonfire_async_job_manager_unregister_type (BonfireAsyncJobManager *manager,
					   gint type)
{
	g_return_if_fail (manager != NULL);

	g_hash_table_remove (manager->priv->types, GINT_TO_POINTER (type));
}

static gboolean
bonfire_async_job_manager_wait_for_results (BonfireAsyncJobManager *manager)
{
	BonfireAsyncJobCtx *ctx;

	g_mutex_lock (manager->priv->lock);

	if (!manager->priv->results) {
		manager->priv->results_id = 0;
		g_mutex_unlock (manager->priv->lock);
		return FALSE;
	}

	ctx = manager->priv->results->data;
	manager->priv->results = g_slist_remove (manager->priv->results, ctx);

	g_mutex_unlock (manager->priv->lock);

	if (ctx->cancel) {
		bonfire_async_job_destroy (ctx);
		return TRUE;
	}

	if (ctx->common->results (ctx->common->obj, ctx->data))
		bonfire_async_job_destroy (ctx);
	else
		g_free (ctx);

	return TRUE;
}

static gpointer
bonfire_async_job_manager_thread (BonfireAsyncJobManager *manager)
{
	gboolean result;
	BonfireAsyncJobCtx *ctx;

	g_mutex_lock (manager->priv->lock);

	while (1) {
		/* say we are unused */
		manager->priv->unused_threads ++;
	
		/* see if a job is waiting to be executed */
		while (!manager->priv->waiting_jobs) {
			if (manager->priv->cancel)
				goto end;

			/* we always keep one thread ready */
			if (manager->priv->num_threads - manager->priv->unused_threads > 0) {
				GTimeVal timeout;

				/* wait to be woken up for 10 sec otherwise quit */
				g_get_current_time (&timeout);
				g_time_val_add (&timeout, 10000000);
				result = g_cond_timed_wait (manager->priv->wake_cond,
							    manager->priv->lock,
							    &timeout);

				if (!result)
					goto end;
			}
			else
				g_cond_wait (manager->priv->wake_cond,
					     manager->priv->lock);
		}
	
		/* say that we are active again */
		manager->priv->unused_threads --;
	
		/* get the data from the list */
		ctx = manager->priv->waiting_jobs->data;
		manager->priv->waiting_jobs = g_slist_remove (manager->priv->waiting_jobs, ctx);
		manager->priv->active_jobs = g_slist_prepend (manager->priv->active_jobs, ctx);
	
		g_mutex_unlock (manager->priv->lock);
	
		result = ctx->common->run (ctx->common->obj, ctx->data);
	
		g_mutex_lock (manager->priv->lock);
	
		if (!result || ctx->cancel || manager->priv->cancel) {
			bonfire_async_job_destroy (ctx);
			continue;
		}
	
		manager->priv->active_jobs = g_slist_remove (manager->priv->active_jobs, ctx);
		manager->priv->results = g_slist_append (manager->priv->results, ctx);
	
		if (!manager->priv->results_id)
			manager->priv->results_id = g_idle_add ((GSourceFunc) bonfire_async_job_manager_wait_for_results, manager);
	
	}

end:
	manager->priv->unused_threads --;
	manager->priv->num_threads --;

	/* maybe finalize is waiting for us to terminate */
	g_cond_broadcast (manager->priv->wake_cond);
	g_mutex_unlock (manager->priv->lock);

	return NULL;
}

gboolean
bonfire_async_job_manager_queue (BonfireAsyncJobManager *manager,
				 gint type,
				 gpointer data)
{
	BonfireAsyncJobCtx *ctx;

	g_return_val_if_fail (manager != NULL, FALSE);
	g_return_val_if_fail (type > 0, FALSE);

	ctx = g_new0 (BonfireAsyncJobCtx, 1);
	ctx->data = data;
	ctx->common = g_hash_table_lookup (manager->priv->types, GINT_TO_POINTER (type));

	g_mutex_lock (manager->priv->lock);
	manager->priv->waiting_jobs = g_slist_append (manager->priv->waiting_jobs, ctx);

	if (manager->priv->unused_threads) {
		/* wake up one thread in the list */
		g_cond_signal (manager->priv->wake_cond);
	}
	else if (manager->priv->num_threads < MANAGER_MAX_THREAD) {
		GError *error = NULL;
		GThread *thread;

		/* we have to start a new thread */
		thread = g_thread_create ((GThreadFunc) bonfire_async_job_manager_thread,
					  manager,
					  FALSE,
					  &error);

		if (!thread) {
			g_warning ("Can't start thread : %s\n", error->message);
			g_error_free (error);

			manager->priv->waiting_jobs = g_slist_remove (manager->priv->waiting_jobs, ctx);
			bonfire_async_job_destroy (ctx);

			g_mutex_unlock (manager->priv->lock);
			return FALSE;
		}

		manager->priv->num_threads++;
	}
	/* else we wait for a currently active thread to be available */
	g_mutex_unlock (manager->priv->lock);

	return TRUE;
}

void
bonfire_async_job_manager_cancel_by_object (BonfireAsyncJobManager *manager,
					    GObject *obj)
{
	GSList *iter, *next;
	BonfireAsyncJobCtx *ctx;

	g_return_if_fail (manager != NULL);
	g_return_if_fail (obj != NULL);

	g_mutex_lock (manager->priv->lock);

	/* find all jobs with this object */
	for (iter = manager->priv->waiting_jobs; iter; iter = next) {
		ctx = iter->data;
		next = iter->next;

		if (ctx->common->obj == obj) {
			manager->priv->waiting_jobs = g_slist_remove (manager->priv->waiting_jobs, ctx);
			bonfire_async_job_destroy (ctx);
		}
	}

	/* have a look at the active_jobs */
	for (iter = manager->priv->active_jobs; iter; iter = iter->next) {
		ctx = iter->data;

		/* we simply call cancel not destroy since it will be done by the 
		  thread handling the job. It's up to the callback_data to return FALSE */
		if (ctx->common->obj == obj)
			bonfire_async_job_cancel (ctx);
	}

	g_mutex_unlock (manager->priv->lock);
}

gboolean
bonfire_async_job_manager_find_urgent_job (BonfireAsyncJobManager *manager,
					   gint type,
					   BonfireAsyncFindJob func,
					   gpointer user_data)
{
	GSList *iter;
	BonfireAsyncJobCtx *ctx;
	BonfireAsyncJobType *real_type;

	g_return_val_if_fail (manager != NULL, FALSE);
	g_return_val_if_fail (func != NULL, FALSE);

	real_type = g_hash_table_lookup (manager->priv->types, GINT_TO_POINTER (type));

	g_mutex_lock (manager->priv->lock);
	for (iter = manager->priv->waiting_jobs; iter; iter = iter->next) {
		ctx = iter->data;

		if (ctx->common == real_type && func (ctx->data, user_data)) {
			manager->priv->waiting_jobs = g_slist_remove (manager->priv->waiting_jobs, ctx);
			manager->priv->waiting_jobs = g_slist_prepend (manager->priv->waiting_jobs, ctx);
			g_mutex_unlock (manager->priv->lock);
			return TRUE;
		}
	}
	g_mutex_unlock (manager->priv->lock);

	return FALSE;
}
