/***************************************************************************
 *            blank-dialog.c
 *
 *  mar jui 26 12:23:01 2005
 *  Copyright  2005  Philippe Rouquier
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>

#include <gtk/gtklabel.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkprogressbar.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkcheckbutton.h>

#include <nautilus-burn-drive-selection.h>
#include <nautilus-burn-recorder.h>

#include "burn.h"
#include "utils.h"
#include "burn-basics.h"
#include "blank-dialog.h"
#include "recorder-selection.h"

extern gint debug;

static void bonfire_blank_dialog_class_init (BonfireBlankDialogClass *
					     klass);
static void bonfire_blank_dialog_init (BonfireBlankDialog * sp);
static void bonfire_blank_dialog_finalize (GObject * object);
static void bonfire_blank_dialog_destroy (GtkObject * object);

static gboolean bonfire_blank_dialog_delete (GtkWidget * widget,
					     GdkEventAny * event);
static void bonfire_blank_dialog_cancel_clicked_cb (GtkWidget * button,
						    BonfireBlankDialog *
						    dialog);
static void bonfire_blank_dialog_blank_clicked_cb (GtkButton * button,
						   BonfireBlankDialog *
						   dialog);
static void bonfire_blank_dialog_device_changed_cb (NautilusBurnDriveSelection *selection,
						    NautilusBurnMediaType can_record,
						    BonfireBlankDialog *dialog);

static void bonfire_blank_dialog_device_opts_setup (BonfireBlankDialog *dialog,
						    NautilusBurnMediaType type,
						    gboolean messages);

struct BonfireBlankDialogPrivate {
	BonfireBurn *burn;

	GtkWidget *upper_box;
	GtkWidget *lower_box;
	GtkWidget *type_toggle;
	GtkWidget *dummy_toggle;
	GtkWidget *selector;
	GtkWidget *progress;
	GtkWidget *blank_button;
};

static GObjectClass *parent_class = NULL;

GType
bonfire_blank_dialog_get_type ()
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireBlankDialogClass),
			NULL,
			NULL,
			(GClassInitFunc) bonfire_blank_dialog_class_init,
			NULL,
			NULL,
			sizeof (BonfireBlankDialog),
			0,
			(GInstanceInitFunc) bonfire_blank_dialog_init,
		};

		type = g_type_register_static (GTK_TYPE_DIALOG,
					       "BonfireBlankDialog",
					       &our_info,
					       0);
	}

	return type;
}

static void
bonfire_blank_dialog_class_init (BonfireBlankDialogClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkObjectClass *gtk_object_class = GTK_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_blank_dialog_finalize;
	gtk_object_class->destroy = bonfire_blank_dialog_destroy;
	widget_class->delete_event = bonfire_blank_dialog_delete;
}

static void
bonfire_blank_dialog_init (BonfireBlankDialog *obj)
{
	GtkWidget *cancel;
	GtkWidget *title;
	GtkWidget *blank;

	obj->priv = g_new0 (BonfireBlankDialogPrivate, 1);
	gtk_window_set_default_size (GTK_WINDOW (obj), 500, 300);
//	gtk_dialog_set_has_separator (GTK_DIALOG (obj), FALSE);
//	gtk_window_set_position (GTK_WINDOW (obj),GTK_WIN_POS_CENTER_ALWAYS);

	/* upper part */
	obj->priv->upper_box = gtk_vbox_new (FALSE, 0);
	obj->priv->selector = bonfire_recorder_selection_new ();
	g_object_set (G_OBJECT (obj->priv->selector),
		      "show-recorders-only", TRUE,
		      "show-properties", FALSE,
		      "file-image", FALSE,
		      NULL);

	g_signal_connect (G_OBJECT (obj->priv->selector),
			  "media-changed",
			  G_CALLBACK (bonfire_blank_dialog_device_changed_cb),
			  obj);

	gtk_box_pack_start (GTK_BOX (obj->priv->upper_box),
			    bonfire_utils_pack_properties (_("<b>Select a recorder:</b>"),
							   obj->priv->selector,
							   NULL),
			    FALSE, FALSE, 0);

	bonfire_recorder_selection_select_default_drive (BONFIRE_RECORDER_SELECTION (obj->priv->selector),
							 BONFIRE_MEDIA_REWRITABLE);

	obj->priv->type_toggle = gtk_check_button_new_with_label (_("fast blanking"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (obj->priv->type_toggle), TRUE);
	obj->priv->dummy_toggle = gtk_check_button_new_with_label (_("simulation"));
	gtk_box_pack_start (GTK_BOX (obj->priv->upper_box),
			    bonfire_utils_pack_properties (_("<b>Options:</b>"),
							   obj->priv->type_toggle,
							   obj->priv->dummy_toggle,
							   NULL), FALSE,
			    FALSE, 0);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    obj->priv->upper_box,
			    FALSE,
			    FALSE,
			    0);
	gtk_widget_show_all (GTK_WIDGET (obj->priv->upper_box));

	/* lower part */
	obj->priv->lower_box = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (obj->priv->lower_box), 6);
	gtk_widget_set_sensitive (obj->priv->lower_box, FALSE);
	gtk_widget_show (obj->priv->lower_box);

	title = gtk_label_new (_("<b>Progress:</b>"));
	gtk_label_set_use_markup (GTK_LABEL (title), TRUE);
	gtk_misc_set_alignment (GTK_MISC (title), 0.0, 0.5);
	gtk_misc_set_padding(GTK_MISC (title), 0, 6);
	gtk_widget_show (title);
	gtk_box_pack_start (GTK_BOX (obj->priv->lower_box),
			    title,
			    FALSE,
			    FALSE,
			    0);

	obj->priv->progress = bonfire_burn_progress_new ();
	gtk_widget_show (obj->priv->progress);
	g_object_set (G_OBJECT (obj->priv->progress),
		      "show-info", FALSE,
		      NULL);

	gtk_box_pack_start (GTK_BOX (obj->priv->lower_box),
			    obj->priv->progress,
			    FALSE,
			    FALSE,
			    0);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    obj->priv->lower_box,
			    FALSE,
			    FALSE,
			    6);

	/* buttons */
	cancel = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	gtk_widget_show (cancel);
	g_signal_connect (G_OBJECT (cancel), "clicked",
			  G_CALLBACK (bonfire_blank_dialog_cancel_clicked_cb),
			  obj);
	gtk_dialog_add_action_widget (GTK_DIALOG (obj), cancel, GTK_RESPONSE_CANCEL);

	blank = bonfire_utils_make_button (_("Blank"), GTK_STOCK_CDROM);
	gtk_widget_show_all (blank);
	g_signal_connect (G_OBJECT (blank), "clicked",
			  G_CALLBACK (bonfire_blank_dialog_blank_clicked_cb),
			  obj);
	gtk_dialog_add_action_widget (GTK_DIALOG (obj), blank, GTK_RESPONSE_OK);
	obj->priv->blank_button = blank;
}

static void
bonfire_blank_dialog_finalize (GObject * object)
{
	BonfireBlankDialog *cobj;

	cobj = BONFIRE_BLANK_DIALOG (object);

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
bonfire_blank_dialog_destroy (GtkObject * object)
{
	BonfireBlankDialog *cobj;

	cobj = BONFIRE_BLANK_DIALOG (object);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

GtkWidget *
bonfire_blank_dialog_new ()
{
	BonfireBlankDialog *obj;
	NautilusBurnMediaType type;
	NautilusBurnDrive *drive = NULL;

	obj = BONFIRE_BLANK_DIALOG (g_object_new (BONFIRE_TYPE_BLANK_DIALOG,
				    "title", _("Disc blanking"), NULL));

	bonfire_recorder_selection_get_drive (BONFIRE_RECORDER_SELECTION (obj->priv->selector),
					      &drive,
					      NULL);
	type = nautilus_burn_drive_get_media_type (drive);
	nautilus_burn_drive_unref (drive);
	bonfire_blank_dialog_device_opts_setup (obj, type, FALSE);

	return GTK_WIDGET (obj);
}

static void
bonfire_blank_dialog_device_opts_setup (BonfireBlankDialog *dialog,
					NautilusBurnMediaType type,
					gboolean messages)
{
	if (type == NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN
	||  type == NAUTILUS_BURN_MEDIA_TYPE_ERROR
	||  type == NAUTILUS_BURN_MEDIA_TYPE_BUSY) {
		gtk_widget_set_sensitive (dialog->priv->type_toggle, FALSE);
		gtk_widget_set_sensitive (dialog->priv->dummy_toggle, FALSE);
		gtk_widget_set_sensitive (dialog->priv->blank_button, FALSE);
	}
	else if (type == NAUTILUS_BURN_MEDIA_TYPE_CDRW) {
		gtk_widget_set_sensitive (dialog->priv->type_toggle, TRUE);
		gtk_widget_set_sensitive (dialog->priv->dummy_toggle, TRUE);
		gtk_widget_set_sensitive (dialog->priv->blank_button, TRUE);
	}
	else if (type == NAUTILUS_BURN_MEDIA_TYPE_DVDRW) {
		if (messages
		&&  gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->dummy_toggle))) {
			GtkWidget *message;

			message = gtk_message_dialog_new (GTK_WINDOW (dialog),
							  GTK_DIALOG_MODAL |
							  GTK_DIALOG_DESTROY_WITH_PARENT,
							  GTK_MESSAGE_INFO,
							  GTK_BUTTONS_CLOSE,
							  _("DVDRW discs don't support simulated blanking."));

			gtk_window_set_title (GTK_WINDOW (message), _("Unneeded operation"));

			gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
								  _("You can nevertheless blank it."));
			gtk_dialog_run (GTK_DIALOG (message));
			gtk_widget_destroy (message);
		}

		/* simulation doesn't work with DVDs */
		gtk_widget_set_sensitive (dialog->priv->type_toggle, TRUE);
		gtk_widget_set_sensitive (dialog->priv->dummy_toggle, FALSE);
		gtk_widget_set_sensitive (dialog->priv->blank_button, TRUE);
	}
	else if (type == NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW) {
		if (messages
		&&  gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->type_toggle))) {
			GtkWidget *message;

			/* Tell the user fast blanking is not required with this kind of drive */
			message = gtk_message_dialog_new (GTK_WINDOW (dialog),
							  GTK_DIALOG_MODAL |
							  GTK_DIALOG_DESTROY_WITH_PARENT,
							  GTK_MESSAGE_INFO,
							  GTK_BUTTONS_CLOSE,
							  _("You can use this type of DVD without prior blanking.\n"
							    "NOTE: it doesn't support simulation."));

			gtk_window_set_title (GTK_WINDOW (message), _("Unneeded operation"));

			gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
								  _("You can nevertheless blank it with the slow option if you want to."));
			gtk_dialog_run (GTK_DIALOG (message));
			gtk_widget_destroy (message);
		}

		/* simulation doesn't work with DVDs */
		/* this type can't go with fast blanking so grey it out */
		gtk_widget_set_sensitive (dialog->priv->type_toggle, TRUE);
		gtk_widget_set_sensitive (dialog->priv->dummy_toggle, FALSE);
		gtk_widget_set_sensitive (dialog->priv->blank_button, TRUE);
	}
	else {
		if (messages) {
			GtkWidget *message;
	
			/* we don't need / can't blank(ing) so tell the user */
			message = gtk_message_dialog_new (GTK_WINDOW (dialog),
							  GTK_DIALOG_MODAL |
							  GTK_DIALOG_DESTROY_WITH_PARENT,
							  GTK_MESSAGE_INFO,
							  GTK_BUTTONS_CLOSE,
							  _("This type of disc can't or doesn't require to be blanked."));

			gtk_window_set_title (GTK_WINDOW (message), _("Unneeded operation"));

			gtk_dialog_run (GTK_DIALOG (message));
			gtk_widget_destroy (message);
		}

		gtk_widget_set_sensitive (dialog->priv->type_toggle, FALSE);
		gtk_widget_set_sensitive (dialog->priv->dummy_toggle, FALSE);
		gtk_widget_set_sensitive (dialog->priv->blank_button, FALSE);
	}
}

static void
bonfire_blank_dialog_device_changed_cb (NautilusBurnDriveSelection *selection,
					NautilusBurnMediaType media_type,
					BonfireBlankDialog *dialog)
{
	bonfire_blank_dialog_device_opts_setup (dialog, media_type, TRUE);
}

static gboolean
bonfire_blank_dialog_cancel_dialog (GtkWidget *toplevel)
{
	gint result;
	GtkWidget *message;

	message = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					  GTK_DIALOG_DESTROY_WITH_PARENT|
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  _("Do you really want to quit?"));

	gtk_window_set_title (GTK_WINDOW (message), _("Confirm"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  _("Interrupting the process may make disc unusable."));

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	if (result != GTK_RESPONSE_OK)
		return FALSE;

	return TRUE;
}

static gboolean
bonfire_blank_dialog_delete (GtkWidget *widget, GdkEventAny *event)
{
	BonfireBlankDialog *dialog;
	BonfireBurnResult result = BONFIRE_BURN_OK;

	dialog = BONFIRE_BLANK_DIALOG (widget);

	if (dialog->priv->burn)
		result = bonfire_burn_cancel (dialog->priv->burn, TRUE);

	if (result == BONFIRE_BURN_DANGEROUS) {
		if (bonfire_blank_dialog_cancel_dialog (GTK_WIDGET (dialog))) {
			if (dialog->priv->burn)
				bonfire_burn_cancel (dialog->priv->burn, FALSE);
		}
		else
			return FALSE;
	}
	else
		gtk_widget_destroy (widget);

	if (GTK_WIDGET_CLASS (parent_class)->delete_event)
		return GTK_WIDGET_CLASS (parent_class)->delete_event (widget, event);
	else
		return FALSE;
}

static void
bonfire_blank_dialog_cancel_clicked_cb (GtkWidget *button,
					BonfireBlankDialog *dialog)
{
	if (dialog->priv->burn) {
		BonfireBurnResult result;

		if (dialog->priv->burn)
			result = bonfire_burn_cancel (dialog->priv->burn, TRUE);

		if (result == BONFIRE_BURN_DANGEROUS) {
			if (bonfire_blank_dialog_cancel_dialog (GTK_WIDGET (dialog))) {
				if (dialog->priv->burn)
					bonfire_burn_cancel (dialog->priv->burn, FALSE);
			}
		}
	}
	else
		gtk_widget_destroy (GTK_WIDGET (dialog));
}

static BonfireBurnResult
bonfire_dialog_common_insert_disc_cb (BonfireBurn *burn,
				      BonfireBurnError error,
				      BonfireMediaType type,
				      BonfireBlankDialog *dialog)
{
	return BONFIRE_BURN_CANCEL;
}

static void
bonfire_blank_dialog_blank_clicked_cb (GtkButton *button,
				       BonfireBlankDialog *dialog)
{
	NautilusBurnDrive *drive;
	BonfireBurnResult result;
	BonfireBurnFlag flags;
	GError *error = NULL;
	gboolean fast_blank;
	GdkCursor *cursor;

	/* set up */
	gtk_widget_set_sensitive (dialog->priv->upper_box, FALSE);
	gtk_widget_set_sensitive (dialog->priv->lower_box, TRUE);
	gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);

	cursor = gdk_cursor_new (GDK_WATCH);
	gdk_window_set_cursor (GTK_WIDGET (dialog)->window, cursor);
	gdk_cursor_unref (cursor);

	/* get the drive */
	bonfire_recorder_selection_get_drive (BONFIRE_RECORDER_SELECTION (dialog->priv->selector),
					      &drive,
					      NULL);

	/* get the flags */
	flags = BONFIRE_BURN_FLAG_EJECT | BONFIRE_BURN_FLAG_NOGRACE;
	if (debug)
		flags |= BONFIRE_BURN_FLAG_DEBUG;

	if (GTK_WIDGET_SENSITIVE (dialog->priv->dummy_toggle)
	&&  gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->dummy_toggle)))
		flags |= BONFIRE_BURN_FLAG_DUMMY;

	if (GTK_WIDGET_SENSITIVE (dialog->priv->type_toggle)
	&&  gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->type_toggle)))
		fast_blank = TRUE;
	else
		fast_blank = FALSE;

	dialog->priv->burn = bonfire_burn_new ();
	bonfire_burn_plug_progress_widget (dialog->priv->burn,
					   BONFIRE_BURN_PROGRESS (dialog->priv->progress));

	g_signal_connect (G_OBJECT (dialog->priv->burn),
			  "insert_media",
			  G_CALLBACK (bonfire_dialog_common_insert_disc_cb),
			  dialog);

	result = bonfire_burn_blank (dialog->priv->burn,
				     flags,
				     drive,
				     fast_blank,
				     &error);

	/* Tell the user the result of the operation */
	if (result == BONFIRE_BURN_ERR || error) {
		GtkWidget *message;

		message =  gtk_message_dialog_new (GTK_WINDOW (dialog),
						   GTK_DIALOG_DESTROY_WITH_PARENT|
						   GTK_DIALOG_MODAL,
						   GTK_MESSAGE_ERROR,
						   GTK_BUTTONS_CLOSE,
						   _("Error Blanking:"));

		gtk_window_set_title (GTK_WINDOW (dialog), _("Blanking finished"));

		if (error) {
			gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
								  "%s.",
								  error->message);
			g_error_free (error);
		}
		else
			gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
								  _("Unexpected error"));

		gtk_dialog_run (GTK_DIALOG (message));
		gtk_widget_destroy (message);
	}
	else if (result == BONFIRE_BURN_OK) {
		GtkWidget *message;

		message = gtk_message_dialog_new (GTK_WINDOW (dialog),
						  GTK_DIALOG_DESTROY_WITH_PARENT|
						  GTK_DIALOG_MODAL,
						  GTK_MESSAGE_INFO,
						  GTK_BUTTONS_CLOSE,
						  _("The disc was successfully blanked:"));

		gtk_window_set_title (GTK_WINDOW (dialog), _("Blanking finished"));

		gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
							  _("the disc is ready for use."));
		gtk_dialog_run (GTK_DIALOG (message));
		gtk_widget_destroy (message);
	}
	else if (result == BONFIRE_BURN_NOT_SUPPORTED) {
		g_warning ("operation not supported");
	}
	else if (result == BONFIRE_BURN_NOT_READY) {
		g_warning ("operation not ready");
	}
	else if (result == BONFIRE_BURN_NOT_RUNNING) {
		g_warning ("job not running");
	}
	else if (result == BONFIRE_BURN_RUNNING) {
		g_warning ("job running");
	}

	nautilus_burn_drive_unref (drive);
	g_object_unref (dialog->priv->burn);
	dialog->priv->burn = NULL;
	gtk_widget_destroy (GTK_WIDGET (dialog));
}
