/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
/***************************************************************************
 *            bonfire-session.c
 *
 *  Thu May 18 18:32:37 2006
 *  Copyright  2006  Philippe Rouquier
 *  <bonfire-app@wanadoo.fr>
 ****************************************************************************/

#include <string.h>

#include <glib.h>
#include <glib/gstdio.h>

#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-client.h>

#include <libxml/xmlerror.h>
#include <libxml/xmlwriter.h>
#include <libxml/parser.h>
#include <libxml/xmlstring.h>

#include "bonfire-app.h"

static GnomeClient *client = NULL;

#define SESSION_VERSION "0.1"

static char*
bonfire_session_get_file_path (void)
{
	return gnome_util_home_file ("bonfire.session");
}

gboolean
bonfire_session_load (BonfireApp *app)
{
	char *height_str = NULL;
	char *width_str = NULL;
	char *state_str = NULL;
	char *version = NULL;
	int height = 640;
	int width = 800;
	int state = 0;

	char *session_path;
	xmlNodePtr item;
	xmlDocPtr session = NULL;

	session_path = bonfire_session_get_file_path ();
	if (!session_path)
		goto end;

	session = xmlParseFile (session_path);
	g_free (session_path);

	if (!session)
		goto end;

	item = xmlDocGetRootElement (session);
	if (!item)
		goto end;

	if (xmlStrcmp (item->name, (const xmlChar *) "Session") || item->next)
		goto end;

	item = item->children;
	while (item) {
		if (!xmlStrcmp (item->name, (const xmlChar *) "version")) {
			if (version)
				goto end;

			version = (char *) xmlNodeListGetString (session,
								 item->xmlChildrenNode,
								 1);
		}
		else if (!xmlStrcmp (item->name, (const xmlChar *) "width")) {
			if (width_str)
				goto end;

			width_str = (char *) xmlNodeListGetString (session,
								   item->xmlChildrenNode,
								   1);
		}
		else if (!xmlStrcmp (item->name, (const xmlChar *) "height")) {
			if (height_str)
				goto end;

			height_str = (char *) xmlNodeListGetString (session,
								    item->xmlChildrenNode,
								    1);
		}
		else if (!xmlStrcmp (item->name, (const xmlChar *) "state")) {
			if (state_str)
				goto end;

			state_str = (char *) xmlNodeListGetString (session,
								   item->xmlChildrenNode,
								   1);
		}
		else if (item->type == XML_ELEMENT_NODE)
			goto end;

		item = item->next;
	}

	if (!version || strcmp (version, SESSION_VERSION))
		goto end;

	/* restore the window state */
	if (height_str)
		height = (int) g_strtod (height_str, NULL);

	if (width_str)
		width = (int) g_strtod (width_str, NULL);

	if (state_str)
		state = (int) g_strtod (state_str, NULL);

end:
	if (height_str)
		g_free (height_str);

	if (width_str)
		g_free (width_str);

	if (state_str)
		g_free (state_str);

	if (version)
		g_free (version);

	xmlFreeDoc (session);

	gtk_window_set_default_size (GTK_WINDOW (app->mainwin),
				     width,
				     height);

	if (state)
		gtk_window_maximize (GTK_WINDOW (app->mainwin));

	return TRUE;
}

gboolean
bonfire_session_save (BonfireApp *app)
{
	int success;
	char *session_path;
	xmlTextWriter *session;

	session_path = bonfire_session_get_file_path ();
	if (!session_path)
		return FALSE;

	/* write information */
	session = xmlNewTextWriterFilename (session_path, 0);
	if (!session) {
		g_free (session_path);
		return FALSE;
	}

	xmlTextWriterSetIndent (session, 1);
	xmlTextWriterSetIndentString (session, (xmlChar *) "\t");

	success = xmlTextWriterStartDocument (session,
					      NULL,
					      NULL,
					      NULL);
	if (success < 0)
		goto error;

	success = xmlTextWriterStartElement (session,
					     (xmlChar *) "Session");
	if (success < 0)
		goto error;

	success = xmlTextWriterWriteElement (session,
					     (xmlChar *) "version",
					     (xmlChar *) SESSION_VERSION);
	if (success < 0)
		goto error;

	success = xmlTextWriterWriteFormatElement (session,
						   (xmlChar *) "width",
						   "%i",
						   app->width);
	if (success < 0)
		goto error;

	success = xmlTextWriterWriteFormatElement (session,
						   (xmlChar *) "height",
						   "%i",
						   app->height);
	if (success < 0)
		goto error;

	success = xmlTextWriterWriteFormatElement (session,
						   (xmlChar *) "state",
						   "%i",
						   app->is_maximised);
	if (success < 0)
		goto error;

	success = xmlTextWriterEndElement (session);
	if (success < 0)
		goto error;

	xmlTextWriterEndDocument (session);
	xmlFreeTextWriter (session);
	g_free (session_path);

	return TRUE;

error:
	xmlTextWriterEndDocument (session);
	xmlFreeTextWriter (session);
	g_remove (session_path);
	g_free (session_path);

	return FALSE;
}

static void
bonfire_session_die_cb (GnomeClient *client_loc,
			BonfireApp *app)
{
	bonfire_session_save (app);
	gtk_widget_destroy (app->mainwin);
}

static gboolean
bonfire_session_save_yourself_cb (GnomeClient *client_loc,
				  gint phase,
				  GnomeSaveStyle arg2,
				  gboolean is_shutting_down,
				  GnomeInteractStyle allowed_interaction,
				  gboolean fast_shutdown,
				  BonfireApp *app)
{
	/* if we want to save the current open project, this need a
	 * modification in BonfireProject to bypass ask_status in case
	 * DataDisc has not finished exploration */
	bonfire_session_save (app);
	gtk_widget_destroy (app->mainwin);
	return TRUE; /* successs */
}

gboolean
bonfire_session_connect (BonfireApp *app)
{
	/* connect to the session manager */
	if (client)
		return TRUE;

	client = gnome_master_client ();
	if (client) {
		g_signal_connect (client,
				  "die",
				  G_CALLBACK (bonfire_session_die_cb),
				  app);
		g_signal_connect (client,
				  "save-yourself",
				  G_CALLBACK (bonfire_session_save_yourself_cb),
				  app);

		return TRUE;
	}

	return FALSE;
}

void
bonfire_session_disconnect (BonfireApp *app)
{
	g_signal_handlers_disconnect_by_func (client,
					      bonfire_session_die_cb,
					      app);
	g_signal_handlers_disconnect_by_func (client,
					      bonfire_session_save_yourself_cb,
					      app);

	client = NULL;
}
