/***************************************************************************
 *            cdrdao.c
 *
 *  dim jan 22 15:38:18 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <nautilus-burn-drive.h>

#include "burn-cdrdao.h"
#include "burn-common.h"
#include "burn-basics.h"
#include "burn-process.h"
#include "burn-recorder.h"
#include "burn-imager.h"

static void bonfire_cdrdao_class_init (BonfireCdrdaoClass *klass);
static void bonfire_cdrdao_init (BonfireCdrdao *sp);
static void bonfire_cdrdao_finalize (GObject *object);
static void bonfire_cdrdao_iface_init_record (BonfireRecorderIFace *iface);
static void bonfire_cdrdao_iface_init_image (BonfireImagerIFace *iface);

static BonfireBurnResult
bonfire_cdrdao_read_stderr (BonfireProcess *process,
			    const char *line);
static BonfireBurnResult
bonfire_cdrdao_set_argv (BonfireProcess *process,
			 GPtrArray *argv,
			 gboolean has_master, 
			 GError **error);
static BonfireBurnResult
bonfire_cdrdao_post (BonfireProcess *process,
		     BonfireBurnResult retval);

static BonfireBurnResult
bonfire_cdrdao_get_record_status (BonfireRecorder *recorder,
				  int *speed,
				  int *fifo,
				  gint64 *bytes_written);

static BonfireBurnResult
bonfire_cdrdao_set_drive (BonfireRecorder *recorder,
			  NautilusBurnDrive *drive,
			  GError **error);

static BonfireBurnResult
bonfire_cdrdao_set_flags (BonfireRecorder *recorder,
			  BonfireRecorderFlag flags,
			  GError **error);
static BonfireBurnResult
bonfire_cdrdao_set_speed (BonfireJob *job,
			  gint64 speed);

static BonfireBurnResult
bonfire_cdrdao_record (BonfireRecorder *recorder,
		       GError **error);
static BonfireBurnResult
bonfire_cdrdao_blank (BonfireRecorder *recorder,
		      GError **error);

static BonfireBurnResult
bonfire_cdrdao_get_track_type (BonfireImager *imager,
			       BonfireTrackSourceType *type);

static BonfireBurnResult
bonfire_cdrdao_get_size_image (BonfireImager *imager,
			       gint64 *size,
			       gboolean sectors,
			       GError **error);
static BonfireBurnResult
bonfire_cdrdao_get_speed_image (BonfireImager *imager,
				int *speed);
static BonfireBurnResult
bonfire_cdrdao_get_track (BonfireImager *imager,
			  BonfireTrackSource **track,
			  GError **error);

static BonfireBurnResult
bonfire_cdrdao_set_source (BonfireJob *job,
			   const BonfireTrackSource *source,
			   GError **error);
static BonfireBurnResult
bonfire_cdrdao_set_output (BonfireImager *imager,
			   const char *ouput,
			   gboolean overwrite,
			   gboolean clean,
			   GError **error);
static BonfireBurnResult
bonfire_cdrdao_set_output_type (BonfireImager *imager,
				BonfireTrackSourceType type,
				GError **error);

static BonfireBurnResult
bonfire_cdrdao_get_action_string (BonfireJob *job, 
				  BonfireBurnAction action,
				  char **string);

typedef enum {
	BONFIRE_CDRDAO_ACTION_NONE,
	BONFIRE_CDRDAO_ACTION_IMAGE,
	BONFIRE_CDRDAO_ACTION_GET_SIZE,
	BONFIRE_CDRDAO_ACTION_RECORD,
	BONFIRE_CDRDAO_ACTION_BLANK,
} BonfireCdrdaoAction;

struct BonfireCdrdaoPrivate {
	BonfireCdrdaoAction action;

	GTimer *timer;
	GSList *rates;
	guint64 start;

	BonfireTrackSource *track;
	NautilusBurnDrive *drive;

	char *src_output;
	char *toc;

	int fifo;
	int speed;               /* speed at which we should write */
	int cur_speed;           /* speed at which we're actually writing/imaging */

	int track_num;
	int track_start;

	int sectors_num;
	gint64 isosize;
	gint64 b_written;

	int dao:1;
	int dummy:1;
	int multi:1;
	int nograce:1;
	int overburn:1;
	int burnproof:1;

	int overwrite:1;
	int clean:1;

	int blank_fast:1;

	int track_ready:1;
};

static GObjectClass *parent_class = NULL;

GType
bonfire_cdrdao_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireCdrdaoClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_cdrdao_class_init,
			NULL,
			NULL,
			sizeof (BonfireCdrdao),
			0,
			(GInstanceInitFunc)bonfire_cdrdao_init,
		};

		static const GInterfaceInfo imager_info =
		{
			(GInterfaceInitFunc) bonfire_cdrdao_iface_init_image,
			NULL,
			NULL
		};
		static const GInterfaceInfo recorder_info =
		{
			(GInterfaceInitFunc) bonfire_cdrdao_iface_init_record,
			NULL,
			NULL
		};

		type = g_type_register_static (BONFIRE_TYPE_PROCESS, 
					       "BonfireCdrdao", 
					       &our_info,
					       0);
		g_type_add_interface_static (type,
					     BONFIRE_TYPE_IMAGER,
					     &imager_info);
		g_type_add_interface_static (type,
					     BONFIRE_TYPE_RECORDER,
					     &recorder_info);
	}

	return type;
}

static void
bonfire_cdrdao_iface_init_record (BonfireRecorderIFace *iface)
{
	iface->get_status = bonfire_cdrdao_get_record_status;
	iface->set_drive = bonfire_cdrdao_set_drive;
	iface->set_flags = bonfire_cdrdao_set_flags;
	iface->record = bonfire_cdrdao_record;
	iface->blank = bonfire_cdrdao_blank;
}

static void
bonfire_cdrdao_iface_init_image (BonfireImagerIFace *iface)
{
	iface->get_size = bonfire_cdrdao_get_size_image;
	iface->get_speed = bonfire_cdrdao_get_speed_image;
	iface->get_track = bonfire_cdrdao_get_track;
	iface->get_track_type = bonfire_cdrdao_get_track_type;

	iface->set_output = bonfire_cdrdao_set_output;
	iface->set_output_type = bonfire_cdrdao_set_output_type;
}

static void
bonfire_cdrdao_class_init (BonfireCdrdaoClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	BonfireJobClass *job_class = BONFIRE_JOB_CLASS (klass);
	BonfireProcessClass *process_class = BONFIRE_PROCESS_CLASS (klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_cdrdao_finalize;

	job_class->set_source = bonfire_cdrdao_set_source;
	job_class->set_rate = bonfire_cdrdao_set_speed;
	job_class->get_action_string = bonfire_cdrdao_get_action_string;

	process_class->stderr_func = bonfire_cdrdao_read_stderr;
	process_class->set_argv = bonfire_cdrdao_set_argv;
	process_class->post = bonfire_cdrdao_post;
}

static void
bonfire_cdrdao_init (BonfireCdrdao *obj)
{
	obj->priv = g_new0(BonfireCdrdaoPrivate, 1);
	obj->priv->isosize = -1;
	obj->priv->clean = TRUE;
	obj->priv->cur_speed = -1;
}

static void
bonfire_cdrdao_finalize (GObject *object)
{
	BonfireCdrdao *cobj;
	cobj = BONFIRE_CDRDAO(object);

	if (cobj->priv->timer) {
		g_timer_destroy (cobj->priv->timer);
		cobj->priv->timer = NULL;
	}
	
	if (cobj->priv->drive) {
		nautilus_burn_drive_unref (cobj->priv->drive);
		cobj->priv->drive = NULL;
	}
	
	if (cobj->priv->rates) {
		g_slist_free (cobj->priv->rates);
		cobj->priv->rates = NULL;
	}

	if (cobj->priv->track) {
		bonfire_track_source_free (cobj->priv->track);
		cobj->priv->track = NULL;
	}

	if (cobj->priv->src_output) {
		if (cobj->priv->clean && cobj->priv->track_ready)
			g_remove (cobj->priv->src_output);

		g_free (cobj->priv->src_output);
		cobj->priv->src_output = NULL;
	}

	if (cobj->priv->toc) {
		if (cobj->priv->clean  && cobj->priv->track_ready)
			g_remove (cobj->priv->toc);

		g_free (cobj->priv->toc);
		cobj->priv->toc = NULL;
	}

	g_free(cobj->priv);
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

BonfireCdrdao *
bonfire_cdrdao_new ()
{
	BonfireCdrdao *obj;
	
	obj = BONFIRE_CDRDAO (g_object_new (BONFIRE_TYPE_CDRDAO, NULL));
	
	return obj;
}

static gboolean
bonfire_cdrdao_read_stderr_image (BonfireCdrdao *cdrdao, const char *line)
{
	int min, sec, sub, s1;

	if (sscanf (line, "%d:%d:%d", &min, &sec, &sub) == 3) {
		if (cdrdao->priv->isosize > 0) {
			gdouble fraction;
			long remaining = -1;
			guint64 secs = min * 60 + sec;

			cdrdao->priv->cur_speed = -1;
			fraction = (gdouble) secs / cdrdao->priv->isosize;

			if (!cdrdao->priv->timer && secs > 2) {
				/* let the speed regulate before keeping track */
				cdrdao->priv->timer = g_timer_new ();
				cdrdao->priv->start = secs;
			}

			if (cdrdao->priv->timer) {
				gdouble elapsed;
				gdouble rate;

				elapsed = g_timer_elapsed (cdrdao->priv->timer, NULL);
				rate = (gdouble) (secs - cdrdao->priv->start) / elapsed;

				if (rate > 0.0) {
					gdouble ave_rate;

					ave_rate = bonfire_burn_common_get_average_rate (&cdrdao->priv->rates, rate);
					remaining = (cdrdao->priv->isosize - secs) / ave_rate;

					/* NOTE: for me (Philippe)
					 * ave_rate is sec/sec
					 * 1 sec = 75 sectors,
					 * 1 sector = 2352 bytes,
					 * speed (CD) = 153600 bytes (2048*75) / sec for data CD
					 * speed (CD) = 176400 bytes (2352*75) / sec for audio CD */
					cdrdao->priv->cur_speed = ave_rate;
				}
			}

			if (remaining >= 0)
				bonfire_job_progress_changed (BONFIRE_JOB (cdrdao),
							      fraction,
							      remaining);
			else
				bonfire_job_progress_changed (BONFIRE_JOB (cdrdao),
							      fraction,
							      -1);
		}
	}
	else if (sscanf (line, "Leadout %*s %*d %d:%d:%*d(%i)", &min, &sec, &s1) == 3) {
		/* we get the number of sectors. As we added -raw each sector = 2352 bytes */
		cdrdao->priv->sectors_num = s1;
		cdrdao->priv->isosize = min * 60 + sec;

		if (cdrdao->priv->action == BONFIRE_CDRDAO_ACTION_GET_SIZE)
			bonfire_job_finished (BONFIRE_JOB (cdrdao));
	}
	else if (strstr (line, "Copying audio tracks")) {
		bonfire_job_action_changed (BONFIRE_JOB (cdrdao),
					    BONFIRE_BURN_ACTION_DRIVE_COPY,
					    FALSE);
	}
	else if (strstr (line, "Copying data track")) {
		bonfire_job_action_changed (BONFIRE_JOB (cdrdao),
					    BONFIRE_BURN_ACTION_DRIVE_COPY,
					    FALSE);
	}
	else if (strstr (line, "toc and track data finished successfully")) {
		bonfire_job_progress_changed (BONFIRE_JOB (cdrdao),
					      1.0,
					      0);
	}
	else
		return FALSE;

	return TRUE;
}

static gboolean
bonfire_cdrdao_read_stderr_record (BonfireCdrdao *cdrdao, const char *line)
{
	int fifo, track, min, sec;
	guint written, total;

	if (sscanf (line, "Wrote %u of %u (Buffers %d%%  %*s", &written, &total, &fifo) >= 2) {
		long secs = -1;
		gdouble fraction;

		bonfire_job_set_dangerous (BONFIRE_JOB (cdrdao), TRUE);
		cdrdao->priv->fifo = fifo;
		cdrdao->priv->cur_speed = -1;
		cdrdao->priv->b_written = written * 1048576;

		fraction = (gdouble) written / total;

		if (!cdrdao->priv->timer) {
			/* let the speed regulate before keeping track */
			cdrdao->priv->timer = g_timer_new ();
			cdrdao->priv->start = (gint64) written;
		}
		else if (cdrdao->priv->timer) {
			gdouble elapsed;
			gdouble rate;

			elapsed = g_timer_elapsed (cdrdao->priv->timer, NULL);
			rate = (gdouble) (written - cdrdao->priv->start);

			if (rate > 0 && elapsed > 1) {
				gdouble ave_rate;

				rate /= (gdouble) elapsed;
				ave_rate = bonfire_burn_common_get_average_rate (&cdrdao->priv->rates, rate);

				secs = (total - written) / ave_rate;
				cdrdao->priv->cur_speed = ave_rate * 1048576 / 153600;
			}
		}

		bonfire_job_progress_changed (BONFIRE_JOB (cdrdao), 
					      fraction * 0.98,
					      secs);

		bonfire_job_action_changed (BONFIRE_JOB (cdrdao),
					    BONFIRE_BURN_ACTION_WRITING,
					    FALSE);
	}
	else if (sscanf (line, "Wrote %*s blocks. Buffer fill min") == 1) {
		/* this is for fixating phase */
		bonfire_job_progress_changed (BONFIRE_JOB (cdrdao), 
					      0.98,
					      -1);
		bonfire_job_action_changed (BONFIRE_JOB (cdrdao),
					    BONFIRE_BURN_ACTION_FIXATING,
					    FALSE);
	}
	else if (sscanf (line, "Analyzing track %d %*s start %d:%d:%*d, length %*d:%*d:%*d", &track, &min, &sec) == 3) {
		cdrdao->priv->track_num = track;
		cdrdao->priv->track_start = min * 60 + sec;

		bonfire_job_action_changed (BONFIRE_JOB (cdrdao),
					    BONFIRE_BURN_ACTION_ANALYSING,
					    TRUE);
	}
	else if (sscanf (line, "%d:%d:%*d", &min, &sec) == 2) {
		gdouble fraction;
		guint64 secs = min * 60 + sec + cdrdao->priv->track_start;

		if (!cdrdao->priv->track_start && !cdrdao->priv->isosize)
			return TRUE;

		fraction = (gdouble) secs / (gdouble) cdrdao->priv->isosize;
		bonfire_job_progress_changed (BONFIRE_JOB (cdrdao),
					      fraction,
					      -1);
	}
	else if (strstr (line, "Q sub-channels with CRC errors")) {
		cdrdao->priv->track_num = 0;
	}
	else if (strstr (line, "Starting on-the-fly CD copy")
	      ||  strstr (line, "Starting write at speed")) {
		bonfire_job_action_changed (BONFIRE_JOB (cdrdao),
					    BONFIRE_BURN_ACTION_PREPARING,
					    FALSE);
	}
	else if (strstr (line, "Executing power calibration")) {
		bonfire_job_action_changed (BONFIRE_JOB (cdrdao),
					    BONFIRE_BURN_ACTION_PREPARING,
					    FALSE);
	}
	else if (strstr (line, "Writing track")) {
		bonfire_job_set_dangerous (BONFIRE_JOB (cdrdao), TRUE);
		bonfire_job_action_changed (BONFIRE_JOB (cdrdao),
					    BONFIRE_BURN_ACTION_WRITING,
					    FALSE);
	}
	else if (strstr (line, "Writing finished successfully")
	     ||  strstr (line, "On-the-fly CD copying finished successfully")) {
		bonfire_job_set_dangerous (BONFIRE_JOB (cdrdao), FALSE);
		bonfire_job_progress_changed (BONFIRE_JOB (cdrdao), 1.0, -1);
	}
	else /* this is to report progress when a cd is being analysed before copy */
		return FALSE;

	return TRUE;
}

static BonfireBurnResult
bonfire_cdrdao_read_stderr (BonfireProcess *process, const char *line)
{
	BonfireCdrdao *cdrdao;
	gboolean result = FALSE;

	cdrdao = BONFIRE_CDRDAO (process);

	if (cdrdao->priv->action == BONFIRE_CDRDAO_ACTION_RECORD
	||  cdrdao->priv->action == BONFIRE_CDRDAO_ACTION_BLANK)
		result = bonfire_cdrdao_read_stderr_record (cdrdao, line);
	else if (cdrdao->priv->action == BONFIRE_CDRDAO_ACTION_IMAGE
	      ||  cdrdao->priv->action == BONFIRE_CDRDAO_ACTION_GET_SIZE)
		result = bonfire_cdrdao_read_stderr_image (cdrdao, line);

	if (result)
		return BONFIRE_BURN_OK;

	if (strstr (line, "Cannot setup device")) {
		bonfire_job_error (BONFIRE_JOB (cdrdao),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_BUSY_DRIVE,
						_("the drive seems to be busy")));
	}
	else if (strstr (line, "Illegal command")) {
		bonfire_job_error (BONFIRE_JOB (cdrdao),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_GENERAL,
						_("your version of cdrdao doesn't seem to be supported by libbonfire")));
	}
	else if (strstr (line, "Operation not permitted. Cannot send SCSI")) {
		bonfire_job_error (BONFIRE_JOB (cdrdao),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_SCSI_IOCTL,
						_("You don't seem to have the required permission to use this drive")));
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_get_track_type (BonfireImager *imager,
			       BonfireTrackSourceType *type)
{
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (imager);

	if (!cdrdao->priv->track)
		return BONFIRE_BURN_NOT_READY;

	if (cdrdao->priv->track->type != BONFIRE_TRACK_SOURCE_DISC)
		return BONFIRE_BURN_NOT_SUPPORTED;

	*type = BONFIRE_TRACK_SOURCE_CUE;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_set_source (BonfireJob *job,
			   const BonfireTrackSource *source,
			   GError **error)
{
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (job);

	/* Remove any current output */
	if (cdrdao->priv->src_output) {
		if (cdrdao->priv->clean && cdrdao->priv->track_ready)
			g_remove (cdrdao->priv->src_output);
		if (cdrdao->priv->clean && cdrdao->priv->track_ready)
			g_remove (cdrdao->priv->toc);
	}
	cdrdao->priv->track_ready = 0;

	/* NOTE: we can accept ourselves as our source (cdrdao is both imager
	 * and recorder). In this case we don't delete the previous source */
	if (source->type == BONFIRE_TRACK_SOURCE_IMAGER
	&&  BONFIRE_IMAGER (cdrdao) == source->contents.imager.obj)
		return BONFIRE_BURN_OK;

	if (cdrdao->priv->track) {
		bonfire_track_source_free (cdrdao->priv->track);
		cdrdao->priv->track = NULL;
	}

	/* NOTE: we don't check the type of source precisely since this check partly
	 * depends on what we do with cdrdao afterward (it's both imager/recorder) */
        if (source->type != BONFIRE_TRACK_SOURCE_DISC
	&&  source->type != BONFIRE_TRACK_SOURCE_CUE)
		return BONFIRE_BURN_NOT_SUPPORTED;

	cdrdao->priv->track = bonfire_track_source_copy (source);
	cdrdao->priv->isosize = -1;
	cdrdao->priv->sectors_num = 0;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_set_output (BonfireImager *imager,
			   const char *output,
			   gboolean overwrite,
			   gboolean clean,
			   GError **error)
{
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (imager);

	if (cdrdao->priv->src_output) {
		if (cdrdao->priv->clean && cdrdao->priv->track_ready)
			g_remove (cdrdao->priv->src_output);

		g_free (cdrdao->priv->src_output);
		cdrdao->priv->src_output = NULL;
	}

	if (cdrdao->priv->toc) {
		if (cdrdao->priv->clean && cdrdao->priv->track_ready)
			g_remove (cdrdao->priv->toc);

		g_free (cdrdao->priv->toc);
		cdrdao->priv->toc = NULL;
	}

	cdrdao->priv->track_ready = 0;

	if (output)
		cdrdao->priv->src_output = g_strdup (output);

	cdrdao->priv->overwrite = overwrite;
	cdrdao->priv->clean = clean;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_set_output_type (BonfireImager *imager,
				BonfireTrackSourceType type,
				GError **error)
{
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (imager);

	if (type != BONFIRE_TRACK_SOURCE_DEFAULT
	&&  type != BONFIRE_TRACK_SOURCE_CUE)
		return BONFIRE_BURN_NOT_SUPPORTED;

	/* NOTE: no need to keep this value since cdrdao can only output cue */
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_get_track (BonfireImager *imager,
			  BonfireTrackSource **track,
			  GError **error)
{
	BonfireCdrdao *cdrdao;
	BonfireTrackSource *retval;

	cdrdao = BONFIRE_CDRDAO (imager);

	if (!cdrdao->priv->track)
		return BONFIRE_BURN_NOT_READY;

	if (cdrdao->priv->track->type != BONFIRE_TRACK_SOURCE_DISC)
		return BONFIRE_BURN_NOT_SUPPORTED;

	if (!cdrdao->priv->track_ready) {
		BonfireBurnResult result;

		cdrdao->priv->action = BONFIRE_CDRDAO_ACTION_IMAGE;
		result = bonfire_job_run (BONFIRE_JOB (imager), error);
		cdrdao->priv->action = BONFIRE_CDRDAO_ACTION_NONE;

		if (result != BONFIRE_BURN_OK)
			return result;

		cdrdao->priv->track_ready = 1;
	}

	retval = g_new0 (BonfireTrackSource, 1);

	retval->type = BONFIRE_TRACK_SOURCE_CUE;
	retval->contents.cue.toc = g_strdup_printf ("%s.toc", cdrdao->priv->src_output);
	retval->contents.cue.image = g_strdup (cdrdao->priv->src_output);

	*track = retval;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_get_size_image (BonfireImager *imager,
			       gint64 *size,
			       gboolean sectors,
			       GError **error)
{
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (imager);

	if (cdrdao->priv->sectors_num < 1) {
		BonfireBurnResult result;

		if (bonfire_job_is_running (BONFIRE_JOB (imager)))
			return BONFIRE_BURN_RUNNING;

		/* unfortunately cdrdao doesn't allow it beforehand 
		 * so we cheat here : we start cdrdao and stop it as
		 * soon as we get the size */
		cdrdao->priv->action = BONFIRE_CDRDAO_ACTION_GET_SIZE;
		result = bonfire_job_run (BONFIRE_JOB (imager), error);
		cdrdao->priv->action = BONFIRE_CDRDAO_ACTION_NONE;

		if (result != BONFIRE_BURN_OK)
			return result;

		/* now we must remove the .toc and .bin in case they were created */
		g_remove (cdrdao->priv->src_output);
		g_remove (cdrdao->priv->toc);
	}

	if (sectors) {
		/* NOTE: 1 sec = 75 sectors, 1 sector = 2352 bytes */
		*size = cdrdao->priv->sectors_num;
	}
	else
		*size = cdrdao->priv->sectors_num * 2352;
	
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_get_speed_image (BonfireImager *imager, int *speed)
{
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (imager);

	if (cdrdao->priv->cur_speed == -1)
		return BONFIRE_BURN_NOT_SUPPORTED;

	*speed = cdrdao->priv->cur_speed;

	return BONFIRE_BURN_OK;
}

static void
bonfire_cdrdao_set_argv_device (BonfireCdrdao *cdrdao,
				GPtrArray *argv)
{
	/* it seems that cdrdao (1.2.0) doesn't like the cdrecord_id on my computer */
	g_ptr_array_add (argv, g_strdup ("--device"));
	if (cdrdao->priv->drive->cdrecord_id)
		g_ptr_array_add (argv, g_strdup (cdrdao->priv->drive->cdrecord_id));
	else if (cdrdao->priv->drive->device)
		g_ptr_array_add (argv, g_strdup (cdrdao->priv->drive->device));
}

static void
bonfire_cdrdao_set_argv_common_rec (BonfireCdrdao *cdrdao,
				    GPtrArray *argv)
{
	char *speed_str;

	g_ptr_array_add (argv, g_strdup ("--speed"));
	speed_str = g_strdup_printf ("%d", cdrdao->priv->speed);
	g_ptr_array_add (argv, speed_str);

	if (cdrdao->priv->overburn)
		g_ptr_array_add (argv, g_strdup ("--overburn"));
	if (cdrdao->priv->multi)
		g_ptr_array_add (argv, g_strdup ("--multi"));
}

static void
bonfire_cdrdao_set_argv_common (BonfireCdrdao *cdrdao,
				    GPtrArray *argv)
{
	if (cdrdao->priv->dummy)
		g_ptr_array_add (argv, g_strdup ("--simulate"));

	/* cdrdao manual says it is a similar option to gracetime */
	if (cdrdao->priv->nograce)
		g_ptr_array_add (argv, g_strdup ("-n"));

	g_ptr_array_add (argv, g_strdup ("-v"));
	g_ptr_array_add (argv, g_strdup ("2"));
}

static BonfireBurnResult
bonfire_cdrdao_set_argv_record (BonfireCdrdao *cdrdao,
				GPtrArray *argv)
{
	BonfireTrackSource *track;

	if (!cdrdao->priv->drive)
		return BONFIRE_BURN_NOT_READY;

	track = cdrdao->priv->track;
	if (!track)
		return BONFIRE_BURN_NOT_READY;

        if (track->type == BONFIRE_TRACK_SOURCE_DISC) {
		NautilusBurnDrive *source;

		g_ptr_array_add (argv, g_strdup ("copy"));
		bonfire_cdrdao_set_argv_device (cdrdao, argv);
		bonfire_cdrdao_set_argv_common (cdrdao, argv);
		bonfire_cdrdao_set_argv_common_rec (cdrdao, argv);

		source = cdrdao->priv->track->contents.drive.disc;
		if (!nautilus_burn_drive_equal (source, cdrdao->priv->drive))
			g_ptr_array_add (argv, g_strdup ("--on-the-fly"));

		g_ptr_array_add (argv, g_strdup ("--source-device"));
		if (source->cdrecord_id)
			g_ptr_array_add (argv, g_strdup (source->cdrecord_id));
		else
			g_ptr_array_add (argv, g_strdup (source->device));
	}
	else if (track->type == BONFIRE_TRACK_SOURCE_CUE) {
		g_ptr_array_add (argv, g_strdup ("write"));

		bonfire_cdrdao_set_argv_device (cdrdao, argv);
		bonfire_cdrdao_set_argv_common (cdrdao, argv);
		bonfire_cdrdao_set_argv_common_rec (cdrdao, argv);

		g_ptr_array_add (argv, g_strdup (track->contents.cue.toc));
	}
	else
		return BONFIRE_BURN_NOT_SUPPORTED;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_set_argv_blank (BonfireCdrdao *cdrdao,
			       GPtrArray *argv)
{
	if (!cdrdao->priv->drive)
		return BONFIRE_BURN_NOT_READY;

	g_ptr_array_add (argv, g_strdup ("blank"));

	bonfire_cdrdao_set_argv_device (cdrdao, argv);
	bonfire_cdrdao_set_argv_common (cdrdao, argv);

	if (!cdrdao->priv->blank_fast) {
		g_ptr_array_add (argv, g_strdup ("--blank-mode"));
		g_ptr_array_add (argv, g_strdup ("full"));
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_set_argv_image (BonfireCdrdao *cdrdao,
			       GPtrArray *argv,
			       GError **error)
{
	BonfireBurnResult result;
	NautilusBurnDrive *source;

	if (!cdrdao->priv->track)
		return BONFIRE_BURN_NOT_READY;

	g_ptr_array_add (argv, g_strdup ("read-cd"));
	g_ptr_array_add (argv, g_strdup ("--device"));

	source = cdrdao->priv->track->contents.drive.disc;
	if (source->cdrecord_id)
		g_ptr_array_add (argv, g_strdup (source->cdrecord_id));
	else if (source->device)
		g_ptr_array_add (argv, g_strdup (source->device));

	g_ptr_array_add (argv, g_strdup ("--read-raw"));

	result = bonfire_burn_common_check_output (&cdrdao->priv->src_output,
						   cdrdao->priv->overwrite,
						   &cdrdao->priv->toc,
						   error);
	if (result != BONFIRE_BURN_OK)
		return result;

	g_ptr_array_add (argv, g_strdup ("--datafile"));
	g_ptr_array_add (argv, g_strdup (cdrdao->priv->src_output));

	g_ptr_array_add (argv, g_strdup ("-v"));
	g_ptr_array_add (argv, g_strdup ("2"));

	g_ptr_array_add (argv, g_strdup (cdrdao->priv->toc));

	if (cdrdao->priv->action == BONFIRE_CDRDAO_ACTION_GET_SIZE)
		bonfire_job_action_changed (BONFIRE_JOB (cdrdao),
					    BONFIRE_BURN_ACTION_GETTING_SIZE,
					    FALSE);

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_set_argv (BonfireProcess *process,
			 GPtrArray *argv,
			 gboolean has_master,
			 GError **error)
{
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (process);

	if (has_master)
		return BONFIRE_BURN_NOT_SUPPORTED;

	cdrdao->priv->start = 0;
	bonfire_job_set_run_slave (BONFIRE_JOB (cdrdao), FALSE);

	/* sets the first argv */
	g_ptr_array_add (argv, g_strdup ("cdrdao"));

	if (cdrdao->priv->action == BONFIRE_CDRDAO_ACTION_RECORD)
		return bonfire_cdrdao_set_argv_record (cdrdao, argv);
	else if (cdrdao->priv->action == BONFIRE_CDRDAO_ACTION_BLANK)
		return bonfire_cdrdao_set_argv_blank (cdrdao, argv);
	else if (cdrdao->priv->action == BONFIRE_CDRDAO_ACTION_IMAGE)
		return bonfire_cdrdao_set_argv_image (cdrdao, argv, error);
	else if (cdrdao->priv->action == BONFIRE_CDRDAO_ACTION_GET_SIZE)
		return bonfire_cdrdao_set_argv_image (cdrdao, argv, error);


	return BONFIRE_BURN_NOT_READY;
}

static BonfireBurnResult
bonfire_cdrdao_post (BonfireProcess *process,
		     BonfireBurnResult retval)
{
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (process);

	if (cdrdao->priv->timer) {
		g_timer_destroy (cdrdao->priv->timer);
		cdrdao->priv->timer = NULL;
	}

	if (cdrdao->priv->rates) {
		g_slist_free (cdrdao->priv->rates);
		cdrdao->priv->rates = NULL;
	}

	if (retval == BONFIRE_BURN_CANCEL) {
		if (cdrdao->priv->src_output)
			g_remove (cdrdao->priv->src_output);
		if (cdrdao->priv->toc)
			g_remove (cdrdao->priv->toc);
	}

	cdrdao->priv->cur_speed = -1;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_set_drive (BonfireRecorder *recorder,
			  NautilusBurnDrive *drive,
			  GError **error)
{
	NautilusBurnMediaType media;
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (recorder);

	media = nautilus_burn_drive_get_media_type (drive);
	if (media > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
		return BONFIRE_BURN_NOT_SUPPORTED;

	if (cdrdao->priv->drive) {
		nautilus_burn_drive_unref (cdrdao->priv->drive);
		cdrdao->priv->drive = NULL;
	}

	cdrdao->priv->drive = drive;
	nautilus_burn_drive_ref (drive);

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_set_flags (BonfireRecorder *recorder,
			  BonfireRecorderFlag flags,
			  GError **error)
{
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (recorder);

	cdrdao->priv->multi = (flags & BONFIRE_RECORDER_FLAG_MULTI);
	cdrdao->priv->dummy = (flags & BONFIRE_RECORDER_FLAG_DUMMY);
	cdrdao->priv->dao = (flags & BONFIRE_RECORDER_FLAG_DAO);
	cdrdao->priv->nograce = (flags & BONFIRE_RECORDER_FLAG_NOGRACE);
	cdrdao->priv->burnproof = (flags & BONFIRE_RECORDER_FLAG_BURNPROOF);
	cdrdao->priv->overburn = (flags & BONFIRE_RECORDER_FLAG_OVERBURN);
	cdrdao->priv->blank_fast = (flags & BONFIRE_RECORDER_FLAG_FAST_BLANK);

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_set_speed (BonfireJob *job,
			  gint64 speed)
{
	BonfireCdrdao *cdrdao;

	if (bonfire_job_is_running (job))
		return BONFIRE_BURN_RUNNING;
	
	cdrdao = BONFIRE_CDRDAO (job);
	cdrdao->priv->speed = speed / CDR_SPEED;

	return BONFIRE_BURN_OK;

}

static BonfireBurnResult
bonfire_cdrdao_record (BonfireRecorder *recorder,
		       GError **error)
{
	BonfireCdrdao *cdrdao;
	BonfireBurnResult result;

	cdrdao = BONFIRE_CDRDAO (recorder);

	cdrdao->priv->action = BONFIRE_CDRDAO_ACTION_RECORD;
	result = bonfire_job_run (BONFIRE_JOB (cdrdao), error);
	cdrdao->priv->action = BONFIRE_CDRDAO_ACTION_NONE;

	return result;
}

static BonfireBurnResult
bonfire_cdrdao_blank (BonfireRecorder *recorder,
		      GError **error)
{
	BonfireCdrdao *cdrdao;
	BonfireBurnResult result;

	cdrdao = BONFIRE_CDRDAO (recorder);

	if (!nautilus_burn_drive_can_write (cdrdao->priv->drive)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the drive cannot rewrite CDs or DVDs"));
		return BONFIRE_BURN_ERR;
	}

	cdrdao->priv->action = BONFIRE_CDRDAO_ACTION_BLANK;
	result = bonfire_job_run (BONFIRE_JOB (cdrdao), error);
	cdrdao->priv->action = BONFIRE_CDRDAO_ACTION_NONE;

	return result;
}

static BonfireBurnResult
bonfire_cdrdao_get_record_status (BonfireRecorder *recorder,
				  int *speed,
				  int *fifo,
				  gint64 *bytes_written)
{
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (recorder);

	if (speed)
		*speed = cdrdao->priv->cur_speed;

	if (fifo)
		*fifo = cdrdao->priv->fifo;

	if (bytes_written)
		*bytes_written = cdrdao->priv->b_written;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_cdrdao_get_action_string (BonfireJob *job, 
				  BonfireBurnAction action,
				  char **string)
{
	BonfireCdrdao *cdrdao;

	cdrdao = BONFIRE_CDRDAO (job);

	if (action == BONFIRE_BURN_ACTION_ANALYSING)
		*string = g_strdup_printf (_("Analysing track %02i"), cdrdao->priv->track_num);
	else {
		const char *tmp;

		tmp = bonfire_burn_action_to_string (action);
		*string = g_strdup (tmp);
	}

	return BONFIRE_BURN_OK;
}
