/***************************************************************************
 *            mkisofs-case.c
 *
 *  mar jan 24 16:41:02 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <libgnomevfs/gnome-vfs.h>

#include <nautilus-burn-drive.h>

#include "bonfire-marshal.h"
#include "burn-basics.h"
#include "burn-common.h"
#include "burn-mkisofs-base.h"
#include "burn-process.h"
#include "burn-imager.h"
#include "utils.h"

static void bonfire_mkisofs_base_class_init (BonfireMkisofsBaseClass *klass);
static void bonfire_mkisofs_base_init (BonfireMkisofsBase *sp);
static void bonfire_mkisofs_base_finalize (GObject *object);
static void bonfire_mkisofs_base_iface_init_image (BonfireImagerIFace *iface);

static BonfireBurnResult
bonfire_mkisofs_base_get_action_string (BonfireJob *job,
					BonfireBurnAction action,
					char **string);
static BonfireBurnResult
bonfire_mkisofs_base_start (BonfireJob *job,
			    int fd_in,
			    int *fd_out,
			    GError **error);
static BonfireBurnResult
bonfire_mkisofs_base_stop (BonfireJob *job,
			   BonfireBurnResult retval);
static BonfireBurnResult
bonfire_mkisofs_base_set_source (BonfireJob *job,
				 const BonfireTrackSource *source,
				 GError **error);
static BonfireBurnResult
bonfire_mkisofs_base_set_output_type (BonfireImager *imager,
				      BonfireTrackSourceType type,
				      GError **error);
static BonfireBurnResult
bonfire_mkisofs_base_set_output (BonfireImager *imager,
				 const char *output,
				 gboolean overwrite,
				 gboolean clean,
				 GError **error);
static BonfireBurnResult
bonfire_mkisofs_base_get_track (BonfireImager *imager,
				BonfireTrackSource **source,
				GError **error);
static BonfireBurnResult
bonfire_mkisofs_base_get_size (BonfireImager *imager,
			       gint64 *size,
			       gboolean sectors,
			       GError **error);

struct _BonfireMkisofsData {
	int grafts_fd;
	int excluded_fd;

	GHashTable *nonlocals;
	GHashTable *grafts;
};
typedef struct _BonfireMkisofsData BonfireMkisofsData;

struct _BonfireDownloadNonlocalData {
	BonfireMkisofsData *data;
	GSList *list;
};
typedef struct _BonfireDownloadNonlocalData BonfireDownloadNonlocalData;

struct _BonfireWriteGraftData {
	BonfireMkisofsData *data;
	GError **error;
};
typedef struct _BonfireWriteGraftData BonfireWriteGraftData;

struct BonfireMkisofsBasePrivate {
	BonfireMkisofsData *data;
	char *tmpdir;

	BonfireTrackSource *source;

	char *emptydir;
	char *grafts_path;
	char *excluded_path;
	GSList *nonlocals;

	const char *current_download;

	GnomeVFSAsyncHandle *xfer_handle;
	gint64 current_copy_size;
	gint64 current_bytes_copied;
	GTimer *timer;
	gint report_id;
	GSList *rates;

	int overwrite:1;
	int clean:1;
};

static GObjectClass *parent_class = NULL;

static char *NOT_DOWNLOADED = "not downloaded";

GType
bonfire_mkisofs_base_get_type()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireMkisofsBaseClass),
			NULL,
			NULL,
			(GClassInitFunc) bonfire_mkisofs_base_class_init,
			NULL,
			NULL,
			sizeof (BonfireMkisofsBase),
			0,
			(GInstanceInitFunc) bonfire_mkisofs_base_init,
		};

		static const GInterfaceInfo imager_info = {
			(GInterfaceInitFunc) bonfire_mkisofs_base_iface_init_image,
			NULL,
			NULL
		};

		type = g_type_register_static (BONFIRE_TYPE_JOB, 
					       "BonfireMkisofsBase",
					       &our_info,
					       0);

		g_type_add_interface_static (type,
					     BONFIRE_TYPE_IMAGER,
					     &imager_info);
	}

	return type;
}

static void
bonfire_mkisofs_base_class_init(BonfireMkisofsBaseClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	BonfireJobClass *job_class = BONFIRE_JOB_CLASS (klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_mkisofs_base_finalize;

	/* this is only used when we are copying files 
	 * otherwise we pass the call on to process class */
	job_class->get_action_string = bonfire_mkisofs_base_get_action_string;
	job_class->start = bonfire_mkisofs_base_start;
	job_class->stop = bonfire_mkisofs_base_stop;
	job_class->set_source = bonfire_mkisofs_base_set_source;
}

static void
bonfire_mkisofs_base_iface_init_image (BonfireImagerIFace *iface)
{
	iface->get_size = bonfire_mkisofs_base_get_size;
	iface->get_track = bonfire_mkisofs_base_get_track;
	iface->set_output = bonfire_mkisofs_base_set_output;
	iface->set_output_type = bonfire_mkisofs_base_set_output_type;
}

static void
bonfire_mkisofs_base_init (BonfireMkisofsBase *obj)
{
	obj->priv = g_new0 (BonfireMkisofsBasePrivate, 1);
}

static void
bonfire_mkisofs_base_clean_track (BonfireMkisofsBase *base)
{
	if (base->priv->clean) {
		if (base->priv->grafts_path)
			g_remove (base->priv->grafts_path);

		if (base->priv->excluded_path) 
			g_remove (base->priv->excluded_path);

		if (base->priv->emptydir)
			g_remove (base->priv->emptydir);

		if (base->priv->nonlocals)
			g_slist_foreach (base->priv->nonlocals,
					 (GFunc) bonfire_burn_common_rm,
					 NULL);
	}

	if (base->priv->nonlocals) {
		g_slist_foreach (base->priv->nonlocals, (GFunc) g_free, NULL);
		g_slist_free (base->priv->nonlocals);
		base->priv->nonlocals = NULL;
	}

	if (base->priv->emptydir) {
		g_free (base->priv->emptydir);
		base->priv->emptydir = NULL;
	}

	if (base->priv->grafts_path) {
		g_free (base->priv->grafts_path);
		base->priv->grafts_path = NULL;
	}

	if (base->priv->excluded_path) {
		g_free (base->priv->excluded_path);
		base->priv->excluded_path = NULL;
	}
}

static void
bonfire_mkisofs_base_clean_tmpdir (BonfireMkisofsBase *base)
{
	bonfire_mkisofs_base_clean_track (base);

	if (base->priv->clean) {
		/* since we remove only our files if a file created
		 * by user remains directory won't be removed */
		if (base->priv->tmpdir)
			g_remove (base->priv->tmpdir);
	}

	if (base->priv->tmpdir) {
		g_free (base->priv->tmpdir);
		base->priv->tmpdir = NULL;
	}
}

static void
bonfire_mkisofs_base_finalize (GObject *object)
{
	BonfireMkisofsBase *cobj;

	cobj = BONFIRE_MKISOFS_BASE (object);

	bonfire_mkisofs_base_clean_tmpdir (cobj);

	if (cobj->priv->source) {
		bonfire_track_source_free (cobj->priv->source);
		cobj->priv->source = NULL;
	}

	if (cobj->priv->timer) {
		g_timer_destroy (cobj->priv->timer);
		cobj->priv->timer = NULL;
	}

	if (cobj->priv->report_id) {
		g_source_remove (cobj->priv->report_id);
		cobj->priv->report_id = 0;
	}

	if (cobj->priv->rates) {
		g_slist_free (cobj->priv->rates);
		cobj->priv->rates = NULL;
	}

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static BonfireBurnResult
_write_line (int fd, const char *filepath, GError **error)
{
	int len;
	int w_len;

	if (lseek (fd, 0, SEEK_CUR)
	&&  write (fd, "\n", 1) != 1) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     strerror (errno));
		return BONFIRE_BURN_ERR;
	}

	len = strlen (filepath);
	w_len = write (fd, filepath, len);

	if (w_len != len) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     strerror (errno));
		return BONFIRE_BURN_ERR;
	}

	return BONFIRE_BURN_OK;
}

static char *
bonfire_mkisofs_base_translate_nonlocal_excluded (BonfireMkisofsData *data,
						  const char *excluded)
{
	char *localpath;
	char *parent;
	char *tmp;

	if (!data->nonlocals)
		return NULL;

	parent = g_path_get_dirname (excluded);
	while (parent [1] != '\0') {
		localpath = g_hash_table_lookup (data->nonlocals, parent);
		if (localpath && localpath != NOT_DOWNLOADED) {
			char *newpath;

			newpath = g_strconcat (localpath,
					       excluded + strlen (parent),
					       NULL);
			g_free (parent);
			return newpath;
		}

		tmp = parent;
		parent = g_path_get_dirname (tmp);
		g_free (tmp);
	}

	g_free (parent);
	return NULL;
}

static BonfireBurnResult
bonfire_mkisofs_base_write_excluded (BonfireMkisofsData *data,
				     const char *uri,
				     GError **error)
{
	char *localpath;
	char *escaped_uri;
	GnomeVFSResult res;
	GnomeVFSFileInfo *info;
	BonfireBurnResult result = BONFIRE_BURN_OK;

	info = gnome_vfs_file_info_new ();
	escaped_uri = gnome_vfs_escape_host_and_path_string (uri);
	res = gnome_vfs_get_file_info (escaped_uri,
				       info,
				       GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS);

	if (res != GNOME_VFS_OK) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     gnome_vfs_result_to_string (res));
		result = BONFIRE_BURN_ERR;
		goto out;
	}

	result = BONFIRE_BURN_OK;
	if (GNOME_VFS_FILE_INFO_LOCAL (info)) {
		char *escaped_path;

		escaped_path = gnome_vfs_get_local_path_from_uri (escaped_uri);
		localpath = gnome_vfs_unescape_string_for_display (escaped_path);
		g_free (escaped_path);
	}
	else  {
		/* we translate the non local uri into a local path */
		localpath = bonfire_mkisofs_base_translate_nonlocal_excluded (data,
									      uri); 
	}

	/* we just ignore if localpath is NULL:
	 * - it could be a non local whose graft point couldn't be downloaded */
	if (localpath)
		result = _write_line (data->excluded_fd, localpath, error);
	
out:

	g_free (escaped_uri);
	gnome_vfs_file_info_clear (info);
	gnome_vfs_file_info_unref (info);
	return result;
}

static char *
_escape_path (const char *str)
{
	char *escaped, *d;
	const char *s;
	int len;

	s = str;
	len = 1;
	while (*s != 0) {
		if (*s == '\\' || *s == '=') {
			len++;
		}

		len++;
		s++;
	}

	escaped = g_malloc (len);

	s = str;
	d = escaped;
	while (*s != 0) {
		if (*s == '\\' || *s == '=') {
			*d++ = '\\';
		}

		*d++ = *s++;
	}
	*d = 0;

	return escaped;
}

static char *
_build_graft_point (const char *escaped_uri, const char *discpath) {
	char *escaped_discpath;
	char *escaped_path;
	char *graft_point;
	char *path;

	/* make up the graft point */
	if (*escaped_uri != '/') {
		escaped_path = gnome_vfs_get_local_path_from_uri (escaped_uri);
		path = gnome_vfs_unescape_string_for_display (escaped_path);
		g_free (escaped_path);
	}
	else
		path = g_strdup (escaped_uri);

	if (discpath) {
		char *escaped_path;

		escaped_path = _escape_path (path);
		g_free (path);

		escaped_discpath = _escape_path (discpath);
		graft_point = g_strconcat (escaped_discpath,
					   "=",
					   escaped_path,
					   NULL);
		g_free (escaped_path);
		g_free (escaped_discpath);
	}
	else
		graft_point = path;

	return graft_point;
}

/* This one is for error reporting */
static int
bonfire_mkisofs_base_xfer_async_cb (GnomeVFSAsyncHandle *handle,
				    GnomeVFSXferProgressInfo *info,
				    BonfireMkisofsBase *base)
{
	if (!base->priv->xfer_handle)
		return FALSE;

	if (info->phase == GNOME_VFS_XFER_PHASE_COMPLETED) {
		bonfire_job_finished (BONFIRE_JOB (base));
		return FALSE;
	}
	else if (info->status != GNOME_VFS_XFER_PROGRESS_STATUS_OK) {
		bonfire_job_error (BONFIRE_JOB (base),
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_GENERAL,
						gnome_vfs_result_to_string (info->vfs_status)));
		return GNOME_VFS_XFER_ERROR_ACTION_ABORT;
	}

	return TRUE;
}

static gboolean
bonfire_mkisofs_base_report_progress (BonfireMkisofsBase *base)
{
	glong remaining = -1;
	gdouble fraction;
	gdouble elapsed;

	elapsed = g_timer_elapsed (base->priv->timer, NULL);
	if (elapsed <= 2.0)
		return TRUE;

	fraction = (gdouble) base->priv->current_bytes_copied /
		   (gdouble) base->priv->current_copy_size;

	if (fraction <= 1.0 && fraction >= 0.0) {
		gdouble rate;

		rate = base->priv->current_bytes_copied / elapsed;
		rate = bonfire_burn_common_get_average_rate (&base->priv->rates, rate);

		remaining = (gdouble) base->priv->current_copy_size / rate - elapsed;
	}
	else
		fraction = -1.0;

	bonfire_job_progress_changed (BONFIRE_JOB (base),
				      fraction,
				      remaining);
	return TRUE;
}

/* This one is for progress reporting */
static gint
bonfire_mkisofs_base_xfer_cb (GnomeVFSXferProgressInfo *info,
			      BonfireMkisofsBase *base)
{
	if (!base->priv->xfer_handle)
		return FALSE;

	if (!base->priv->timer) {
		base->priv->timer = g_timer_new ();
		base->priv->report_id = g_timeout_add (200,
						       (GSourceFunc) bonfire_mkisofs_base_report_progress,
						       base);
		return TRUE;
	}

	if (info->file_size)
		base->priv->current_copy_size = info->bytes_total;

	if (info->bytes_copied)
		base->priv->current_bytes_copied = info->total_bytes_copied;

	return TRUE;
}

static BonfireBurnResult
bonfire_mkisofs_base_copy_non_local_file (BonfireMkisofsBase *base,
					  const char *uri_src,
					  char **localpath,
					  GError **error)
{
	GList *src_list, *dest_list;
	BonfireBurnResult result;
	GnomeVFSURI *tmpuri;
	GnomeVFSURI *vfsuri;
	GnomeVFSResult res;
	char *localpath_tmp;
	char *escaped_uri;
	char *uri;

	/* generate a unique name */
	localpath_tmp = g_strconcat (base->priv->tmpdir, "/XXXXXX", NULL);
	if (mkstemp (localpath_tmp) == -1) {
		g_free (localpath_tmp);
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_TMP_DIR,
			     _("a temporary file can't be created: %s"),
			     strerror(errno));
		return BONFIRE_BURN_ERR;
	}
	g_remove (localpath_tmp);
	uri = g_strconcat ("file://", localpath_tmp, NULL);

	/* start the thing */
	escaped_uri = gnome_vfs_escape_host_and_path_string (uri_src);
	vfsuri = gnome_vfs_uri_new (escaped_uri);
	g_free (escaped_uri);
	src_list = g_list_append (NULL, vfsuri);

	escaped_uri = gnome_vfs_escape_host_and_path_string (uri);
	g_free (uri);
	tmpuri = gnome_vfs_uri_new (escaped_uri);
	g_free (escaped_uri);
	dest_list = g_list_append (NULL, tmpuri);
	
	res = gnome_vfs_async_xfer (&base->priv->xfer_handle,
				    src_list,
				    dest_list,
				    GNOME_VFS_XFER_DEFAULT |
				    GNOME_VFS_XFER_USE_UNIQUE_NAMES |
				    GNOME_VFS_XFER_RECURSIVE,
				    GNOME_VFS_XFER_ERROR_MODE_ABORT,
				    GNOME_VFS_XFER_OVERWRITE_MODE_ABORT,
				    GNOME_VFS_PRIORITY_DEFAULT,
				    (GnomeVFSAsyncXferProgressCallback) bonfire_mkisofs_base_xfer_async_cb,
				    base,
				    (GnomeVFSXferProgressCallback) bonfire_mkisofs_base_xfer_cb,
				    base);

	g_list_free (src_list);
	g_list_free (dest_list);
	gnome_vfs_uri_unref (vfsuri);
	gnome_vfs_uri_unref (tmpuri);

	if (res != GNOME_VFS_OK) {
		g_free (localpath_tmp);
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     gnome_vfs_result_to_string (res));
		base->priv->current_download = NULL;
		return BONFIRE_BURN_ERR;
	}

	/* start the job */
	base->priv->current_download = uri_src;
	bonfire_job_action_changed (BONFIRE_JOB (base),
				    BONFIRE_BURN_ACTION_FILE_COPY,
				    TRUE);
	bonfire_job_progress_changed (BONFIRE_JOB (base), -1.0, -1);

	result = bonfire_job_run (BONFIRE_JOB (base), error);
	base->priv->current_download = NULL;
	base->priv->current_copy_size = 0;

	/* finished: even if it fails we return the path so that we can clean it */
	*localpath = localpath_tmp;
	if (result != BONFIRE_BURN_OK)
		return result;

	bonfire_job_progress_changed (BONFIRE_JOB (base), 1.0, -1);
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_base_get_size (BonfireImager *imager,
			       gint64 *size,
			       gboolean sectors,
			       GError **error)
{
	BonfireMkisofsBase *base;

	base = BONFIRE_MKISOFS_BASE (imager);

	if (!bonfire_job_is_running (BONFIRE_JOB (base)))
		return BONFIRE_BURN_NOT_RUNNING;

	if (!base->priv->current_copy_size)
		return BONFIRE_BURN_NOT_READY;

	if (sectors)
		*size = base->priv->current_copy_size / 2048;
	else
		*size = base->priv->current_copy_size;

	return BONFIRE_BURN_OK;
}

static void
_foreach_non_local_cb (const char *uri,
		       const char *local_path,
		       BonfireDownloadNonlocalData *data)
{
	char *localpath;
	char *parent;
	char *tmp;

	parent = g_path_get_dirname (uri);
	while (parent [1] != '\0') {
		localpath = g_hash_table_lookup (data->data->nonlocals, parent);
		if (localpath && localpath != NOT_DOWNLOADED) {
			g_free (parent);
			return;
		}

		tmp = parent;
		parent = g_path_get_dirname (tmp);
		g_free (tmp);
	}
	g_free (parent);
	data->list = g_slist_prepend (data->list, (char *) uri);
}

static BonfireBurnResult
bonfire_mkisofs_base_download_nonlocal (BonfireMkisofsBase *base,
					BonfireMkisofsData *data,
					GError **error)
{
	GSList *list;
	BonfireBurnResult result;
	BonfireDownloadNonlocalData callback_data;

	if (!data->nonlocals)
		return BONFIRE_BURN_OK;

	/* first make a list of all files that need downloading
	 * to avoid downloading a parent and its children */
	callback_data.list = NULL;
	callback_data.data = data;
	g_hash_table_foreach (data->nonlocals,
			      (GHFunc) _foreach_non_local_cb,
			      &callback_data);

	for (list = callback_data.list; list; list = list->next) {
		char *uri;
		char *localpath;

		uri = list->data;
		localpath = NULL;
		result = bonfire_mkisofs_base_copy_non_local_file (base,
								   uri,
								   &localpath,
								   error);

		/* we must append them immediatly so that we can remove them */
		base->priv->nonlocals = g_slist_prepend (base->priv->nonlocals,
							 localpath);

		if (result != BONFIRE_BURN_OK)
			return result;

		g_hash_table_insert (data->nonlocals,
				     uri,
				     localpath);
        }
	g_slist_free (callback_data.list);
	callback_data.list = NULL;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_base_write_graft (BonfireMkisofsData *data,
				  const char *uri,
				  const char *disc_path,
				  GError **error)
{
	char *graft_point;
	char *localpath = NULL;
	BonfireBurnResult result;

	/* see if the uri is in non local if so,
	 * convert it to its new local path */
	if (data->nonlocals)
		localpath = g_hash_table_lookup (data->nonlocals, uri);

	if (localpath) {
		if (localpath == NOT_DOWNLOADED) {
			char *parent;

			/* we are looking for a parent that might have been downloaded */
			parent = g_path_get_dirname (uri);
			while (!localpath && localpath == NOT_DOWNLOADED) {
				char *tmp;

				localpath = g_hash_table_lookup (data->nonlocals,
								 parent);
				tmp = parent;
				parent = g_path_get_dirname (parent);
				g_free (tmp);
			}

			if (!localpath || localpath == NOT_DOWNLOADED) {
				/* that should not happen but who knows */
				g_free (parent);
				return BONFIRE_BURN_ERR;
			}

			/* adapt the local path */
			localpath = g_strconcat (localpath, uri + strlen (parent), NULL);
			g_free (parent);
		}
		else
			localpath = g_strdup (localpath);
	}
	else
		localpath = gnome_vfs_get_local_path_from_uri (uri);

	/* build up graft and write it */
	graft_point = _build_graft_point (localpath, disc_path);
	g_free (localpath);

	if (!graft_point) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("null graft point"));
		return BONFIRE_BURN_ERR;
	}

	result = _write_line (data->grafts_fd, graft_point, error);
	g_free (graft_point);
	if(result != BONFIRE_BURN_OK) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     strerror (errno));
		return result;
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_base_write_excluded_valid_paths (BonfireMkisofsData *data,
						 const char *uri,
						 GError **error)
{
	int size;
	char *tmp;
	char *path;
	char *parent;
	GSList *iter;
	char *excluded;
	GSList *grafts;
	gboolean found;
	BonfireGraftPt *graft;
	BonfireBurnResult result;

	/* we go straight to its parent so that is the excluded uri
	 * is a graft point it won't be taken into account (it has 
	 * already with all other graft points) */
	parent = g_path_get_dirname (uri);
	while (parent [1] != '\0') {
		grafts = g_hash_table_lookup (data->grafts, parent);

		for (; grafts; grafts = grafts->next) {
			graft = grafts->data;

			/* see if the uri or one of its parent is excluded */
			found = FALSE;
			for (iter = graft->excluded; iter; iter = iter->next) {
				excluded = iter->data;
				size = strlen (excluded);

				if (!strncmp (excluded, uri, size)
				&& (*(uri + size) == '\0' ||  *(uri + size) == '/')) {
					found = TRUE;
					break;
				}
			}

			if (found)
				continue;

			/* there is at least one path which is not excluded */
			path = g_strconcat (graft->path,
					    uri + strlen (graft->uri),
					    NULL);

			result = bonfire_mkisofs_base_write_graft (data,
								   uri,
								   path,
								   error);
			g_free (path);

			if (result != BONFIRE_BURN_OK) {
				g_free (parent);
				return result;
			}
		}

		tmp = parent;
		parent = g_path_get_dirname (parent);
		g_free (tmp);
	}
	g_free (parent);

	return BONFIRE_BURN_OK;
}

static gboolean
_foreach_non_local_write_graft (const char *uri,
				GSList *grafts,
				BonfireWriteGraftData *data)
{
	BonfireBurnResult result;
	BonfireGraftPt *graft;

	for (; grafts; grafts = grafts->next) {
		graft = grafts->data;
		result = bonfire_mkisofs_base_write_graft (data->data,
							   graft->uri,
							   graft->path,
							   data->error);
		if (result != BONFIRE_BURN_OK)
			return TRUE;
	}

	return FALSE;
}

static gboolean
bonfire_mkisofs_base_write_grafts (BonfireMkisofsData *data,
				   GError **error)
{
	BonfireWriteGraftData callback_data;
	gpointer result;

	callback_data.error = error;
	callback_data.data = data;

	result = g_hash_table_find (data->grafts,
				    (GHRFunc) _foreach_non_local_write_graft,
				    &callback_data);

	if (result)
		return BONFIRE_BURN_ERR;

	return BONFIRE_BURN_OK;
}


static BonfireBurnResult
bonfire_mkisofs_base_empty_directory (BonfireMkisofsBase *base,
				      BonfireMkisofsData *data,
				      const char *disc_path,
				      GError **error)
{
	BonfireBurnResult result;
	char *graft_point;

	/* This a special case for uri = NULL; that
	 * is treated as if it were a directory */
        if (!base->priv->emptydir) {
		char *dirpath;

		dirpath = g_strconcat (base->priv->tmpdir,
				       "/emptyXXXXXX",
				       NULL);
		dirpath = mkdtemp (dirpath);
		if (dirpath == NULL) {
			g_free (dirpath);
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_TMP_DIR,
				     _("a temporary directory couldn't be created : %s"),
				     strerror (errno));
			return BONFIRE_BURN_ERR;
		}

		base->priv->emptydir = dirpath;
	}

	graft_point = _build_graft_point (base->priv->emptydir, disc_path);
	result = _write_line (data->grafts_fd, graft_point, error);
	g_free (graft_point);

	return result;
}

static BonfireBurnResult
bonfire_mkisofs_base_add_graft (BonfireMkisofsBase *base,
				BonfireGraftPt *graft,
				GError **error)
{
	GSList *list;
	char *escaped_uri;
	GnomeVFSResult res;
	GnomeVFSFileInfo *info;
	BonfireMkisofsData *data;

	data = base->priv->data;
	info = gnome_vfs_file_info_new ();

	escaped_uri = gnome_vfs_escape_host_and_path_string (graft->uri);
	res = gnome_vfs_get_file_info (escaped_uri,
				       info,
				       GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS);
	g_free (escaped_uri);

	if (res != GNOME_VFS_OK) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     gnome_vfs_result_to_string (res));
		goto error;
	}

	/* check the file is local */
	if (GNOME_VFS_FILE_INFO_LOCAL (info)) {
		gboolean unreadable;

		/* check if the file is readable */
		unreadable = FALSE;
		if (info->uid == getuid ()) {
			if (!(info->permissions & GNOME_VFS_PERM_USER_READ))
				unreadable = TRUE;
		}
		else if (info->gid == getgid ()) {
			if (!(info->permissions & GNOME_VFS_PERM_GROUP_READ))
				unreadable = TRUE;
		} 
		else if (!(info->permissions & GNOME_VFS_PERM_OTHER_READ)) {
			unreadable = TRUE;
		}

		/* add the uris and */
		if (unreadable) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_INVALID_FILE,
				     _("the file can't be read"));
			goto error;
		}
	}
	else {
		/* These files will be handled later to avoid downloading
		 * the same file for all the paths it appears at */
		if (!data->nonlocals)
			data->nonlocals = g_hash_table_new (g_str_hash, g_str_equal);

		g_hash_table_insert (data->nonlocals,
				     graft->uri,
				     NOT_DOWNLOADED);
	}

	/* make up the graft point */
	list = g_hash_table_lookup (data->grafts, graft->uri);
	list = g_slist_prepend (list, graft);
	g_hash_table_insert (data->grafts, graft->uri, list);

	gnome_vfs_file_info_clear (info);
	gnome_vfs_file_info_unref (info);

	return BONFIRE_BURN_OK;

      error: {
	      BonfireBurnResult answer= BONFIRE_BURN_ERR;

	      gnome_vfs_file_info_clear (info);
	      gnome_vfs_file_info_unref (info);

              if (answer != BONFIRE_BURN_OK)
		      return BONFIRE_BURN_ERR;
      }

      g_error_free (*error);
      *error = NULL;
      return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_base_init_data (BonfireMkisofsBase *base,
				GError **error)
{
	BonfireMkisofsData *data;
	const char *tmpdir;
	char *path;

	data = base->priv->data;
	if (!data)
		return BONFIRE_BURN_ERR;

	if (base->priv->tmpdir && g_file_test (base->priv->tmpdir, G_FILE_TEST_EXISTS)) {
		if (base->priv->overwrite) {
			char *name;

			name = g_path_get_basename (base->priv->tmpdir);
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("%s already exists"),
				     name);
			g_free (name);
			return BONFIRE_BURN_ERR;
		}

		if (!g_file_test (base->priv->tmpdir, G_FILE_TEST_IS_DIR)) {
			g_remove (base->priv->tmpdir);

			if (g_file_test (base->priv->tmpdir, G_FILE_TEST_EXISTS)) {
				char *name;

				name = g_path_get_basename (base->priv->tmpdir);
				g_set_error (error,
					     BONFIRE_BURN_ERROR,
					     BONFIRE_BURN_ERROR_GENERAL,
					     _("%s can't be removed and is not a directory (%s)"),
					     name,
					     strerror (errno));
				g_free (name);
				return BONFIRE_BURN_ERR;
			}

			if (g_mkdir_with_parents (base->priv->tmpdir, 700) == -1) {
				char *name;
	
				name = g_path_get_basename (base->priv->tmpdir);
				g_set_error (error,
					     BONFIRE_BURN_ERROR,
					     BONFIRE_BURN_ERROR_GENERAL,
					     _("%s can't be created (%s)"),
					     name,
					     strerror (errno));
				g_free (name);
				return BONFIRE_BURN_ERR;
			}
		}
	}
	else if (base->priv->tmpdir) {
		if (g_mkdir_with_parents (base->priv->tmpdir, 700) == -1) {
			char *name;

			name = g_path_get_basename (base->priv->tmpdir);
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("directory \"%s\" can't be created (%s)"),
				     name,
				     strerror (errno));
			g_free (name);
			return BONFIRE_BURN_ERR;
		}
	}
	else {			
		/* create a working directory in tmp */
		tmpdir = g_get_tmp_dir ();
		path = g_strconcat (tmpdir, "/DiscImageXXXXXX", NULL);
		base->priv->tmpdir = mkdtemp (path);

		if (base->priv->tmpdir == NULL) {
			g_free (path);
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("a temporary directory could not be created (%s)"),
				     strerror (errno));
			return BONFIRE_BURN_ERR;
		}
	}

	/* open a new file list */
	if (base->priv->grafts_path) {
		if (base->priv->clean)
			g_remove (base->priv->grafts_path);

		g_free (base->priv->grafts_path);
	}

	base->priv->grafts_path = g_strconcat (base->priv->tmpdir,
					       "/",
					       "filelistXXXXXX",
					       NULL);
	data->grafts_fd = g_mkstemp (base->priv->grafts_path);
	if (data->grafts_fd < 0) {
		g_free (base->priv->grafts_path);
		base->priv->grafts_path = NULL;

		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_TMP_DIR,
			     _("a temporary file couldn't be created: %s"),
			     strerror (errno));
		return BONFIRE_BURN_ERR;
	}

	/* open a new excluded list */
	if (base->priv->excluded_path) {
		if (base->priv->clean)
			g_remove (base->priv->excluded_path);

		g_free (base->priv->excluded_path);
	}

	base->priv->excluded_path = g_strconcat (base->priv->tmpdir, 
						 "/",
						 "excludeXXXXXX",
						 NULL);
	data->excluded_fd = g_mkstemp (base->priv->excluded_path);
	if (data->excluded_fd < 0) {
		g_free (base->priv->excluded_path);
		base->priv->excluded_path = NULL;

		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_TMP_DIR,
			     _("a temporary file couldn't be created: %s"),
			     strerror (errno));
		return BONFIRE_BURN_ERR;
	}

	data->grafts = g_hash_table_new (g_str_hash, g_str_equal);
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult 
bonfire_mkisofs_base_run (BonfireMkisofsBase *base,
			  GError **error)
{
	char *uri;
	GSList *list;
	GSList *grafts;
	GSList *excluded;
	BonfireMkisofsData data;
	BonfireBurnResult result;

	/* initialize data */
	bzero (&data, sizeof (data));
	base->priv->data = &data;
	result = bonfire_mkisofs_base_init_data (base, error);
	if (result != BONFIRE_BURN_OK)
		goto cleanup;

	/* we analyse the graft points:
	 * first add graft points and excluded. At the same time create a hash 
	 * table in which key = uri and value = graft points, a list of all
	 * the uris that have been excluded and a hash for non local files.
	 * once finished, for each excluded use the hash to see if there are not
	 * other paths at which the excluded uri must appear. If so, create an
	 * explicit graft point. */
	grafts = base->priv->source->contents.data.grafts;
	excluded = NULL;
	for (; grafts; grafts = grafts->next) {
		BonfireGraftPt *graft;

		graft = grafts->data;

		if (!graft->uri) {
			result = bonfire_mkisofs_base_empty_directory (base,
								       &data,
								       graft->path,
								       error);
			if (result != BONFIRE_BURN_OK)
				goto cleanup;

			continue;
		}

		result = bonfire_mkisofs_base_add_graft (base,
							 graft,
							 error);
		if (result != BONFIRE_BURN_OK)
			goto cleanup;

		for (list = graft->excluded; list; list = list->next) {
			char *uri;

			uri = list->data;
			excluded = g_slist_prepend (excluded, uri);
		}
	}

	/* download the non local files */
	result = bonfire_mkisofs_base_download_nonlocal (base, &data, error);
	if (result != BONFIRE_BURN_OK)
		goto cleanup;

	/* write the grafts list */
	result = bonfire_mkisofs_base_write_grafts (&data, error);
	if (result != BONFIRE_BURN_OK)
		goto cleanup;

	/* now add the excluded and the paths where they still exist */
	for (; excluded; excluded = g_slist_remove (excluded, uri)) {
		uri = excluded->data;

		result = bonfire_mkisofs_base_write_excluded (&data,
							      uri,
							      error);
		if (result != BONFIRE_BURN_OK)
			goto cleanup;

		/* One thing is a bit tricky here. If we exclude one file mkisofs considers
		 * it to be excluded for all paths. So if one file appears multiple times
		 * and is excluded just once, it will always be excluded that's why we create
		 * explicit graft points where it is not excluded.*/
		result = bonfire_mkisofs_base_write_excluded_valid_paths (&data,
									  uri,
									  error);
		if (result != BONFIRE_BURN_OK)
			goto cleanup;
	}

	/* write the global excluded files list */
	for (excluded = base->priv->source->contents.data.excluded; excluded; excluded = excluded->next) {
		uri = excluded->data;

		result = bonfire_mkisofs_base_write_excluded (&data,
							      uri,
							      error);
		if (result != BONFIRE_BURN_OK)
			goto cleanup;
	}

     cleanup:

	base->priv->data = NULL;

	/* now we clean data we have the most important :
	 * graft and excluded list, flags and that's what
	 * we're going to use when we'll start the image 
	 * creation */
	if (data.grafts_fd)
		close (data.grafts_fd);
	if (data.excluded_fd)
		close (data.excluded_fd);

	if (data.grafts) {
		g_hash_table_destroy (data.grafts);
		data.grafts = NULL;
	}

	if (data.nonlocals) {
		g_hash_table_destroy (data.nonlocals);
		data.nonlocals = NULL;
	}

	return result;
}

static BonfireBurnResult
bonfire_mkisofs_base_start (BonfireJob *job,
			    int fd_in,
			    int *fd_out,
			    GError **error)
{
	g_return_val_if_fail (BONFIRE_IS_MKISOFS_BASE (job), BONFIRE_BURN_ERR);
	
	if (fd_in != -1 || fd_out)
		return BONFIRE_BURN_NOT_SUPPORTED;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_base_stop (BonfireJob *job,
			   BonfireBurnResult retval)
{
	BonfireMkisofsBase *base;

	base = BONFIRE_MKISOFS_BASE (job);

	if (base->priv->xfer_handle) {
		gnome_vfs_async_cancel (base->priv->xfer_handle);
		base->priv->xfer_handle = NULL;
	}

	if (base->priv->report_id) {
		g_source_remove (base->priv->report_id);
		base->priv->report_id = 0;
	}

	if (base->priv->timer) {
		g_timer_destroy (base->priv->timer);
		base->priv->timer = NULL;
	}

	if (base->priv->rates) {
		g_slist_free (base->priv->rates);
		base->priv->rates = NULL;
	}
	
	return retval;
}

static BonfireBurnResult
bonfire_mkisofs_base_set_source (BonfireJob *job,
				 const BonfireTrackSource *source,
				 GError **error)
{
	char *label;
	BonfireMkisofsBase *base;

	g_return_val_if_fail (source != NULL, BONFIRE_BURN_ERR);

	base = BONFIRE_MKISOFS_BASE (job);

	if (source->type != BONFIRE_TRACK_SOURCE_DATA)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_mkisofs_base_clean_track (base);

	if (base->priv->source) {
		bonfire_track_source_free (base->priv->source);
		base->priv->source = NULL;
	}

	label = source->contents.data.label;
	if (label && (strlen (label) > 32)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_INVALID_FILE,
			     _("The label for the image is too long."));
		return BONFIRE_BURN_ERR;
	}

	base->priv->source = bonfire_track_source_copy (source);
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_base_set_output (BonfireImager *imager,
				 const char *output,
				 gboolean overwrite,
				 gboolean clean,
				 GError **error)
{
	BonfireMkisofsBase *base;

	base = BONFIRE_MKISOFS_BASE (imager);

	bonfire_mkisofs_base_clean_tmpdir (base);

	/* NOTE: we are supposed to create it so we own this directory */
	if (output)
		base->priv->tmpdir = g_strdup (output);

	base->priv->clean = clean;
	base->priv->overwrite = overwrite;
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_base_set_output_type (BonfireImager *imager,
				      BonfireTrackSourceType type,
				      GError **error)
{
	BonfireMkisofsBase *base;

	base = BONFIRE_MKISOFS_BASE (imager);

	if (type != BONFIRE_TRACK_SOURCE_GRAFTS
	&&  type != BONFIRE_TRACK_SOURCE_DEFAULT)
		return BONFIRE_BURN_NOT_SUPPORTED;

	/* NOTE : no need to keep this value since it can only ouput grafts */
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_base_get_track (BonfireImager *imager,
				BonfireTrackSource **source,
				GError **error)
{
	BonfireMkisofsBase *base;
	BonfireTrackSource *real;

	g_return_val_if_fail (source != NULL, BONFIRE_BURN_ERR);

	base = BONFIRE_MKISOFS_BASE (imager);

	if (!base->priv->grafts_path) {
		BonfireBurnResult result;

		result = bonfire_mkisofs_base_run (base, error);

		if (result != BONFIRE_BURN_OK)
			return result;
	}

	real = g_new0 (BonfireTrackSource, 1);

	real->type = BONFIRE_TRACK_SOURCE_GRAFTS;
	real->contents.grafts.grafts_path = g_strdup (base->priv->grafts_path);
	real->contents.grafts.excluded_path = g_strdup (base->priv->excluded_path);
	real->contents.grafts.label = g_strdup (base->priv->source->contents.data.label);

	*source = real;
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_base_get_action_string (BonfireJob *job,
					BonfireBurnAction action,
					char **string)
{
	BonfireMkisofsBase *base;

	base = BONFIRE_MKISOFS_BASE (job);

	if (action == BONFIRE_BURN_ACTION_FILE_COPY) {
		char *name;
		
		name = g_path_get_basename (base->priv->current_download);
		*string = g_strdup_printf (_("Copying \"%s\""), name);
		g_free (name);
	}
	else {
		const char *tmp;

		tmp = bonfire_burn_action_to_string (action);
		*string = g_strdup (tmp);
	}

	return BONFIRE_BURN_OK;
}
