/***************************************************************************
 *            recorder.c
 *
 *  dim jan 22 17:31:49 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include <nautilus-burn-drive.h>

#include "bonfire-marshal.h"
#include "burn-basics.h"
#include "burn-recorder.h"
#include "burn-job.h"

static void bonfire_recorder_base_init (gpointer g_class);

GType
bonfire_recorder_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireRecorderIFace),
			bonfire_recorder_base_init,
			NULL,
			NULL,
		};

		type = g_type_register_static (G_TYPE_INTERFACE, 
					       "BonfireRecorder",
					       &our_info,
					       0);
	}

	return type;
}

static void
bonfire_recorder_base_init (gpointer g_class)
{
	static gboolean initialized = FALSE;

        if (initialized)
		return;

	initialized = TRUE;
}

BonfireBurnResult
bonfire_recorder_set_drive (BonfireRecorder *recorder,
			    NautilusBurnDrive *drive,
			    GError **error)
{
	BonfireRecorderIFace *iface;

	g_return_val_if_fail (BONFIRE_IS_RECORDER (recorder), BONFIRE_BURN_ERR);
	g_return_val_if_fail (NAUTILUS_BURN_IS_DRIVE (drive), BONFIRE_BURN_ERR);

	if (!nautilus_burn_drive_can_write (drive)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the drive cannot write CDs or DVDs"));
		return BONFIRE_BURN_ERR;
	}

	if (bonfire_job_is_running (BONFIRE_JOB (recorder)))
		return BONFIRE_BURN_RUNNING;

	iface = BONFIRE_RECORDER_GET_IFACE (recorder);
	if (!iface->set_drive)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (recorder),
				   "recorder (%s) set_drive",
				   G_OBJECT_TYPE_NAME (recorder));
	return  (* iface->set_drive) (recorder,
				       drive,
				       error);
}

BonfireBurnResult
bonfire_recorder_set_flags (BonfireRecorder *recorder,
			    BonfireRecorderFlag flags,
			    GError **error)
{
	BonfireRecorderIFace *iface;

	g_return_val_if_fail (BONFIRE_IS_RECORDER (recorder), BONFIRE_BURN_ERR);

	if (bonfire_job_is_running (BONFIRE_JOB (recorder)))
		return BONFIRE_BURN_RUNNING;

	iface = BONFIRE_RECORDER_GET_IFACE (recorder);
	if (!iface->set_flags)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (recorder),
				   "recorder (%s) set_flags",
				   G_OBJECT_TYPE_NAME (recorder));
	return  (*iface->set_flags) (recorder, flags, error);
}

BonfireBurnResult
bonfire_recorder_blank (BonfireRecorder *recorder,
			GError **error)
{
	BonfireRecorderIFace *iface;

	g_return_val_if_fail (BONFIRE_IS_RECORDER (recorder), BONFIRE_BURN_ERR);

	if (bonfire_job_is_running (BONFIRE_JOB (recorder)))
		return BONFIRE_BURN_RUNNING;

	iface = BONFIRE_RECORDER_GET_IFACE (recorder);
	if (!iface->blank)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (recorder),
				   "recorder (%s) blank",
				   G_OBJECT_TYPE_NAME (recorder));
	return  (* iface->blank) (recorder, error);
}

BonfireBurnResult
bonfire_recorder_record (BonfireRecorder *recorder,
			 GError **error)
{
	BonfireRecorderIFace *iface;
	BonfireBurnResult result;

	g_return_val_if_fail (BONFIRE_IS_RECORDER (recorder), BONFIRE_BURN_ERR);

	if (bonfire_job_is_running (BONFIRE_JOB (recorder)))
		return BONFIRE_BURN_RUNNING;

	iface = BONFIRE_RECORDER_GET_IFACE (recorder);
	if (!iface->record)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (recorder),
				   "recorder (%s) record",
				   G_OBJECT_TYPE_NAME (recorder));
	result = (*iface->record) (recorder, error);

	if (result == BONFIRE_BURN_OK) {
		bonfire_job_progress_changed (BONFIRE_JOB (recorder), 
					      1.0,
					      -1);
	}

	return result;
}

BonfireBurnResult
bonfire_recorder_get_status (BonfireRecorder *recorder,
			     int *speed,
			     int *fifo,
			     gint64 *bytes_written)
{
	BonfireRecorderIFace *iface;

	g_return_val_if_fail (BONFIRE_IS_RECORDER (recorder), BONFIRE_BURN_ERR);

	if (!bonfire_job_is_running (BONFIRE_JOB (recorder)))
		return BONFIRE_BURN_NOT_RUNNING;

	iface = BONFIRE_RECORDER_GET_IFACE (recorder);
	if (!iface->get_status)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (recorder),
				   "recorder (%s) get_status",
				   G_OBJECT_TYPE_NAME (recorder));
	return  (* iface->get_status) (recorder,
				       speed,
				       fifo,
				       bytes_written);
}
