/***************************************************************************
 *            burn.c
 *
 *  ven mar  3 18:50:18 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include <libgnomevfs/gnome-vfs.h>

#include <nautilus-burn-drive.h>

#include "bonfire-marshal.h"
#include "burn-basics.h"
#include "burn.h"
#include "burn-job.h"
#include "burn-imager.h"
#include "burn-recorder.h"
#include "progress.h"
#include "burn-common.h"
#include "burn-caps.h"

static void bonfire_burn_class_init (BonfireBurnClass *klass);
static void bonfire_burn_init (BonfireBurn *sp);
static void bonfire_burn_finalize (GObject *object);

struct BonfireBurnPrivate {
	BonfireBurnCaps *caps;

	GMainLoop *sleep_loop;

	BonfireRecorder *recorder;
	BonfireImager *imager;

	NautilusBurnDrive *drive;

	NautilusBurnMediaType src_media_type;
	NautilusBurnMediaType dest_media_type;
	gboolean dest_rewritable;

	gint64 image_size;

	BonfireBurnProgress *progress;
};

#define IS_LOCKED	"LOCKED"

typedef enum {
	ASK_DISABLE_JOLIET_SIGNAL,
	WARN_DATA_LOSS_SIGNAL,
	WARN_REWRITABLE_SIGNAL,
	INSERT_MEDIA_REQUEST_SIGNAL,
	PROGRESS_CHANGED_SIGNAL,
	ACTION_CHANGED_SIGNAL,
	LAST_SIGNAL
} BonfireBurnSignalType;

static guint bonfire_burn_signals [LAST_SIGNAL] = { 0 };
static GObjectClass *parent_class = NULL;

GType
bonfire_burn_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireBurnClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_burn_class_init,
			NULL,
			NULL,
			sizeof (BonfireBurn),
			0,
			(GInstanceInitFunc)bonfire_burn_init,
		};

		type = g_type_register_static (G_TYPE_OBJECT, 
					       "BonfireBurn", 
					       &our_info,
					       0);
	}

	return type;
}

static void
bonfire_burn_class_init (BonfireBurnClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_burn_finalize;
	
	bonfire_burn_signals [ASK_DISABLE_JOLIET_SIGNAL] =
		g_signal_new ("disable_joliet",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (BonfireBurnClass,
					       ask_disable_joliet),
			      NULL, NULL,
			      bonfire_marshal_INT__VOID,
			      G_TYPE_INT, 0);
        bonfire_burn_signals [WARN_DATA_LOSS_SIGNAL] =
		g_signal_new ("warn_data_loss",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (BonfireBurnClass,
					       warn_data_loss),
			      NULL, NULL,
			      bonfire_marshal_INT__VOID,
			      G_TYPE_INT, 0);
        bonfire_burn_signals [WARN_REWRITABLE_SIGNAL] =
		g_signal_new ("warn_rewritable",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (BonfireBurnClass,
					       warn_rewritable),
			      NULL, NULL,
			      bonfire_marshal_INT__VOID,
			      G_TYPE_INT, 0);
        bonfire_burn_signals [INSERT_MEDIA_REQUEST_SIGNAL] =
		g_signal_new ("insert_media",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (BonfireBurnClass,
					       insert_media_request),
			      NULL, NULL,
			      bonfire_marshal_INT__INT_INT,
			      G_TYPE_INT, 
			      2,
			      G_TYPE_INT,
			      G_TYPE_INT);
        bonfire_burn_signals [PROGRESS_CHANGED_SIGNAL] =
		g_signal_new ("progress_changed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (BonfireBurnClass,
					       progress_changed),
			      NULL, NULL,
			      bonfire_marshal_VOID__DOUBLE_LONG,
			      G_TYPE_NONE, 
			      2,
			      G_TYPE_DOUBLE,
			      G_TYPE_LONG);
        bonfire_burn_signals [ACTION_CHANGED_SIGNAL] =
		g_signal_new ("action_changed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (BonfireBurnClass,
					       action_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__INT,
			      G_TYPE_NONE, 
			      1,
			      G_TYPE_INT);
}

static void
bonfire_burn_init (BonfireBurn *obj)
{
	obj->priv = g_new0 (BonfireBurnPrivate, 1);

	obj->priv->caps = bonfire_burn_caps_get_default ();
}

static void
bonfire_burn_finalize (GObject *object)
{
	BonfireBurn *cobj;
	cobj = BONFIRE_BURN(object);

	if (cobj->priv->sleep_loop) {
		g_main_loop_quit (cobj->priv->sleep_loop);
		cobj->priv->sleep_loop = NULL;
	}

	if (cobj->priv->caps)
		g_object_unref (cobj->priv->caps);
	if (cobj->priv->recorder)
		g_object_unref (cobj->priv->recorder);
	if (cobj->priv->imager)
		g_object_unref (cobj->priv->imager);
	if (cobj->priv->progress)
		g_object_unref (cobj->priv->progress);

	g_free (cobj->priv);
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

BonfireBurn *
bonfire_burn_new ()
{
	BonfireBurn *obj;
	
	obj = BONFIRE_BURN (g_object_new (BONFIRE_TYPE_BURN, NULL));
	
	return obj;
}

static gboolean
bonfire_burn_wakeup (BonfireBurn *burn)
{
	if (burn->priv->sleep_loop)
		g_main_loop_quit (burn->priv->sleep_loop);

	return FALSE;
}

static BonfireBurnResult
bonfire_burn_sleep (BonfireBurn *burn, gint msec)
{
	burn->priv->sleep_loop = g_main_loop_new (NULL, FALSE);
	g_timeout_add (msec,
		       (GSourceFunc) bonfire_burn_wakeup,
		       burn);

	g_main_loop_run (burn->priv->sleep_loop);

	if (burn->priv->sleep_loop) {
		g_main_loop_unref (burn->priv->sleep_loop);
		burn->priv->sleep_loop = NULL;
		return BONFIRE_BURN_OK;
	}

	/* if sleep_loop = NULL => We've been cancelled */
	return BONFIRE_BURN_CANCEL;
}

static void
bonfire_burn_progress_changed (BonfireJob *job,
			       double fraction,
			       long time_remaining,
			       BonfireBurn *burn)
{
	double progress;

	if (fraction == -1.0) {
		gint64 isosize = -1;
		gint64 bytes_written = -1;

		/* sometimes when burning on the fly the recorder doesn't know
		 * in advance the size of the image it's burning */
		if (burn->priv->image_size)
			isosize = burn->priv->image_size;
		else if (burn->priv->imager)
			bonfire_imager_get_size (burn->priv->imager,
						 &isosize,
						 FALSE,
						 NULL);

		if (burn->priv->recorder
		&&  bonfire_job_is_running (BONFIRE_JOB (burn->priv->recorder)))
			bonfire_recorder_get_status (burn->priv->recorder,
						     NULL,
						     NULL,
						     &bytes_written);

		if (isosize > 0 && bytes_written > 0)
			fraction = (double) bytes_written / isosize;
		else
			fraction = 0.0;
	}

	if (burn->priv->recorder) {
		if (burn->priv->imager
		&& !bonfire_job_is_running (BONFIRE_JOB (burn->priv->imager))
		&&  BONFIRE_JOB (burn->priv->recorder) != BONFIRE_JOB (burn->priv->imager)
		&& (BONFIRE_JOB (burn->priv->recorder) == job))
			progress = (fraction + 1.0) / 2.0;
		else
			progress = fraction;
	}
	/* this is when an imager has a slave running (for example downloading) */
	else if (burn->priv->imager
	      && !bonfire_job_is_running (BONFIRE_JOB (burn->priv->imager)))
		progress = fraction;
	else if (burn->priv->drive->type != NAUTILUS_BURN_DRIVE_TYPE_FILE)
		progress = fraction / 2.0;
	else
		progress = fraction;

	if (burn->priv->progress) {
		gint64 b_written = -1;
		gint64 isosize = -1;
		int mb_isosize = -1;
		int mb_written = -1;
		int speed = -1;
		int fifo = -1;

		if (burn->priv->image_size) {
			isosize = burn->priv->image_size;
		}
		else if (burn->priv->imager) {
			bonfire_imager_get_size (burn->priv->imager,
						 &isosize,
						 FALSE,
						 NULL);
		}

		mb_isosize = (int) (isosize / 1048576);
		mb_written = (int) mb_isosize * fraction;

		if (burn->priv->recorder
		&&  bonfire_job_is_running (BONFIRE_JOB (burn->priv->recorder))) {
			bonfire_recorder_get_status (burn->priv->recorder,
						     &speed,
						     &fifo,
						     &b_written);
			if (!mb_written)
				mb_written = (int) b_written / 1048576;
		}
		else if (burn->priv->imager
		      &&  bonfire_job_is_running (BONFIRE_JOB (burn->priv->imager)))
			bonfire_imager_get_speed (burn->priv->imager,
						  &speed);

		/* FIXME: with DVDs we'd better write speed in DVD speed not CD */
		bonfire_burn_progress_set_status (burn->priv->progress,
						  progress,
						  time_remaining,
						  mb_isosize,
						  mb_written,
						  speed,
						  fifo);
	}
						  
	g_signal_emit (burn,
		       bonfire_burn_signals [PROGRESS_CHANGED_SIGNAL],
		       0,
		       progress,
		       time_remaining);
}

static void
bonfire_burn_action_changed_real (BonfireBurn *burn,
				  BonfireBurnAction action,
				  BonfireJob *job)
{
	if (burn->priv->progress) {
		char *string = NULL;

		if (action == BONFIRE_BURN_ACTION_FINISHED) {
			string = g_strdup (bonfire_burn_action_to_string (action));
		}
		else if (BONFIRE_JOB (burn->priv->imager) == job) {
			bonfire_job_get_action_string (BONFIRE_JOB (burn->priv->imager),
						       action,
						       &string);
		}
		else if (BONFIRE_JOB (burn->priv->recorder) == job) {
			bonfire_job_get_action_string (BONFIRE_JOB (burn->priv->recorder),
						       action,
						       &string);
		}

		bonfire_burn_progress_set_action (burn->priv->progress,
						  action,
						  string);

		if (string)
			g_free (string);
	}

	g_signal_emit (burn,
		       bonfire_burn_signals [ACTION_CHANGED_SIGNAL],
		       0,
		       action);
}

static void
bonfire_burn_action_changed (BonfireJob *job,
			     BonfireBurnAction action,
			     BonfireBurn *burn)
{
	bonfire_burn_action_changed_real (burn, action, job);
}

BonfireBurnResult
bonfire_burn_status (BonfireBurn *burn,
		     gint64 *isosize,
		     gint64 *written,
		     gint *speed,
		     gint *fifo)
{
	if (burn->priv->recorder
	&&  bonfire_job_is_running (BONFIRE_JOB (burn->priv->recorder))) {
		bonfire_recorder_get_status (burn->priv->recorder,
					     speed,
					     fifo,
					     written);
		if (burn->priv->imager)
			bonfire_imager_get_size (burn->priv->imager,
						 isosize,
						 FALSE,
						 NULL);
	}
	else if (burn->priv->imager) {
		if (burn->priv->image_size)
			*isosize = burn->priv->image_size;
		else
			bonfire_imager_get_size (burn->priv->imager,
						 isosize,
						 FALSE,
						 NULL);

		if (bonfire_job_is_running (BONFIRE_JOB (burn->priv->imager)))
			bonfire_imager_get_speed (burn->priv->imager, speed);
	}
	else
		return BONFIRE_BURN_NOT_READY;

	return BONFIRE_BURN_OK;
}

static void
bonfire_burn_check_media (BonfireBurn *burn,
			  NautilusBurnDrive *drive,
			  NautilusBurnMediaType *type,
			  gboolean *is_rewritable,
			  gboolean *can_write,
			  gboolean *is_blank,
			  gboolean *has_audio,
			  gboolean *has_data)
{
	NautilusBurnMediaType real_type;
	gboolean is_blank_real = FALSE;

	/* if drive is mounted then unmount before checking anything */
	if (nautilus_burn_drive_is_mounted (drive)) {
		if (!nautilus_burn_drive_unmount (drive))
			g_warning ("Couldn't unmount volume in drive: %s", drive->device);
	}

	real_type = nautilus_burn_drive_get_media_type_full (drive,
							     is_rewritable,
							     &is_blank_real,
							     has_audio,
							     has_data);
	if (type)
		*type = real_type;
	if (is_blank)
		*is_blank = is_blank_real;
	if (can_write)
		*can_write = nautilus_burn_drive_media_type_is_writable (real_type,
									 is_blank_real);
}

static BonfireBurnResult
bonfire_burn_ask_for_media (BonfireBurn *burn,
			    NautilusBurnDrive *drive,
			    BonfireBurnError error_type,
			    BonfireMediaType required_media,
			    gboolean eject,
			    GError **error)
{
	gint64 media_size;
	gboolean is_reload;
	gboolean is_mounted;
	BonfireBurnResult result;
	NautilusBurnMediaType type;

	media_size = nautilus_burn_drive_get_media_size (drive);
	type = nautilus_burn_drive_get_media_type (drive);

	is_reload = (media_size > 0);
	if (type == NAUTILUS_BURN_MEDIA_TYPE_ERROR)
		is_reload = FALSE;

	/* check one more time */
	is_mounted = nautilus_burn_drive_is_mounted (drive);

	if (is_mounted == TRUE)
		error_type = BONFIRE_BURN_ERROR_MEDIA_BUSY;
	if (is_reload == FALSE)
		error_type = BONFIRE_BURN_ERROR_MEDIA_NONE;

	if (type != NAUTILUS_BURN_MEDIA_TYPE_ERROR
	&&  GPOINTER_TO_INT (g_object_get_data (G_OBJECT (drive), IS_LOCKED))
	&&  !nautilus_burn_drive_unlock (drive)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the drive can't be unlocked"));
		return BONFIRE_BURN_ERROR;
	}

	g_object_set_data (G_OBJECT (drive), IS_LOCKED, GINT_TO_POINTER (0));

	if (eject && !nautilus_burn_drive_eject (drive)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the disc can't be ejected"));
		return BONFIRE_BURN_ERR;
	}

	result = BONFIRE_BURN_CANCEL;
	g_signal_emit (burn,
		       bonfire_burn_signals [INSERT_MEDIA_REQUEST_SIGNAL],
		       0,
		       error_type,
		       required_media,
		       &result);

	return result;
}

BonfireBurnResult
bonfire_burn_wait_for_source_media (BonfireBurn *burn,
				    NautilusBurnDrive *drive,
				    gboolean eject,
				    GError **error)
{
	char *failure;
	gboolean is_blank;
	BonfireBurnResult result;
	NautilusBurnMediaType type;

 again:

	bonfire_burn_check_media (burn,
				  drive,
				  &type,
				  NULL,
				  NULL,
				  &is_blank,
				  NULL,
				  NULL);

	if (is_blank) {
		result = bonfire_burn_ask_for_media (burn,
						     drive,
						     BONFIRE_BURN_ERROR_MEDIA_BLANK,
						     BONFIRE_MEDIA_WITH_DATA,
						     eject,
						     error);
		if (result != BONFIRE_BURN_OK)
			return result;

		goto again;
	}

	/* we set IS_LOCKED to remind ourselves that we were the ones that locked it */
	if (!GPOINTER_TO_INT (g_object_get_data (G_OBJECT (drive), IS_LOCKED))
	&&  !nautilus_burn_drive_lock (drive, _("ongoing copying process"), &failure)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the drive can't be locked (%s)"),
			     failure);
		return BONFIRE_BURN_ERR;
	}

	g_object_set_data (G_OBJECT (drive), IS_LOCKED, GINT_TO_POINTER (1));
	burn->priv->src_media_type = type;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_burn_reload_src_media (BonfireBurn *burn,
			       BonfireBurnError error_code,
			       BonfireBurnFlag flags,
			       const BonfireTrackSource *source,
			       GError **error)
{
	BonfireBurnResult result;

	if (source->type != BONFIRE_TRACK_SOURCE_DISC)
		return BONFIRE_BURN_ERR;

	result = bonfire_burn_ask_for_media (burn,
					     source->contents.drive.disc,
					     error_code,
					     BONFIRE_MEDIA_WITH_DATA,
					     (flags & BONFIRE_BURN_FLAG_EJECT),
					     error);
	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_burn_wait_for_source_media (burn,
						     source->contents.drive.disc,
						     (flags & BONFIRE_BURN_FLAG_EJECT),
						     error);

	return result;
}

static BonfireBurnResult
bonfire_burn_wait_for_rewritable_media (BonfireBurn *burn,
					NautilusBurnDrive *drive,
					gboolean eject,
					GError **error)
{
	char *failure;
	gboolean is_blank;
	gboolean can_write;
	gboolean is_rewritable;
	BonfireBurnResult result;
	NautilusBurnMediaType type;

	if (!nautilus_burn_drive_can_rewrite (drive)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the drive has no rewriting capabilities"));
		return BONFIRE_BURN_NOT_SUPPORTED;
	}

 again:

	bonfire_burn_check_media (burn,
				  drive,
				  &type,
				  &is_rewritable,
				  &can_write,
				  &is_blank,
				  NULL,
				  NULL);

	if (is_blank || !is_rewritable) {
		result = bonfire_burn_ask_for_media (burn,
						     drive,
						     is_blank ? BONFIRE_BURN_ERROR_MEDIA_BLANK : BONFIRE_BURN_ERROR_MEDIA_NOT_REWRITABLE,
						     BONFIRE_MEDIA_REWRITABLE|
						     BONFIRE_MEDIA_WITH_DATA,
						     eject,
						     error);
		if (result != BONFIRE_BURN_OK)
			return result;

		goto again;
	}

	if (!GPOINTER_TO_INT (g_object_get_data (G_OBJECT (drive), IS_LOCKED))
	&&  !nautilus_burn_drive_lock (drive, _("ongoing blanking process"), &failure)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the drive can't be locked (%s)"),
			     failure);
		return BONFIRE_BURN_ERR;
	}

	g_object_set_data (G_OBJECT (drive), IS_LOCKED, GINT_TO_POINTER (1));

	burn->priv->dest_media_type = type;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_burn_blank_real (BonfireBurn *burn,
			 NautilusBurnDrive *drive,
			 BonfireRecorderFlag flags,
			 gboolean is_debug,
			 GError **error)
{
	BonfireBurnResult result;

	result = bonfire_burn_caps_create_recorder_for_blanking (burn->priv->caps,
								 &burn->priv->recorder,
								 burn->priv->dest_media_type,
								 error);
	if (result != BONFIRE_BURN_OK)
		return result;

	g_signal_connect (burn->priv->recorder,
			  "progress_changed",
			  G_CALLBACK (bonfire_burn_progress_changed),
			  burn);
	g_signal_connect (burn->priv->recorder,
			  "action_changed",
			  G_CALLBACK (bonfire_burn_action_changed),
			  burn);

	bonfire_job_set_debug (BONFIRE_JOB (burn->priv->recorder), is_debug);
	result = bonfire_recorder_set_drive (burn->priv->recorder,
					     drive,
					     error);
	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_recorder_set_flags (burn->priv->recorder,
					     flags,
					     error);
	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_recorder_blank (burn->priv->recorder, error);
	g_object_unref (burn->priv->recorder);
	burn->priv->recorder = NULL;

	return result;
}

BonfireBurnResult
bonfire_burn_blank (BonfireBurn *burn,
		    BonfireBurnFlag burn_flags,
		    NautilusBurnDrive *drive,
		    gboolean fast,
		    GError **error)
{
	BonfireRecorderFlag blank_flags;
	BonfireBurnResult result;
	GError *ret_error = NULL;

	/* we wait for the insertion of a media and lock it */
	g_return_val_if_fail (drive != NULL, BONFIRE_BURN_ERR);
	result = bonfire_burn_wait_for_rewritable_media (burn,
							 drive,
							 (burn_flags & BONFIRE_BURN_FLAG_EJECT),
							 error);
	if (result != BONFIRE_BURN_OK)
		return result;

	blank_flags = BONFIRE_RECORDER_FLAG_NONE;
	if (burn_flags & BONFIRE_BURN_FLAG_NOGRACE)
		blank_flags |= BONFIRE_RECORDER_FLAG_NOGRACE;

	if (burn_flags & BONFIRE_BURN_FLAG_DUMMY)
		blank_flags |= BONFIRE_RECORDER_FLAG_DUMMY;

	if (fast)
		blank_flags |= BONFIRE_RECORDER_FLAG_FAST_BLANK;

	result = bonfire_burn_blank_real (burn,
					  drive,
					  blank_flags,
					  (burn_flags & BONFIRE_BURN_FLAG_DEBUG) != 0,
					  &ret_error);

	while (result == BONFIRE_BURN_ERR
	&&     ret_error
	&&     ret_error->code == BONFIRE_BURN_ERROR_MEDIA_NOT_REWRITABLE) {
		g_error_free (ret_error);
		ret_error = NULL;
		
		result = bonfire_burn_ask_for_media (burn,
						     drive,
						     BONFIRE_BURN_ERROR_MEDIA_NOT_REWRITABLE,
						     BONFIRE_MEDIA_REWRITABLE|
						     BONFIRE_MEDIA_WITH_DATA,
						     (burn_flags & BONFIRE_BURN_FLAG_EJECT),
						     error);
		if (result != BONFIRE_BURN_OK)
			break;

		result = bonfire_burn_wait_for_rewritable_media (burn,
								 drive,
								 (burn_flags & BONFIRE_BURN_FLAG_EJECT),
								 error);
		if (result != BONFIRE_BURN_OK)
			return result;

		result = bonfire_burn_blank_real (burn,
						  drive,
						  blank_flags,
						  (burn_flags & BONFIRE_BURN_FLAG_DEBUG) != 0,
						  &ret_error);
	}

	if (ret_error)
		g_propagate_error (error, ret_error);

	nautilus_burn_drive_unlock (drive);
	g_object_set_data (G_OBJECT (drive), IS_LOCKED, GINT_TO_POINTER (0));

	if (burn_flags & BONFIRE_BURN_FLAG_EJECT)
		bonfire_burn_common_eject_async (drive);

	if (result == BONFIRE_BURN_OK)
		bonfire_burn_action_changed_real (burn,
						  BONFIRE_BURN_ACTION_FINISHED,
						  NULL);
	return result;
}

static BonfireBurnResult
bonfire_burn_wait_for_dest_media (BonfireBurn *burn,
				  BonfireBurnFlag flags,
				  NautilusBurnDrive *drive,
				  BonfireTrackSourceType track_type,
				  GError **error)
{
	char *failure;
	gint64 media_size;
	gboolean is_blank;
	gboolean can_write;
	BonfireBurnResult result;
	NautilusBurnMediaType type;
	gboolean is_rewritable_real;
	BonfireBurnError berror;

	if (!nautilus_burn_drive_can_write (drive)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the drive has no burning capabilities"));
		return BONFIRE_BURN_NOT_SUPPORTED;
	}

	result = BONFIRE_BURN_OK;

 again:

	berror = BONFIRE_BURN_ERROR_NONE;
	bonfire_burn_check_media (burn,
				  drive,
				  &type,
				  &is_rewritable_real,
				  &can_write,
				  &is_blank,
				  NULL,
				  NULL);

	if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (drive), IS_LOCKED))) {
		burn->priv->dest_media_type = type;
		burn->priv->dest_rewritable = is_rewritable_real;

		/* the drive has already been checked just return its characteristics */
		/* NOTE: after a blanking, for nautilus_burn the CD/DVD is still full of
		 * data so if the drive has already been checked there is no need to do
		 * that again since we would be asked if we want to blank it again */
		return result;
	}

	if (type == NAUTILUS_BURN_MEDIA_TYPE_BUSY) {
		result = BONFIRE_BURN_NEED_RELOAD;
		berror = BONFIRE_BURN_ERROR_MEDIA_BUSY;
		goto end;
	}

	if (!can_write) {
		result = BONFIRE_BURN_NEED_RELOAD;
		berror = BONFIRE_BURN_ERROR_MEDIA_NOT_WRITABLE;
		goto end;
	}

	if ((track_type != BONFIRE_TRACK_SOURCE_ISO
	&&   track_type != BONFIRE_TRACK_SOURCE_ISO_JOLIET
	&&   track_type != BONFIRE_TRACK_SOURCE_DATA
	&&   track_type != BONFIRE_TRACK_SOURCE_GRAFTS
	&&   track_type != BONFIRE_TRACK_SOURCE_DISC)
	&&  type > NAUTILUS_BURN_MEDIA_TYPE_CDRW) {
		result = BONFIRE_BURN_NEED_RELOAD;
		berror = BONFIRE_BURN_ERROR_DVD_NOT_SUPPORTED;
		goto end;
	}

	/* check that if we copy a CD/DVD we are copying it to an
	 * equivalent media (not a CD => DVD or a DVD => CD) */
	if (track_type == BONFIRE_TRACK_SOURCE_DISC) {
		gboolean is_src_DVD;
		gboolean is_dest_DVD;

		is_src_DVD = burn->priv->src_media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW;
		is_dest_DVD = burn->priv->dest_media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW;

		if (is_src_DVD != is_dest_DVD) {
			result = BONFIRE_BURN_NEED_RELOAD;
			if (is_src_DVD)
				berror = BONFIRE_BURN_ERROR_DVD_NOT_SUPPORTED;
			else
				berror = BONFIRE_BURN_ERROR_CD_NOT_SUPPORTED;

			goto end;
		}
	}

	/* we check that the image will fit on the media */
	/* FIXME: that doesn't really work with multisession medias since a part
	 * of the size returned by next function is occupied by data. Wait for NCB 2.15 */
	media_size = nautilus_burn_drive_get_media_size (drive);
	if (!(flags & BONFIRE_BURN_FLAG_OVERBURN)
	&&  media_size < burn->priv->image_size) {
		/* This is a recoverable error so try to ask the user again */
		result = BONFIRE_BURN_NEED_RELOAD;
		berror = BONFIRE_BURN_ERROR_MEDIA_SPACE;
		goto end;
	}

	if (!nautilus_burn_drive_lock (drive, _("ongoing burning process"), &failure)) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the drive can't be locked (%s)"),
			     failure);
		return BONFIRE_BURN_ERR;
	}

	g_object_set_data (G_OBJECT (drive), IS_LOCKED, GINT_TO_POINTER (1));

	if (!nautilus_burn_drive_can_rewrite (drive))
		is_rewritable_real = FALSE;

	burn->priv->dest_media_type = type;
	burn->priv->dest_rewritable = is_rewritable_real;

	/* silently ignore if the drive is not rewritable */
	/* NOTE: we corrected such contradictory flags setting (might be an error) as
	 * BONFIRE_BURN_FLAG_MERGE|BONFIRE_BURN_FLAG_APPEND|BONFIRE_BURN_FLAG_BLANK_BEFORE_WRITE */
	/* FIXME: strange according to DVD_RW are blank ?? */
	if ((flags & (BONFIRE_BURN_FLAG_MERGE|BONFIRE_BURN_FLAG_APPEND)) == 0
	&&  (flags & BONFIRE_BURN_FLAG_BLANK_BEFORE_WRITE)
	&&  is_rewritable_real
	&&  !is_blank) {
		g_signal_emit (burn,
			       bonfire_burn_signals [WARN_DATA_LOSS_SIGNAL],
			       0,
			       &result);

		if (result != BONFIRE_BURN_OK)
			goto end;
	
		if (type != NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW) {
			BonfireRecorderFlag blank_flags = BONFIRE_RECORDER_FLAG_FAST_BLANK;

			if (flags & BONFIRE_BURN_FLAG_DUMMY)
				blank_flags |= BONFIRE_RECORDER_FLAG_DUMMY;
	
			if (flags & BONFIRE_BURN_FLAG_NOGRACE)
				blank_flags |= BONFIRE_RECORDER_FLAG_NOGRACE;
	
			result = bonfire_burn_blank_real (burn,
							  drive,
							  blank_flags,
							  (flags & BONFIRE_BURN_FLAG_DEBUG) != 0,
							  error);
		}
	}

end:

	if (result == BONFIRE_BURN_NEED_RELOAD) {
		BonfireMediaType required_media;

		required_media = bonfire_burn_caps_get_required_media_type (burn->priv->caps,
									    track_type);

		result = bonfire_burn_ask_for_media (burn,
						     drive,
						     berror,
						     required_media,
						     (flags & BONFIRE_BURN_FLAG_EJECT),
						     error);
		if (result == BONFIRE_BURN_OK)
			goto again;
	}

	if (result != BONFIRE_BURN_OK) {
		g_object_set_data (G_OBJECT (drive), IS_LOCKED, GINT_TO_POINTER (0));
		nautilus_burn_drive_unlock (drive);
	}

	return result;
}

static BonfireBurnResult
bonfire_burn_reload_dest_media (BonfireBurn *burn,
				BonfireBurnError error_code,
				BonfireBurnFlag flags,
				NautilusBurnDrive *drive,
				const BonfireTrackSource *source,
				GError **error)
{
	BonfireTrackSourceType track_type;
	BonfireMediaType required_media;
	BonfireBurnResult result;

again:

	/* eject and ask the user to reload a disc */
	if (source->type != BONFIRE_TRACK_SOURCE_ISO
	||  source->type != BONFIRE_TRACK_SOURCE_ISO_JOLIET
	||  source->type != BONFIRE_TRACK_SOURCE_DATA
	||  source->type != BONFIRE_TRACK_SOURCE_GRAFTS
	||  source->type != BONFIRE_TRACK_SOURCE_DISC)
		required_media = BONFIRE_MEDIA_WRITABLE|BONFIRE_MEDIA_TYPE_CD;
	else if (track_type == BONFIRE_TRACK_SOURCE_DISC) {
		/* the required media depends on the source */
		if (burn->priv->src_media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
			required_media = BONFIRE_MEDIA_WRITABLE|BONFIRE_MEDIA_TYPE_DVD;
		else
			required_media = BONFIRE_MEDIA_WRITABLE|BONFIRE_MEDIA_TYPE_CD;
	}
	else /* we accept DVD and CD */
		required_media = BONFIRE_MEDIA_WRITABLE;

	result = bonfire_burn_ask_for_media (burn,
					     drive,
					     error_code,
					     required_media,
					     (flags & BONFIRE_BURN_FLAG_EJECT),
					     error);
	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_burn_wait_for_dest_media (burn,
						   flags,
						   drive,
						   source->type,
						   error);
	if (result == BONFIRE_BURN_NEED_RELOAD)
		goto again;

	return result;
}

static BonfireBurnResult
bonfire_burn_set_recorder_speed (BonfireBurn *burn,
				 gint speed)
{
	gint64 rate;
	BonfireBurnResult result;

	/* set up the object */
	if (burn->priv->dest_media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
		rate = speed * DVD_SPEED;
	else
		rate = speed * CDR_SPEED;

	result = bonfire_job_set_rate (BONFIRE_JOB (burn->priv->recorder), rate);
	if (result != BONFIRE_BURN_OK
	&&  result != BONFIRE_BURN_NOT_SUPPORTED)
		return result;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_burn_setup_recorder (BonfireBurn *burn,
			     BonfireBurnFlag flags,
			     NautilusBurnDrive *drive,
			     gint speed,
			     BonfireTrackSource *track,
			     GError **error)
{
	BonfireRecorderFlag rec_flags;
	BonfireBurnResult result;

	if ((flags & BONFIRE_BURN_FLAG_DEBUG) != 0)
		bonfire_job_set_debug (BONFIRE_JOB (burn->priv->recorder), TRUE);

	/* set up the flags */
	rec_flags = BONFIRE_RECORDER_FLAG_NONE;

	if (flags & BONFIRE_BURN_FLAG_NOGRACE)
		rec_flags |= BONFIRE_RECORDER_FLAG_NOGRACE;

	if (flags & BONFIRE_BURN_FLAG_OVERBURN)
		rec_flags |= BONFIRE_RECORDER_FLAG_OVERBURN;

	if (flags & BONFIRE_BURN_FLAG_BURNPROOF)
		rec_flags |= BONFIRE_RECORDER_FLAG_BURNPROOF;

	if (flags & BONFIRE_BURN_FLAG_DAO)
		rec_flags |= BONFIRE_RECORDER_FLAG_DAO;

	if (flags & BONFIRE_BURN_FLAG_DONT_CLOSE)
		rec_flags |= BONFIRE_RECORDER_FLAG_MULTI;

	if (flags & BONFIRE_BURN_FLAG_DUMMY)
		rec_flags |= BONFIRE_RECORDER_FLAG_DUMMY;

	/* set up the object */
	bonfire_burn_set_recorder_speed (burn, speed);
	result = bonfire_recorder_set_drive (burn->priv->recorder,
					     drive,
					     error);
	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_recorder_set_flags (burn->priv->recorder,
					     rec_flags,
					     error);
	if (result != BONFIRE_BURN_OK)
		return result;

	if (track)
		result = bonfire_job_set_source (BONFIRE_JOB (burn->priv->recorder),
						 track,
						 error);

	return result;
}

static BonfireBurnResult
bonfire_burn_get_recorder (BonfireBurn *burn,
			   BonfireBurnFlag flags,
			   NautilusBurnDrive *drive,
			   BonfireTrackSource *track,
			   gint speed,
			   GError **error)
{
	BonfireBurnResult result;
	BonfireRecorder *recorder = NULL;

	if (burn->priv->recorder) {
		/* just in case */
		g_object_unref (burn->priv->recorder);
		burn->priv->recorder = NULL;
	}

	/* create the appropriate recorder object */
	result = bonfire_burn_caps_create_recorder (burn->priv->caps,
						    &recorder,
						    track,
						    burn->priv->dest_media_type,
						    error);
	if (result != BONFIRE_BURN_OK)
		return result;

	if (!recorder)
		return BONFIRE_BURN_NOT_SUPPORTED;

	burn->priv->recorder = recorder;
	g_signal_connect (burn->priv->recorder,
			  "progress_changed",
			  G_CALLBACK (bonfire_burn_progress_changed),
			  burn);
	g_signal_connect (burn->priv->recorder,
			  "action_changed",
			  G_CALLBACK (bonfire_burn_action_changed),
			  burn);

	return bonfire_burn_setup_recorder (burn,
					     flags,
					     drive,
					     speed,
					     track,
					     error);
}

static BonfireBurnResult
bonfire_burn_setup_imager (BonfireBurn *burn,
			   BonfireBurnFlag flags,
			   NautilusBurnDrive *drive,
			   const char *output,
			   GError **error)
{
	BonfireBurnResult result;

	if (!burn->priv->imager)
		return BONFIRE_BURN_OK;

	result = bonfire_imager_set_output (burn->priv->imager,
					    output,
					    (flags & BONFIRE_BURN_FLAG_DONT_OVERWRITE) == 0,
					    (flags & BONFIRE_BURN_FLAG_DONT_CLEAN_OUTPUT) == 0,
					    error);

	if ((result != BONFIRE_BURN_OK && result != BONFIRE_BURN_NOT_SUPPORTED)
	||  (result == BONFIRE_BURN_NOT_SUPPORTED && output != NULL))
		return result;

	if ((flags & (BONFIRE_BURN_FLAG_MERGE|BONFIRE_BURN_FLAG_APPEND))) {
		result = bonfire_imager_set_append (burn->priv->imager,
						    drive,
						    (flags & BONFIRE_BURN_FLAG_MERGE) != 0,
						    error);
		if (result != BONFIRE_BURN_OK)
			return result;
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_burn_create_imager (BonfireBurn *burn,
			    BonfireBurnFlag flags,
			    NautilusBurnDrive *drive,
			    const BonfireTrackSource *source,
			    GError **error)
{
	BonfireTrackSourceType target = BONFIRE_TRACK_SOURCE_DEFAULT;
	BonfireBurnResult result = BONFIRE_BURN_OK;
	BonfireImager *imager = NULL;

	if (source->type == BONFIRE_TRACK_SOURCE_DISC)
		target = source->contents.drive.target;

	result = bonfire_burn_caps_create_imager (burn->priv->caps,
						  &imager,
						  source,
						  target,
						  burn->priv->src_media_type,
						  burn->priv->dest_media_type,
						  error);

	if (result != BONFIRE_BURN_OK)
		return result;

	if (!imager)
		return BONFIRE_BURN_NOT_SUPPORTED;

	/* better connect to the signals quite early (especially in the case of 
	 * Mkisofs that might have to download things) */
	burn->priv->imager = imager;
	g_signal_connect (burn->priv->imager,
			  "progress_changed",
			  G_CALLBACK (bonfire_burn_progress_changed),
			  burn);
	g_signal_connect (burn->priv->imager,
			  "action_changed",
			  G_CALLBACK (bonfire_burn_action_changed),
			  burn);

	/* configure the object */
	result = bonfire_job_set_source (BONFIRE_JOB (imager), source, error);
	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_imager_set_output_type (imager, target, error);
	if (result != BONFIRE_BURN_OK)
		return result;

	/* special case for imagers that are also recorders (ie, cdrdao, growisofs)
	 * they usually need the drive if we want to calculate the size */
	if (BONFIRE_IS_RECORDER (imager)) {
		result = bonfire_recorder_set_drive (BONFIRE_RECORDER (imager),
						     drive,
						     error);

		if (result != BONFIRE_BURN_OK)
			return result;
	}

	if ((flags & BONFIRE_BURN_FLAG_DEBUG) != 0)
		bonfire_job_set_debug (BONFIRE_JOB (imager), TRUE);

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_burn_ask_for_joliet (BonfireBurn *burn)
{
	BonfireBurnResult result;

	g_signal_emit (burn,
		       bonfire_burn_signals [ASK_DISABLE_JOLIET_SIGNAL],
		       0,
		       &result);

	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_imager_set_output_type (burn->priv->imager,
						 BONFIRE_TRACK_SOURCE_ISO,
						 NULL);

	return result;
}

static BonfireBurnResult
bonfire_burn_check_volume_free_space (BonfireBurn *burn,
				      const char *output,
				      GError **error)
{
	char *dirname;
	char *uri_str;
	GnomeVFSURI *uri;
	GnomeVFSResult result;
	GnomeVFSFileSize vol_size;

	if (!output)
		dirname = g_strdup (g_get_tmp_dir ());
	else
		dirname = g_path_get_dirname (output);

	uri_str = gnome_vfs_get_uri_from_local_path (dirname);
	g_free (dirname);

	uri = gnome_vfs_uri_new (uri_str);
	g_free (uri_str);

	if (uri == NULL)
		return BONFIRE_BURN_ERR;

	result = gnome_vfs_get_volume_free_space (uri, &vol_size);
	if (result != GNOME_VFS_OK) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the size of the volume can't be checked (%s)"),
			     gnome_vfs_result_to_string (result));
		gnome_vfs_uri_unref (uri);
		return BONFIRE_BURN_ERR;
	}
	gnome_vfs_uri_unref (uri);

	if (burn->priv->image_size > vol_size) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_DISC_SPACE,
			     _("the selected location does not have enough free space to store the disc image (%ld MiB needed)"),
			     (unsigned long) burn->priv->image_size / 1048576);
		return BONFIRE_BURN_ERR;
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_burn_get_imager (BonfireBurn *burn, 
			 BonfireBurnFlag flags,
			 NautilusBurnDrive *drive,
			 const BonfireTrackSource *source,
			 const char *output,
			 GError **error)
{
	BonfireBurnResult result;
	GError *ret_error = NULL;

	/* just in case */
	if (burn->priv->imager) {
		g_object_unref (burn->priv->imager);
		burn->priv->imager = NULL;
	}

	result = bonfire_burn_create_imager (burn,
					     flags,
					     drive,
					     source,
					     error);
	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_burn_setup_imager (burn,
					    flags,
					    drive,
					    output,
					    error);
	if (result != BONFIRE_BURN_OK)
		return result;

	/* we get the size and (will) check that if we're writing to a disc,
	 * the media is big enough and/or if we're writing to a hard drive,
	 * the volume is big enough */
	/* NOTE: this part is important since it is actually a test that the 
	 * imager _can_ work with such a setup */
	result = bonfire_imager_get_size (BONFIRE_IMAGER (burn->priv->imager),
					  &burn->priv->image_size,
					  FALSE,
					  &ret_error);

	if (result == BONFIRE_BURN_ERR) {
		BonfireBurnError error_code;

		error_code = ret_error->code;

		/* we try to handle a possible joliet recoverable error */
		if (error_code == BONFIRE_BURN_ERROR_JOLIET_TREE)
			result = bonfire_burn_ask_for_joliet (burn);

		if (result == BONFIRE_BURN_OK) {
			g_error_free (ret_error);
			ret_error = NULL;

			/* we retry without joliet this time */
			result = bonfire_imager_get_size (BONFIRE_IMAGER (burn->priv->imager),
							  &burn->priv->image_size,
							  FALSE,
							  &ret_error);
		}
	}

	/* other errors are unrecoverable anyway */
	if (ret_error)
		g_propagate_error (error, ret_error);

	return result;
}

static BonfireBurnResult
bonfire_burn_run_imager (BonfireBurn *burn,
			 BonfireBurnFlag flags,
			 const BonfireTrackSource *source,
			 BonfireTrackSource **track,
			 const char *output,
			 GError **error)
{
	BonfireBurnError error_code;
	BonfireBurnResult result;
	GError *ret_error = NULL;


start:

	result = bonfire_imager_get_track (burn->priv->imager,
					   track,
					   &ret_error);
	if (result != BONFIRE_BURN_ERR || !ret_error)
		return result;

	/* See if we can recover from the error */
	error_code = ret_error->code;
	if (error_code == BONFIRE_BURN_ERROR_JOLIET_TREE) {
		/* some files are not conforming to Joliet standard see
		 * if the user wants to carry on with a non joliet disc */
		result = bonfire_burn_ask_for_joliet (burn);
		if (result != BONFIRE_BURN_OK) {
			g_propagate_error (error, ret_error);
			return result;
		}

		g_error_free (ret_error);
		ret_error = NULL;
		goto start;
	}
	else if (error_code == BONFIRE_BURN_ERROR_MEDIA_BLANK) {
		/* clean the error anyway since at worst the user will cancel */
		g_error_free (ret_error);
		ret_error = NULL;

		/* The media has data on it: ask for a new one:
		 * NOTE: we'll check the size later after the retry */
		result = bonfire_burn_reload_src_media (burn,
							error_code,
							flags,
							source,
							error);
		if (result != BONFIRE_BURN_OK)
			return result;

		return BONFIRE_BURN_RETRY;
	}

	/* not recoverable propagate the error */
	g_propagate_error (error, ret_error);

	return BONFIRE_BURN_ERR;
}

static BonfireBurnResult
bonfire_burn_run_recorder (BonfireBurn *burn,
			   BonfireBurnFlag flags,
			   NautilusBurnDrive *drive,
			   int speed,
			   const BonfireTrackSource *source,
			   BonfireTrackSource *track,
			   const char *output,
			   GError **error)
{
	int error_code;
	gboolean has_slept;
	GError *ret_error = NULL;
	BonfireBurnResult result;

	has_slept = FALSE;

start:

	result = bonfire_recorder_record (burn->priv->recorder, &ret_error);
	if (result != BONFIRE_BURN_ERR || !ret_error)
		return result;

	/* see if error is recoverable */
	error_code = ret_error->code;
	if (error_code == BONFIRE_BURN_ERROR_JOLIET_TREE) {
		/* NOTE: this error can only come from the source when 
		 * burning on the fly => no need to recreate an imager */

		/* some files are not conforming to Joliet standard see
		 * if the user wants to carry on with a non joliet disc */
		result = bonfire_burn_ask_for_joliet (burn);
		if (result != BONFIRE_BURN_OK) {
			g_propagate_error (error, ret_error);
			return result;
		}

		g_error_free (ret_error);
		ret_error = NULL;
		goto start;
	}
	else if (error_code == BONFIRE_BURN_ERROR_MEDIA_BLANK) {
		/* NOTE: this error can only come from the source when 
		 * burning on the fly => no need to recreate an imager */

		/* The source media (when copying on the fly) is empty 
		 * so ask the user to reload another media with data */
		g_error_free (ret_error);
		ret_error = NULL;

		result = bonfire_burn_reload_src_media (burn,
							error_code,
							flags,
							source,
							error);
		if (result != BONFIRE_BURN_OK)
			return result;

		return BONFIRE_BURN_RETRY;
	}
	else if (error_code == BONFIRE_BURN_ERROR_SLOW_DMA) {
		/* The whole system has just made a great effort. Sometimes it 
		 * helps to let it rest for a sec or two => that's what we do
		 * before retrying. (That's why usually cdrecord waits a little
	         * bit but sometimes it doesn't). Another solution would be to
		 * lower the speed a little (we could do both) */
		g_error_free (ret_error);
		ret_error = NULL;

		bonfire_burn_sleep (burn, 2000);
		has_slept = TRUE;

		/* set speed at 8x max and even less if speed  */
		if (speed <= 8) {
			speed = speed * 3 / 4;
			if (speed < 1)
				speed = 1;
		}
		else
			speed = 8;

		bonfire_burn_set_recorder_speed (burn, speed);
		goto start;
	}
	else if (error_code >= BONFIRE_BURN_ERROR_DISC_SPACE) {
		/* NOTE: these errors can only come from the dest drive */

		/* clean error and indicates this is a recoverable error */
		g_error_free (ret_error);
		ret_error = NULL;

		/* ask for the destination media reload */
		result = bonfire_burn_reload_dest_media (burn,
							 error_code,
							 flags,
							 drive,
							 source, /* this is not an error since the required media depends on the source track */
							 error);

		if (result != BONFIRE_BURN_OK)
			return result;

		return BONFIRE_BURN_RETRY;
	}

	g_propagate_error (error, ret_error);
	return BONFIRE_BURN_ERR;
}

static BonfireBurnResult
bonfire_burn_imager_get_track (BonfireBurn *burn,
			       BonfireBurnFlag flags,
			       NautilusBurnDrive *drive,
			       const BonfireTrackSource *source,
			       BonfireTrackSource **track,
			       const char *output,
			       GError **error)
{
	GError *ret_error = NULL;
	BonfireBurnResult result;

	result = bonfire_burn_get_imager (burn,
					  flags,
					  drive,
					  source,
					  output,
					  &ret_error);

	if (result != BONFIRE_BURN_OK)
		return result;

	if (flags & BONFIRE_BURN_FLAG_ON_THE_FLY) {
		BonfireTrackSource *source;

		source = g_new0 (BonfireTrackSource, 1);
		source->type = BONFIRE_TRACK_SOURCE_IMAGER;
		source->contents.imager.obj = burn->priv->imager;
		burn->priv->imager = NULL;

		if (track)
			*track = source;

		return BONFIRE_BURN_OK;
	}

	/* Since we are writing to disc we'd better check there is enough space */
	if (flags & BONFIRE_BURN_FLAG_CHECK_SIZE)
		result = bonfire_burn_check_volume_free_space (burn,
							       output,
							       &ret_error);

	if (result == BONFIRE_BURN_OK)
		result = bonfire_burn_run_imager (burn,
						  flags,
						  source,
						  track,
						  output,
						  &ret_error);

	/* propagate error if needed */
	if (ret_error)
		g_propagate_error (error, ret_error);

	return result;
}

static BonfireBurnResult
bonfire_burn_get_size (BonfireBurn *burn,
		       BonfireBurnFlag flags,
		       const BonfireTrackSource *source,
		       GError **error)
{
	BonfireBurnResult result = BONFIRE_BURN_OK;
	GnomeVFSFileInfo *info;
	GnomeVFSResult res;
	char *uri = NULL;

	switch (source->type) {
	case BONFIRE_TRACK_SOURCE_INF:
		return BONFIRE_BURN_NOT_SUPPORTED;

	case BONFIRE_TRACK_SOURCE_CUE:
		uri = g_strconcat ("file://", source->contents.cue.image, NULL);
		break;

	case BONFIRE_TRACK_SOURCE_RAW:
		uri = g_strconcat ("file://", source->contents.raw.image, NULL);
		break;

	case BONFIRE_TRACK_SOURCE_ISO:
	case BONFIRE_TRACK_SOURCE_ISO_JOLIET:
		uri = g_strconcat ("file://", source->contents.iso.image, NULL);
		break;

	default:
		return BONFIRE_BURN_NOT_SUPPORTED;	
	}

	info = gnome_vfs_file_info_new ();
	res = gnome_vfs_get_file_info (uri, 
				       info,
				       GNOME_VFS_FILE_INFO_DEFAULT);

	burn->priv->image_size = info->size;

	gnome_vfs_file_info_clear (info);
	gnome_vfs_file_info_unref (info);
	g_free (uri);

	if (res != GNOME_VFS_OK) {
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("the file %s can't be opened (%s)"),
			     uri,
			     gnome_vfs_result_to_string (res));
		burn->priv->image_size = -1;
		return BONFIRE_BURN_ERR;
	}

	return result;
}

static BonfireBurnResult
bonfire_burn_lock_drives (BonfireBurn *burn,
			  BonfireBurnFlag flags,
			  NautilusBurnDrive *drive,
			  const BonfireTrackSource *source,
			  GError **error)
{
	BonfireBurnResult result;

	burn->priv->src_media_type = NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN;
	burn->priv->dest_media_type = NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN;

	nautilus_burn_drive_ref (drive);
	burn->priv->drive = drive;

	/* For source drive, the rule is if the source type is a disc, lock it
	 * and if source is not the same as dest, lock dest as well */
	if (source->type == BONFIRE_TRACK_SOURCE_DISC) {
		result = bonfire_burn_wait_for_source_media (burn,
							     source->contents.drive.disc,
							     (flags & BONFIRE_BURN_FLAG_EJECT),
							     error);
		if (result != BONFIRE_BURN_OK)
			return result;

		if (nautilus_burn_drive_equal (drive, source->contents.drive.disc)) {
			/* we can't lock the dest since src == dest we
			 * will ask the user to replace the disc later */
			return BONFIRE_BURN_OK;
		}
	}

	/* we don't lock the dest drive if we just want an image of it,
	 * except if we append/merge/burn on the fly we need to make sure now
	 * there is a disc in the dest drive so as to lock it, because:
	 * - if it's a DVD we'll use it anyway
	 * - to append or merge cdr* programs need to know where to start the track */
	if (drive->type == NAUTILUS_BURN_DRIVE_TYPE_FILE)
		return BONFIRE_BURN_OK;

	/* lock the recorder */
	result = bonfire_burn_wait_for_dest_media (burn,
						   flags,
						   drive,
						   source->type,
						   error);

again:

	if (burn->priv->dest_rewritable) {
		BonfireTrackSourceType type = BONFIRE_TRACK_SOURCE_UNKNOWN;

		/* emits a warning for the user if it's a rewritable
		 * disc and he wants to write only audio tracks on it */
		result = BONFIRE_BURN_OK;

		if (source->type == BONFIRE_TRACK_SOURCE_DISC) {
			gboolean has_audio = FALSE;
			gboolean has_data = FALSE;

			bonfire_burn_check_media (burn,
						  source->contents.drive.disc,
						  NULL,
						  NULL,
						  NULL,
						  NULL,
						  &has_audio,
						  &has_data);

			if (has_audio && !has_data)
				type = BONFIRE_TRACK_SOURCE_AUDIO;
			else
				type = BONFIRE_TRACK_SOURCE_DISC;
		}
		else
			type = source->type;

		/* NOTE: no need to error out here since the only thing
		 * we are interested in is if it is AUDIO or not or if
		 * the disc we are copying has audio tracks only or not */
		if (result == BONFIRE_BURN_OK
		&& (type == BONFIRE_TRACK_SOURCE_AUDIO
		||  type == BONFIRE_TRACK_SOURCE_INF
		||  type == BONFIRE_TRACK_SOURCE_SONG)) {
			g_signal_emit (burn,
				       bonfire_burn_signals [WARN_REWRITABLE_SIGNAL],
				       0,
				       &result);

			if (result == BONFIRE_BURN_NEED_RELOAD) {
				result = bonfire_burn_reload_dest_media (burn,
									 BONFIRE_BURN_ERROR_NONE,
									 flags,
									 drive,
									 source,
									 error);
				if (result != BONFIRE_BURN_OK)
					return result;

				goto again;
			}
		}
	}

	return result;
}

static BonfireBurnResult
bonfire_burn_unlock_drives (BonfireBurn *burn,
			    BonfireBurnFlag flags,
			    NautilusBurnDrive *drive,
			    const BonfireTrackSource *source)
{
	/* dest drive */
	nautilus_burn_drive_unref (burn->priv->drive);
	burn->priv->drive = NULL;

	nautilus_burn_drive_unlock (drive);
	g_object_set_data (G_OBJECT (drive), IS_LOCKED, GINT_TO_POINTER (0));

	if ((flags & BONFIRE_BURN_FLAG_EJECT))
		bonfire_burn_common_eject_async (drive);

	/* source drive if any */
	if (source->type != BONFIRE_TRACK_SOURCE_DISC)
		return BONFIRE_BURN_OK;

	if (nautilus_burn_drive_equal (drive, source->contents.drive.disc))
		return BONFIRE_BURN_OK;

	nautilus_burn_drive_unlock (source->contents.drive.disc);
	g_object_set_data (G_OBJECT (source->contents.drive.disc),
			   IS_LOCKED,
			   GINT_TO_POINTER (0));

	if ((flags & BONFIRE_BURN_FLAG_EJECT))
		bonfire_burn_common_eject_async (source->contents.drive.disc);

	return BONFIRE_BURN_OK;
}
			    
/* FIXME: for the moment we don't allow for mixed CD type */
/* FIXME: for the moment we can't deal with non local images (cues, iso, bin) maybe a new object ImagerNull */
static BonfireBurnResult
bonfire_burn_record_real (BonfireBurn *burn,
			  BonfireBurnFlag flags,
			  NautilusBurnDrive *drive,
			  gint speed,
			  const BonfireTrackSource *source,
			  const char *output,
			  GError **error)
{
	BonfireBurnResult result;
	BonfireTrackSource *track = NULL;

	/* transform the source track through an imager if needed */
	if (source->type == BONFIRE_TRACK_SOURCE_DATA
	||  source->type == BONFIRE_TRACK_SOURCE_GRAFTS
	||  source->type == BONFIRE_TRACK_SOURCE_SONG
	||  source->type == BONFIRE_TRACK_SOURCE_DISC) {
		result = bonfire_burn_imager_get_track (burn,
							flags,
							drive,
							source,
							&track,
							output,
							error);
	
		if (result != BONFIRE_BURN_OK)
			goto end;
	}
	else {
		/* The track is ready to be used as is by a recorder */

		/* we get the size and (will) check that if we're writing to a disc,
		 * the media is big enough and/or if we're writing to a hard drive,
		 * the volume is big enough */
		result = bonfire_burn_get_size (burn,
						flags,
						source,
						error);
		if (result != BONFIRE_BURN_OK)
			goto end;

		track = bonfire_track_source_copy (source);
	}

	if (drive->type == NAUTILUS_BURN_DRIVE_TYPE_FILE)
		goto end;

	/* before recording if we are copying a disc check that the source and 
	 * the destination drive are not the same. Otherwise reload the media */
	if (source->type == BONFIRE_TRACK_SOURCE_DISC
	&&  nautilus_burn_drive_equal (drive, source->contents.drive.disc)) {
		/* NOTE: we use track->contents.drive.disc here
		 * so as to keep the IS_LOCKED value consistent */
		if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (source->contents.drive.disc), IS_LOCKED)))
			g_object_set_data (G_OBJECT (drive), IS_LOCKED, GINT_TO_POINTER (1));
		else
			g_object_set_data (G_OBJECT (drive), IS_LOCKED, GINT_TO_POINTER (0));

		result = bonfire_burn_reload_dest_media (burn,
							 BONFIRE_BURN_ERROR_NONE,
							 flags,
							 drive,
							 source,
							 error);
		if (result != BONFIRE_BURN_OK)
			goto end;
	}

	/* it happens with certain objects (cdrdao, growisofs) that they are both
	 * imagers and recorders so there's no need to recreate a recorder */
	if (track->type == BONFIRE_TRACK_SOURCE_IMAGER
	&&  BONFIRE_IS_RECORDER (track->contents.imager.obj)) {
		g_object_ref (track->contents.imager.obj);
		burn->priv->recorder = BONFIRE_RECORDER (track->contents.imager.obj);
		result = bonfire_burn_setup_recorder (burn,
						      flags,
						      drive,
						      speed,
						      NULL,
						      error);
	}
	else
		result = bonfire_burn_get_recorder (burn,
						    flags,
						    drive,
						    track,
						    speed,
						    error);

	if (result == BONFIRE_BURN_OK)
		result = bonfire_burn_run_recorder (burn,
						    flags,
						    drive,
						    speed,
						    source,
						    track,
						    output,
						    error);


end:

	if (track && track != source)
		bonfire_track_source_free (track);

	return result;
}

BonfireBurnResult 
bonfire_burn_record (BonfireBurn *burn,
		     BonfireBurnFlag flags,
		     NautilusBurnDrive *drive,
		     gint speed,
		     const BonfireTrackSource *source,
		     const char *output,
		     GError **error)
{
	GError *ret_error = NULL;
	BonfireBurnResult result;

	g_return_val_if_fail (drive != NULL, BONFIRE_BURN_ERR);
	g_return_val_if_fail (source->type != BONFIRE_TRACK_SOURCE_IMAGER,
			      BONFIRE_BURN_NOT_SUPPORTED);

	/* we do some drive locking quite early to make sure we have a media
	 * in the drive so that we'll have all the necessary information */
	result = bonfire_burn_lock_drives (burn,
					   flags,
					   drive,
					   source,
					   &ret_error);
	if (result != BONFIRE_BURN_OK)
		goto end;

	do {
		/* check flags consistency.
		 * NOTE: it's a necessary step when we retry since a supported 
		 * flag with one element could not be supported by its fallback */
		flags = bonfire_burn_caps_check_flags_consistency (burn->priv->caps,
								   source->type,
								   drive->type,
								   burn->priv->dest_media_type,
								   flags);

		if (ret_error) {
			g_error_free (ret_error);
			ret_error = NULL;
		}

		result = bonfire_burn_record_real (burn,
						   flags,
						   drive,
						   speed,
						   source,
						   output,
						   &ret_error);

		/* clean up */
		if (burn->priv->recorder) {
			g_object_unref (burn->priv->recorder);
			burn->priv->recorder = NULL;
		}

		if (burn->priv->imager) {
			g_object_unref (burn->priv->imager);
			burn->priv->imager = NULL;
		}
	} while (result == BONFIRE_BURN_RETRY);


end:

	/* unlock all drives */
	bonfire_burn_unlock_drives (burn, flags, drive, source);

	/* we handle all results/errors here*/
	if (ret_error) {
		g_propagate_error (error, ret_error);
		ret_error = NULL;
	}

	if (result == BONFIRE_BURN_OK)
		bonfire_burn_action_changed_real (burn,
						  BONFIRE_BURN_ACTION_FINISHED,
						  NULL);
	else if (result == BONFIRE_BURN_NOT_READY
	     ||  result == BONFIRE_BURN_NOT_SUPPORTED
	     ||  result == BONFIRE_BURN_RUNNING
	     ||  result == BONFIRE_BURN_NOT_RUNNING)
		g_set_error (error,
			     BONFIRE_BURN_ERROR,
			     BONFIRE_BURN_ERROR_GENERAL,
			     _("internal error (code %i)"),
			     result);


	return result;
}

BonfireBurnResult
bonfire_burn_cancel (BonfireBurn *burn, gboolean protect)
{
	BonfireBurnResult result = BONFIRE_BURN_OK;

	g_return_val_if_fail (burn != NULL, BONFIRE_BURN_ERR);

	if (burn->priv->sleep_loop) {
		g_main_loop_quit (burn->priv->sleep_loop);
		burn->priv->sleep_loop = NULL;
	}

	if (burn->priv->recorder
	&&  bonfire_job_is_running (BONFIRE_JOB (burn->priv->recorder)))
		result = bonfire_job_cancel (BONFIRE_JOB (burn->priv->recorder), protect);
	else if (burn->priv->imager
	      &&  BONFIRE_JOB (burn->priv->imager) != BONFIRE_JOB (burn->priv->recorder))
		result = bonfire_job_cancel (BONFIRE_JOB (burn->priv->imager), protect);
	else if (burn->priv->recorder)
		result = bonfire_job_cancel (BONFIRE_JOB (burn->priv->recorder), protect);
	else if (burn->priv->imager)
		result = bonfire_job_cancel (BONFIRE_JOB (burn->priv->imager), protect);

	return result;
}
			     
void
bonfire_burn_plug_progress_widget (BonfireBurn *burn,
				   BonfireBurnProgress *progress)
{
	g_return_if_fail (BONFIRE_IS_BURN_PROGRESS (progress));

	if (burn->priv->progress)
		g_object_unref (burn->priv->progress);

	g_object_ref (progress);
	burn->priv->progress = progress;
}
