/***************************************************************************
 *            project-size.c
 *
 *  dim nov 27 14:33:46 2005
 *  Copyright  2005  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <gtk/gtkhbox.h>
#include <gtk/gtkprogressbar.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtkcombobox.h>
#include <gtk/gtkalignment.h>
#include <gtk/gtkcelllayout.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkstock.h>

#include <nautilus-burn-drive.h>

#include "project-size.h"
#include "utils.h"

static void bonfire_project_size_class_init (BonfireProjectSizeClass *klass);
static void bonfire_project_size_init (BonfireProjectSize *sp);
static void bonfire_project_size_finalize (GObject *object);

static void
bonfire_project_size_combo_destroy_cb (GtkObject *object,
				       BonfireProjectSize *size);
static void
bonfire_project_size_refresh (BonfireProjectSize *obj);

static void
bonfire_project_size_fill_model (BonfireProjectSize *self);

static void
bonfire_project_size_row_changed_cb (GtkTreeModel *model,
				     GtkTreePath *path,
				     GtkTreeIter *iter,
				     BonfireProjectSize *size);
static void
bonfire_project_size_combo_changed_cb (GtkComboBox *combo,
				       BonfireProjectSize *size);

static void
bonfire_project_size_disc_added_cb (NautilusBurnDrive *drive,
				    BonfireProjectSize *size);
static void
bonfire_project_size_disc_removed_cb (NautilusBurnDrive *drive,
				      BonfireProjectSize *size);

struct BonfireProjectSizePrivate {
	GtkWidget *progress;
	GtkWidget *combo;

	gint64 disc_size;
	gint64 max_size;
	gint64 size;

	char *size_string;
	int refresh_id;

	gboolean is_audio_context:1;
	gboolean is_model_filled:1;
};

enum {
	BONFIRE_PROJECT_SIZE_COMBO_DISPLAY_COL,
	BONFIRE_PROJECT_SIZE_COMBO_DRIVE_COL,
	BONFIRE_PROJECT_SIZE_COMBO_SIZE_COL,
	BONFIRE_PROJECT_SIZE_COMBO_VISIBLE_COL,
	BONFIRE_PROJECT_SIZE_COMBO_NB_COL
};

typedef enum {
	DISC_CHANGED_SIGNAL,
	LAST_SIGNAL
} BonfireProjectSizeSignalType;


static guint bonfire_project_size_signals[LAST_SIGNAL] = { 0 };
static GObjectClass *parent_class = NULL;

GType
bonfire_project_size_get_type()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireProjectSizeClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_project_size_class_init,
			NULL,
			NULL,
			sizeof (BonfireProjectSize),
			0,
			(GInstanceInitFunc)bonfire_project_size_init,
		};

		type = g_type_register_static (GTK_TYPE_HBOX, 
					       "BonfireProjectSize",
					       &our_info, 0);
	}

	return type;
}

static void
bonfire_project_size_class_init (BonfireProjectSizeClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	
	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_project_size_finalize;

	bonfire_project_size_signals [DISC_CHANGED_SIGNAL] =
	    g_signal_new ("disc_changed",
			  G_OBJECT_CLASS_TYPE (object_class),
			  G_SIGNAL_ACTION | G_SIGNAL_RUN_FIRST,
			  G_STRUCT_OFFSET (BonfireProjectSizeClass, disc_changed),
			  NULL, NULL,
			  g_cclosure_marshal_VOID__VOID,
			  G_TYPE_NONE,
			  0);

}

static void
bonfire_project_size_init (BonfireProjectSize *obj)
{
	GtkListStore *store;
	GtkTreeModel *filter;
	GtkCellRenderer *renderer;

	obj->priv = g_new0 (BonfireProjectSizePrivate, 1);
	gtk_box_set_spacing (GTK_BOX (obj), 6);

	/* progress bar */
	obj->priv->progress = gtk_progress_bar_new ();
	gtk_progress_bar_set_text (GTK_PROGRESS_BAR (obj->priv->progress), _("empty"));
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (obj->priv->progress), 0.0);

	/* combo box */
	obj->priv->combo = gtk_combo_box_new ();
	g_signal_connect (G_OBJECT (obj->priv->combo),
			  "destroy",
			  G_CALLBACK (bonfire_project_size_combo_destroy_cb),
			  obj);

	store = gtk_list_store_new (BONFIRE_PROJECT_SIZE_COMBO_NB_COL,
				    G_TYPE_STRING,
				    G_TYPE_POINTER,
				    G_TYPE_INT64,
				    G_TYPE_BOOLEAN);

	filter = gtk_tree_model_filter_new (GTK_TREE_MODEL (store), NULL);
	gtk_tree_model_filter_set_visible_column (GTK_TREE_MODEL_FILTER (filter), BONFIRE_PROJECT_SIZE_COMBO_VISIBLE_COL);
	g_object_unref (store);

	gtk_combo_box_set_model (GTK_COMBO_BOX (obj->priv->combo), GTK_TREE_MODEL (filter));
	g_object_unref (filter);

	g_signal_connect_after (G_OBJECT (store),
				"row-changed",
				G_CALLBACK (bonfire_project_size_row_changed_cb),
				obj);
	g_signal_connect_after (G_OBJECT (obj->priv->combo),
				"changed",
				G_CALLBACK (bonfire_project_size_combo_changed_cb),
				obj);

	renderer = gtk_cell_renderer_text_new ();
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (obj->priv->combo),
				    renderer,
				    TRUE);
	gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (obj->priv->combo),
				       renderer,
				       "text", BONFIRE_PROJECT_SIZE_COMBO_DISPLAY_COL);

	gtk_combo_box_set_active (GTK_COMBO_BOX (obj->priv->combo), 0);

	gtk_box_pack_end (GTK_BOX (obj), obj->priv->combo, FALSE, FALSE, 0);
	gtk_box_pack_end (GTK_BOX (obj), obj->priv->progress, TRUE, TRUE, 0);
}

static void
bonfire_project_size_free_store (BonfireProjectSize *size)
{
	NautilusBurnDrive *drive;
	GtkTreeModel *store;
	GtkTreeIter iter;

	store = gtk_combo_box_get_model (GTK_COMBO_BOX (size->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (store));

	if (!gtk_tree_model_get_iter_first (store, &iter))
		return;

	do {
		drive = NULL;
		gtk_tree_model_get (store, &iter,
				    BONFIRE_PROJECT_SIZE_COMBO_DRIVE_COL, &drive,
				    -1);

		if (drive) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (drive),
							      bonfire_project_size_disc_added_cb,
							      size);
			g_signal_handlers_disconnect_by_func (G_OBJECT (drive),
							      bonfire_project_size_disc_removed_cb,
							      size);
			nautilus_burn_drive_unref (drive);
		}
	} while (gtk_tree_model_iter_next (store, &iter));

	gtk_list_store_clear (GTK_LIST_STORE (store));
	size->priv->is_model_filled = 0;
}

static void
bonfire_project_size_combo_destroy_cb (GtkObject *combo,
				       BonfireProjectSize *size)
{
	if (size->priv->combo) {
		GtkTreeModel *filter;

		filter = gtk_combo_box_get_model (GTK_COMBO_BOX (combo));
		g_signal_handlers_disconnect_by_func (G_OBJECT (filter),
						      bonfire_project_size_row_changed_cb,
						      size);
		bonfire_project_size_free_store (size);
		size->priv->combo = NULL;
	}
}

static void
bonfire_project_size_finalize (GObject *object)
{
	BonfireProjectSize *cobj;

	cobj = BONFIRE_PROJECT_SIZE (object);
	
	if (cobj->priv->refresh_id)
		g_source_remove (cobj->priv->refresh_id);

	if (cobj->priv->size_string)
		g_free (cobj->priv->size_string);

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
bonfire_project_size_new ()
{
	BonfireProjectSize *obj;
	
	obj = BONFIRE_PROJECT_SIZE (g_object_new (BONFIRE_TYPE_PROJECT_SIZE, NULL));

	return GTK_WIDGET (obj);
}

/************************************* COMBO STUFF *****************************/
static gchar *
bonfire_project_size_get_media_string (BonfireProjectSize *self,
				       NautilusBurnDrive *drive,
				       gint64 size)
{
	gchar *text;

	/* we should round the disc sizes / length */
	if (size == -1) {
		if (drive) {
			char *name;

			name = nautilus_burn_drive_get_name_for_display (drive);
			text = g_strdup_printf (_("(mounted drive) %s "), name);
			g_free (name);
		}
		else
			return NULL;
	}
	else {
		char *size_string = NULL;

		if (self->priv->is_audio_context)
			size_string = bonfire_utils_get_time_string_from_size (size, TRUE, TRUE);
		else
			size_string = bonfire_utils_get_size_string (size, TRUE, TRUE);

		if (drive) {
			char *name;

			name = nautilus_burn_drive_get_name_for_display (drive);
			text = g_strdup_printf ("%s (%s)",
						size_string,
						name);
			g_free (name);
		}
		else
			text = g_strdup_printf ("%s", size_string);

		g_free (size_string);
	}

	return text;
}

static void
bonfire_project_size_add_default_medias (BonfireProjectSize *self)
{
	gint i, max;
	gchar *text;
	GtkTreeIter row;
	GtkTreeModel *store;
	const gint64 len_array [] = { 650.4 * 1024 * 1024,
					700 * 1024 * 1024,
					800 * 1024 * 1024,
					900 * 1024 * 1024, 
					4.3 * 1024 * 1024 * 1024,
					4.7 * 1024 * 1024 * 1024 };

	store = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (store));

	/* we add the fixed sizes */
	if (self->priv->is_audio_context)
		max = 4;
	else
		max = sizeof (len_array) / sizeof (len_array [0]);

	for (i = 0; i < max; i++) {
		text = bonfire_project_size_get_media_string (self,
							      NULL,
							      len_array [i]);

		gtk_list_store_append (GTK_LIST_STORE (store), &row);
		gtk_list_store_set (GTK_LIST_STORE (store), &row,
				    BONFIRE_PROJECT_SIZE_COMBO_DISPLAY_COL, text,
				    BONFIRE_PROJECT_SIZE_COMBO_SIZE_COL, len_array [i],
				    BONFIRE_PROJECT_SIZE_COMBO_DRIVE_COL, NULL,
				    BONFIRE_PROJECT_SIZE_COMBO_VISIBLE_COL, TRUE,
				    -1);
		g_free (text);
	}
}

static void
bonfire_project_size_add_media (BonfireProjectSize *self,
				NautilusBurnDrive *drive,
				gint64 size)
{
	char *text;
	GtkTreeIter row;
	GtkTreeModel *store;

	/* search for the right line */
	store = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (store));

	if (!gtk_tree_model_get_iter_first (store, &row))
		return;

	text = bonfire_project_size_get_media_string (self,
						      drive,
						      size);

	do {
		NautilusBurnDrive *list_drive;

		list_drive = NULL;
		gtk_tree_model_get (store, &row,
				    BONFIRE_PROJECT_SIZE_COMBO_DRIVE_COL, &list_drive,
				    -1);

		if (list_drive && nautilus_burn_drive_equal (drive, list_drive)) {
			gtk_list_store_set (GTK_LIST_STORE (store), &row,
					    BONFIRE_PROJECT_SIZE_COMBO_DISPLAY_COL, text,
					    BONFIRE_PROJECT_SIZE_COMBO_VISIBLE_COL, TRUE,
					    BONFIRE_PROJECT_SIZE_COMBO_SIZE_COL, size,
					    -1);
			break;
		}

	} while (gtk_tree_model_iter_next (store, &row));
	g_free (text);
}

static void
bonfire_project_size_disc_added_cb (NautilusBurnDrive *drive,
				    BonfireProjectSize *self)
{
	NautilusBurnMediaType media;
	gboolean is_blank;
	gint64 size;

	media = nautilus_burn_drive_get_media_type_full (drive,
							 NULL,
							 &is_blank,
							 NULL,
							 NULL);

	if (!nautilus_burn_drive_media_type_is_writable (media, is_blank))
		return;

	size = nautilus_burn_drive_get_media_size (drive);
	bonfire_project_size_add_media (self,
					drive,
					size);
}

static void
bonfire_project_size_disc_removed_cb (NautilusBurnDrive *drive,
				      BonfireProjectSize *size)
{
	NautilusBurnDrive *list_drive;
	GtkTreeModel *store;
	GtkTreeIter row;

	store = gtk_combo_box_get_model (GTK_COMBO_BOX (size->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (store));

	if (gtk_tree_model_get_iter_first (store, &row) == FALSE)
		return;

	do {
		gtk_tree_model_get (store,
				    &row,
				    BONFIRE_PROJECT_SIZE_COMBO_DRIVE_COL, &list_drive,
				    -1);

		if (!list_drive)
			continue;

		if (nautilus_burn_drive_equal (drive, list_drive)) {
			gtk_list_store_set (GTK_LIST_STORE (store), &row,
					    BONFIRE_PROJECT_SIZE_COMBO_VISIBLE_COL, FALSE,
					    -1);
			break;
		}
	} while (gtk_tree_model_iter_next (store, &row) == TRUE);
}

static void
bonfire_project_size_fill_model (BonfireProjectSize *self)
{
	GList *iter, *list;
	GtkTreeIter row;
	GtkTreeModel *store;
	NautilusBurnDrive *drive;
	NautilusBurnMediaType media;

	bonfire_project_size_free_store (self);
	bonfire_project_size_add_default_medias (self);

	list = nautilus_burn_drive_get_list (TRUE, FALSE);

	store = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (store));

	for (iter = list; iter; iter = iter->next) {
		gboolean is_blank;
		gint64 size;

		drive = iter->data;

		/* add a callback if media changes */
		g_object_set (drive, "enable-monitor", TRUE, NULL);
		g_signal_connect (G_OBJECT (drive), "media-added",
				  G_CALLBACK (bonfire_project_size_disc_added_cb),
				  self);
		g_signal_connect (G_OBJECT (drive), "media-removed",
				  G_CALLBACK (bonfire_project_size_disc_removed_cb),
				  self);

		/* FIXME: in 2.15 there is this function IS_DVD */
		media = nautilus_burn_drive_get_media_type_full (drive,
								 NULL,
								 &is_blank,
								 NULL,
								 NULL);

		/* we add the drive to the combo */
		gtk_list_store_prepend (GTK_LIST_STORE (store), &row);
		gtk_list_store_set (GTK_LIST_STORE (store), &row,
				    BONFIRE_PROJECT_SIZE_COMBO_DISPLAY_COL, NULL,
				    BONFIRE_PROJECT_SIZE_COMBO_DRIVE_COL, drive,
				    BONFIRE_PROJECT_SIZE_COMBO_VISIBLE_COL, FALSE,
				    -1);

		if (!nautilus_burn_drive_media_type_is_writable (media, is_blank))
			continue;

		/* FIXME: when we add an appendable disc maybe we could display 
		 * its label. Moreover we should get the name of the already 
		 * existing label in burn-dialog.c */
		size = nautilus_burn_drive_get_media_size (drive);
		bonfire_project_size_add_media (self,
						drive,
						size);
	}

	g_list_free (list);
	self->priv->is_model_filled = 1;
}

void
bonfire_project_size_set_context (BonfireProjectSize *size,
				  gboolean is_audio)
{
	if (size->priv->is_model_filled && is_audio == size->priv->is_audio_context)
		return;

	size->priv->is_audio_context = is_audio;
	bonfire_project_size_fill_model (size);
}

static void
bonfire_project_size_set_first_disc_active (BonfireProjectSize *size, GtkTreeIter *iter)
{
	NautilusBurnDrive *drive;
	GtkTreeModel *filter;
	GtkTreeModel *store;
	gboolean visible;

	filter = gtk_combo_box_get_model (GTK_COMBO_BOX (size->priv->combo));
	gtk_tree_model_get (filter,
			    iter,
			    BONFIRE_PROJECT_SIZE_COMBO_DRIVE_COL, &drive,
			    BONFIRE_PROJECT_SIZE_COMBO_VISIBLE_COL, &visible,
			    -1);

	if (!drive) {
		GtkTreeIter row;

		/* The currently selected row is not a real disc.
		 * Try to replace it with the newly added real disc */
		store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (filter));
		gtk_tree_model_get_iter_first (store, &row);

		while (1) {
			visible = FALSE;
			drive = NULL;

			gtk_tree_model_get (store, &row,
					    BONFIRE_PROJECT_SIZE_COMBO_DRIVE_COL, &drive,
					    BONFIRE_PROJECT_SIZE_COMBO_VISIBLE_COL, &visible,
					    -1);

			if (drive && visible) {
				gtk_tree_model_filter_convert_child_iter_to_iter (GTK_TREE_MODEL_FILTER (filter),
										  iter,
										  &row);

				g_signal_handlers_block_by_func (G_OBJECT (size->priv->combo),
								 bonfire_project_size_combo_changed_cb,
								 size);

				gtk_combo_box_set_active_iter (GTK_COMBO_BOX (size->priv->combo), iter);

				g_signal_handlers_unblock_by_func (G_OBJECT (size->priv->combo),
								   bonfire_project_size_combo_changed_cb,
								   size);
				break;
			}
	
			if (!gtk_tree_model_iter_next (store, &row))
				break;
		}
	}
}

static void
bonfire_project_size_combo_changed (BonfireProjectSize *size, gboolean added)
{
	GtkTreeModel *store;
	GtkTreeIter iter;
	gint64 cd_size;

	store = gtk_combo_box_get_model (GTK_COMBO_BOX (size->priv->combo));
	if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (size->priv->combo), &iter) == FALSE) {
		NautilusBurnDrive *drive = NULL;
		gboolean visible;

		/* no default active iter => set one preferably an inserted writable disc */
		if (!gtk_tree_model_get_iter_first (store, &iter))
			return;

		while (1) {
			drive = NULL;
			gtk_tree_model_get (store,
					    &iter,
					    BONFIRE_PROJECT_SIZE_COMBO_DRIVE_COL, &drive,
					    BONFIRE_PROJECT_SIZE_COMBO_VISIBLE_COL, &visible,
					    -1);

			if (drive != NULL)
				break;

			if (!gtk_tree_model_iter_next (store, &iter)) {
				gtk_tree_model_get_iter_first (store, &iter);
				break;
			}
		}

		g_signal_handlers_block_by_func (G_OBJECT (size->priv->combo),
						 bonfire_project_size_combo_changed_cb,
						 size);

		gtk_combo_box_set_active_iter (GTK_COMBO_BOX (size->priv->combo),
					       &iter);

		g_signal_handlers_unblock_by_func (G_OBJECT (size->priv->combo),
						   bonfire_project_size_combo_changed_cb,
						   size);
	}
	else if (added)
		bonfire_project_size_set_first_disc_active (size, &iter);

	gtk_tree_model_get (store, &iter,
			    BONFIRE_PROJECT_SIZE_COMBO_SIZE_COL, &cd_size,
			    -1);

	size->priv->disc_size = cd_size;
	size->priv->max_size = cd_size * 105 / 100;
	bonfire_project_size_refresh (size);

	g_signal_emit (size,
		       bonfire_project_size_signals [DISC_CHANGED_SIGNAL],
		       0);
}

static void
bonfire_project_size_row_changed_cb (GtkTreeModel *model,
				     GtkTreePath *path,
				     GtkTreeIter *iter,
				     BonfireProjectSize *size)
{
	/* this means that a row was inserted (we only change them after insertion) */
	bonfire_project_size_combo_changed (size, TRUE);
}

static void
bonfire_project_size_combo_changed_cb (GtkComboBox *combo,
				       BonfireProjectSize *size)
{
	bonfire_project_size_combo_changed (size, FALSE);
}

static gboolean
bonfire_project_size_update_size (BonfireProjectSize *obj)
{
	double fraction = 0.0;
	char *string;

	if (!obj->priv->disc_size) {
		fraction = 0.0;
		string = g_strdup_printf (_("no disc chosen (%s)"),
					  obj->priv->size_string);
	}
	else if (obj->priv->size > obj->priv->disc_size) {
		fraction = 1.0;
		string = g_strdup_printf (_("oversized (%s)"),
					  obj->priv->size_string);
	}
	else if (obj->priv->size <= 0.0) {
		fraction = 0.0;
		string = g_strdup (_("empty"));
	}
	else if (obj->priv->size_string
	     &&  obj->priv->size_string [0] != '\0') {
		string = g_strdup (obj->priv->size_string);
		fraction = (gdouble) ((gdouble)obj->priv->size / (gdouble) obj->priv->disc_size);
	}
	else
		string = g_strdup (_("empty"));

	gtk_progress_bar_set_text (GTK_PROGRESS_BAR (obj->priv->progress), string);
	g_free (string);

	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (obj->priv->progress),
				       fraction);

	obj->priv->refresh_id = 0;
	return FALSE;
}

static void
bonfire_project_size_refresh (BonfireProjectSize *obj)
{
	if (!obj->priv->refresh_id)
		obj->priv->refresh_id = g_timeout_add (500,
						       (GSourceFunc) bonfire_project_size_update_size,
						       obj);
}

void
bonfire_project_size_set_size (BonfireProjectSize *obj,
			       double size,
			       const char *string)
{
	obj->priv->size = size;
	if (obj->priv->size_string)
		g_free (obj->priv->size_string);

	obj->priv->size_string = g_strdup (string);

	/* we don't update the size right now but in half a second.
	 * when exploring directories size can changed repeatedly
	 * and we don't want to lose too much time updating.
	 * if a size is already set, we know that we're waiting for
	 * a size update, so, just replace the old size. otherwise
	 * we add a g_timeout_add */
	bonfire_project_size_refresh (obj);
}

gboolean
bonfire_project_size_check_status (BonfireProjectSize *size,
				   gboolean *overburn)
{
	if (size->priv->max_size < size->priv->size) {
		*overburn = FALSE;
		return FALSE;
	}

	if (size->priv->disc_size < size->priv->size) {
		*overburn = TRUE;
		return FALSE;
	}

	return TRUE;
}
