#include <stdint.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/mman.h>
#include <linux/fb.h>
#include <ppm.h>


static void
usage()
{
	fprintf( stderr, "usage: boot-icon posX posY image.ppm\n" );
	exit( 1 );
}

// framebuffer state
static int fd;			// file descriptor of fb
static uint8_t* fb;	// framebuffer memory

static struct fb_fix_screeninfo finf;
static struct fb_var_screeninfo vinf;

static struct fb_cmap cmap;

static void (*putpixel)(int,int,int,int,int,int);

static int bytes_per_pixel;
static int mult_r, mult_g, mult_b;
static int shift_r, shift_g, shift_b;


// framebuffer handling



// 8 bits
static void
putpixel_8( int x, int y, int r, int g, int b, int maxval )
{
	// get palette index
	fb[ x + y*finf.line_length ] = 16 +
		(r*6/maxval) + (g*6/maxval)*6 + (b*6/maxval)*36;
}

static void
init_8()
{
	int i;
	// use 'web-palette' for colors>16
	cmap.start=16;
	cmap.len=216;
	cmap.red = malloc( 216 * sizeof(unsigned short) );
	cmap.green = malloc( 216 * sizeof(unsigned short) );
	cmap.blue = malloc( 216 * sizeof(unsigned short) );
	cmap.transp = NULL;
	for( i=0; i<216; i++ ) {
		cmap.red[i] = (i % 6) * (65535 / 6);
		cmap.green[i] = ((i/6) % 6) * (65535 / 6);
		cmap.blue[i] = ((i/36) % 6) * (65535 / 6);
	}
	ioctl(fd, FBIOPUTCMAP, &cmap);

	putpixel = putpixel_8;
}


static void
putpixel_direct( int x, int y, int r, int g, int b, int maxval )
{
	uint8_t *p;
	uint32_t val;

	r = r * mult_r / maxval;
	g = g * mult_g / maxval;
	b = b * mult_b / maxval;

	val = r<<shift_r | g<<shift_g | b<<shift_b;
	p = fb + y * finf.line_length + x * bytes_per_pixel;
	memcpy( p, &val, bytes_per_pixel );
}

static void
init_direct()
{
	bytes_per_pixel = (vinf.bits_per_pixel+7) / 8;

	mult_r = (1<<vinf.red.length) - 1;
	mult_g = (1<<vinf.green.length) - 1;
	mult_b = (1<<vinf.blue.length) - 1;

	shift_r = vinf.red.offset;
	shift_g = vinf.green.offset;
	shift_b = vinf.blue.offset;

	putpixel = putpixel_direct;
}

static void
open_fb()
{
	fd = open( "/dev/fb/0", O_RDWR );
	if( fd==-1 )
	fd = open( "/dev/fb0", O_RDWR );
	if( fd==-1 ) {
		perror("open framebuffer");
		exit( 2 );
	}
	// get information about the framebuffer
	ioctl(fd, FBIOGET_FSCREENINFO, &finf);
	ioctl(fd, FBIOGET_VSCREENINFO, &vinf);

	if( finf.type != FB_TYPE_PACKED_PIXELS ) {
		fprintf( stderr, "unsupported framebuffer type: %d\n",
				finf.type );
		exit( 2 );
	}
	switch( finf.visual ) {
	case FB_VISUAL_PSEUDOCOLOR:
		init_8();
		break;
	case FB_VISUAL_TRUECOLOR:
	case FB_VISUAL_DIRECTCOLOR:
		init_direct();
		break;
	default:
		fprintf( stderr, "unsupported visual: %d\n", finf.visual );
		exit( 2 );
	}

	// map framebuffer
	fb = mmap((void *)0, finf.smem_len, PROT_READ | PROT_WRITE,MAP_SHARED, fd, 0);
}

static void
close_fb()
{
	munmap( (void *)fb, finf.smem_len );
	close( fd );
}

int
main(int argc, char** argv)
{
	FILE *f;
	int x, y, posx, posy;
	int width, height;
	pixval maxval;
	pixel **p;

	ppm_init( &argc, argv );
	if( argc != 4 ) usage();

	// get position
	posx = atoi( argv[1] );
	posy = atoi( argv[2] );

	// open image
	if( strcmp( argv[3], "-" )==0 ) {
		f = stdin;
	} else {
		f = fopen( argv[3], "rb" );
		if( !f ) {
			perror( "open" );
			return( 1 );
		}
	}
	// read image
	p = ppm_readppm( f, &width, &height, &maxval );
	// close image
	if( f!=stdin )
		fclose( f );

	// open framebuffer
	open_fb();

	if( posx < 0 ) posx += vinf.xres_virtual - width;
	if( posy < 0 ) posy += vinf.yres_virtual - height;

	// draw image
	for( y=0; y < height; y++ )
		for( x=0; x < width; x++ )
			putpixel( posx+x, posy+y, 
				PPM_GETR(p[y][x]),
				PPM_GETG(p[y][x]),
				PPM_GETB(p[y][x]),
				maxval );

	// close framebuffer
	close_fb();
	return( 0 );
}
