//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/SampleItem.cpp
//! @brief     Implements class SampleItem.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/SampleItem.h"
#include "Base/Util/Vec.h"
#include "GUI/Model/Sample/LayerItem.h"
#include "GUI/Model/Type/PredefinedColors.h"
#include "GUI/Model/Util/Backup.h"
#include "GUI/Model/Util/UtilXML.h"

namespace {
namespace Tag {

const QString Name("Name");
const QString Description("Description");
const QString CrossCorrelationLength("CrossCorrelationLength");
const QString MaterialsSet("MaterialsSet");
const QString Layer("Layer");
const QString ExternalField("ExternalField");
const QString ExpandInfoGroupbox("ExpandInfoGroupbox");

} // namespace Tag
} // namespace

SampleItem::SampleItem()
    : NamedItem("Sample")
{
    m_cross_correlation_length.init("Cross-correlation length (nm)",
                                    "Cross correlation length of roughnesses between interfaces",
                                    0.0, 5, RealLimits::nonnegative(), "cross");
    m_external_field.init("External field", "External field (A/m)", "extField");
}

SampleItem::~SampleItem() = default;

SampleItem* SampleItem::clone() const
{
    auto* result = new SampleItem;
    GUI::Util::copyContents(this, result);
    return result;
}

std::vector<ItemWithMaterial*> SampleItem::itemsWithMaterial() const
{
    std::vector<ItemWithMaterial*> result;
    for (LayerItem* layer : m_layers)
        Vec::concat(result, layer->itemsWithMaterial());
    return result;
}

void SampleItem::addStandardMaterials()
{
    // add only non-existing materials
    QString name = materialMap.key(DefaultMaterials::Default);
    if (!m_materials.materialItemFromName(name))
        m_materials.addRefractiveMaterialItem(name, 1e-3, 1e-5);

    name = materialMap.key(DefaultMaterials::Vacuum);
    if (!m_materials.materialItemFromName(name))
        m_materials.addRefractiveMaterialItem(name, 0.0, 0.0);

    name = materialMap.key(DefaultMaterials::Particle);
    if (!m_materials.materialItemFromName(name))
        m_materials.addRefractiveMaterialItem(name, 6e-4, 2e-8);

    name = materialMap.key(DefaultMaterials::Core);
    if (!m_materials.materialItemFromName(name))
        m_materials.addRefractiveMaterialItem(name, 2e-4, 1e-8);

    name = materialMap.key(DefaultMaterials::Substrate);
    if (!m_materials.materialItemFromName(name))
        m_materials.addRefractiveMaterialItem(name, 6e-6, 2e-8);
}

LayerItem* SampleItem::createLayerItemAt(int index)
{
    if (index < 0)
        index = m_layers.size();

    auto* layer = new LayerItem(&m_materials);
    m_layers.insert_at(index, layer);
    updateTopBottom();
    return layer;
}

void SampleItem::updateTopBottom()
{
    for (LayerItem* l : m_layers) {
        l->setIsTopLayer(l == m_layers.front());
        l->setIsBottomLayer(l == m_layers.back());
    }
}

void SampleItem::removeLayer(LayerItem* layer)
{
    m_layers.delete_element(layer);
    updateTopBottom();
}

void SampleItem::moveLayer(LayerItem* layer, LayerItem* aboveThisLayer)
{
    if (layer == aboveThisLayer)
        return;

    int currentIndex = Vec::indexOfPtr(layer, layerItems());
    m_layers.release_at(currentIndex);

    int destIndex = m_layers.size();
    if (aboveThisLayer != nullptr)
        destIndex = Vec::indexOfPtr(aboveThisLayer, layerItems());
    m_layers.insert_at(destIndex, layer);

    updateTopBottom();
}

void SampleItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeTaggedValue(w, Tag::Name, name());
    XML::writeTaggedValue(w, Tag::Description, description());
    m_cross_correlation_length.writeTo2(w, Tag::CrossCorrelationLength);
    XML::writeTaggedElement(w, Tag::ExternalField, m_external_field);
    XML::writeTaggedElement(w, Tag::MaterialsSet, m_materials);
    for (const auto* layer : m_layers)
        XML::writeTaggedElement(w, Tag::Layer, *layer);

    XML::writeTaggedValue(w, Tag::ExpandInfoGroupbox, expandInfo);
}

void SampleItem::readFrom(QXmlStreamReader* r)
{
    m_layers.clear();

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();
        if (tag == Tag::Name)
            setName(XML::readTaggedString(r, tag));
        else if (tag == Tag::Description)
            setDescription(XML::readTaggedString(r, tag));
        else if (tag == Tag::CrossCorrelationLength)
            m_cross_correlation_length.readFrom2(r, tag);
        else if (tag == Tag::ExternalField)
            XML::readTaggedElement(r, tag, m_external_field);
        else if (tag == Tag::MaterialsSet)
            XML::readTaggedElement(r, tag, m_materials);
        else if (tag == Tag::Layer)
            XML::readTaggedElement(r, tag, *createLayerItemAt());
        else if (tag == Tag::ExpandInfoGroupbox)
            expandInfo = XML::readTaggedBool(r, tag);
        else
            r->skipCurrentElement();
    }
}

void SampleItem::updateDefaultLayerColors()
{
    const auto& colors = GUI::Colors::layerDefaults();

    int col = 0;
    for (auto* l : layerItems()) {
        if (!l->color().isValid())
            l->setColor(colors[col]);
        col = (col + 1) % colors.size();
    }
}
