//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sample/Interface/RoughnessModels.cpp
//! @brief     Implements RoughnessModels enumerator and Python wrapper.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Sample/Interface/RoughnessModels.h"
#include "Base/Math/Functions.h"
#include "Base/Util/Assert.h"
#include <cmath>
#include <numbers>

using std::numbers::pi;

namespace {
const double tanh_prefactor = pi / 2. / std::sqrt(3);
const double erf_prefactor = std::sqrt(1.0 / 2.0);
const double pi_2 = std::sqrt(pi);
} // namespace


ErfRoughness::ErfRoughness()
    : RoughnessModel()
{
}

double ErfRoughness::transient(double x, double sigma) const
{
    ASSERT(sigma >= 0);
    if (sigma == 0.0)
        return Math::Heaviside(x);

    return (1.0 + std::erf(erf_prefactor * x / sigma)) / 2.0;
}

double ErfRoughness::distribution(double x, double sigma) const
{
    ASSERT(sigma > 0);
    return erf_prefactor / pi_2 / sigma * std::exp(-pow(erf_prefactor * x / sigma, 2));
}

double ErfRoughness::sigmaRange() const
{
    return 6.;
}

ErfRoughness* ErfRoughness::clone() const
{
    return new ErfRoughness;
}

TanhRoughness::TanhRoughness()
    : RoughnessModel()
{
}

double TanhRoughness::transient(double x, double sigma) const
{
    ASSERT(sigma >= 0);
    if (sigma == 0.0)
        return Math::Heaviside(x);

    return (1.0 + std::tanh(tanh_prefactor * x / sigma)) / 2.0;
}

double TanhRoughness::distribution(double x, double sigma) const
{
    ASSERT(sigma > 0);
    return tanh_prefactor / 2 / sigma / pow(std::cosh(tanh_prefactor * x / sigma), 2);
}

double TanhRoughness::sigmaRange() const
{
    return 11.;
}

TanhRoughness* TanhRoughness::clone() const
{
    return new TanhRoughness;
}
