(**************************************************************************)
(*                                                                        *)
(*  Copyright (C) 2012 Johannes 'josch' Schauer <j.schauer@email.de>      *)
(*  Copyright (C) 2012 Pietro Abate <pietro.abate@pps.jussieu.fr>         *)
(*                                                                        *)
(*  This library is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU Lesser General Public License as        *)
(*  published by the Free Software Foundation, either version 3 of the    *)
(*  License, or (at your option) any later version.  A special linking    *)
(*  exception to the GNU Lesser General Public License applies to this    *)
(*  library, see the COPYING file for more information.                   *)
(**************************************************************************)

open ExtLib
open Common
open Debian
open Datatypes_t
open DoseparseNoRpm

include Util.Logging(struct let label = __FILE__ end) ;;

let str_list_option ?(default=Some []) ?(metavar = "STRLST") =
  let sep = "," in
  let coerce s = ExtString.String.nsplit s sep in
  fun () ->
    OptParse.Opt.value_option metavar default coerce
    (fun _ s -> Printf.sprintf "Invalid String '%s'" s)

module StringSet = BootstrapCommon.StringSet
module IntSet = BootstrapCommon.IntSet

module Options = struct
  open OptParse
  let usage = "%prog [options] --deb-native-arch=ARCH buildgraph srcgraph Packages Sources"
  let description = "output statistics in JSON format"
  let options = OptParser.make ~description ~ usage
  include BootstrapCommon.MakeOptions(struct let options = options end)

  let cycle_length = StdOpt.int_option ~default:2 ()
  let cycle_length_fas = StdOpt.int_option ~default:8 ()
  let remove_weak = StdOpt.str_option ()
  let remove_reduced = str_list_option ()
  let sapsb = StdOpt.store_true ()
  let allowsrcmismatch = StdOpt.store_true ()
  let available = StdOpt.str_option ()
  let noindep = StdOpt.store_false ()

  open OptParser

  let prog_group = add_group options "Program specific options" in

  add options ~group:prog_group ~long_name:"max-length" ~help:"maximum length of found cycles (default=2)" cycle_length;
  add options ~group:prog_group ~long_name:"max-length-fas" ~help:"maximum length of found cycles for fas search (default=4)" cycle_length_fas;
  add options ~group:prog_group ~long_name:"remove-weak" ~help:"path to list of weak build dependencies" remove_weak;
  add options ~group:prog_group ~long_name:"sapsb" ~help:"calculate strong articulation points and strong bridges" sapsb;
  add options ~group:prog_group ~long_name:"remove-reduced" ~help:"remove droppable build dependencies supplied by comma separated list of reduced dep files" remove_reduced;
  add options ~group:prog_group ~short_name:'A' ~long_name:"available"
    ~help:"List of available packages (arch:all, crossed...) in control file format" available;
  add options ~group:prog_group ~long_name:"keep-indep" ~help:"Do not drop Build-Depends-Indep dependencies" noindep;

  add options ~group:prog_group ~long_name:"allowsrcmismatch" ~help:("If a binary package is "^
    "without a source package but there is a source package of same name but "^ 
    "different version, match this binary package to that source package.") allowsrcmismatch;

  include StdOptions.InputOptions;;
  let default = List.filter (fun e -> not (List.mem e ["checkonly"; "latest";"outfile"])) StdOptions.InputOptions.default_options in
  StdOptions.InputOptions.add_options ~default options ;;

  include StdOptions.DistribOptions;;
  let default = List.filter (fun e -> not (List.mem e ["deb-ignore-essential"; "inputtype"])) StdOptions.DistribOptions.default_options in
  StdOptions.DistribOptions.add_options ~default options ;;

end

let main () =
  let posargs = OptParse.OptParser.parse_argv Options.options in
  StdDebug.enable_debug (OptParse.Opt.get Options.verbose);
  Util.Debug.disable "Depsolver_int";
  Util.Warning.disable "Sources"; (* disable MismatchSrc warnings as exception is caught *)
  StdDebug.all_quiet (OptParse.Opt.get Options.quiet);
  let maxlengthfas = OptParse.Opt.get Options.cycle_length_fas in
  let maxlength = OptParse.Opt.get Options.cycle_length in
  let reduced_deps_files = OptParse.Opt.get Options.remove_reduced in
  let allowsrcmismatch = OptParse.Opt.get Options.allowsrcmismatch in
  let noindep = OptParse.Opt.get Options.noindep in
  let options = Options.set_deb_options () in
  let buildarch = Option.get options.Debian.Debcudf.native in
  let hostarch = match options.Debian.Debcudf.host with None -> "" | Some s -> s in
  let foreignarchs = options.Debian.Debcudf.foreign in

  let bgf, sgf, posargs = match posargs with
   | bgf::sgf::l -> bgf,sgf,l
   | _ -> fatal "you must provide buildgraph, srcgraph, Packages and Sources"
  in

  let (binlist, (fgsrclist,bgsrclist), _) = BootstrapCommon.parse_packages ~noindep Options.parse_cmdline buildarch hostarch foreignarchs posargs in

  let tables = Debian.Debcudf.init_tables (fgsrclist@bgsrclist@binlist) in
  let fgsl = List.map (Debian.Debcudf.tocudf ~options tables) fgsrclist in
  let bgsl = List.map (Debian.Debcudf.tocudf ~options tables) bgsrclist in
  let pkglist = List.map (Debian.Debcudf.tocudf ~options tables) binlist in
  let universe = Cudf.load_universe (BootstrapCommon.unique [pkglist;fgsl;bgsl]) in

  let module BG = BuildGraph.G in
  let module SG = SrcGraph.G in

  let ic = open_in bgf in
  let bg = BuildGraph.from_ic universe ic in
  close_in ic;

  let ic = open_in sgf in
  let sg = SrcGraph.from_ic universe ic in
  close_in ic;

  (* read package list for available packages *)
  let availableset =
    if OptParse.Opt.is_set Options.available then
      BootstrapCommon.read_package_file ~archs:(buildarch::hostarch::foreignarchs) (Debcudf.tocudf ~options tables) (OptParse.Opt.get    Options.available)
    else CudfAdd.Cudf_set.empty
  in

  let module BGE = BuildGraphExtras.Make(struct let univ = universe end) in
  let module BGS = BuildGraphStats.Make(struct let univ = universe end) in
  let module SGE = SrcGraphExtras.Make(struct let univ = universe end) in
  let module SGS = SrcGraphStats.Make(struct let univ = universe end) in

  let binset = BootstrapCommon.get_bin_packages (BootstrapCommon.srcbin_table ~available:availableset ~allowmismatch:allowsrcmismatch universe) in

  let type1, type2, type3 = SGS.self_cycles binset sg in

  let srcpkglist = BGE.srcpkglist_of_g bg in
  let weak_file =
    if OptParse.Opt.is_set Options.remove_weak then
      OptParse.Opt.get Options.remove_weak
    else
      ""
  in
  let reduced_deps_ht, weak_deps_set = BootstrapCommon.get_reduced_deps_ht ~weak_file (OptParse.Opt.is_set Options.remove_weak) (buildarch::foreignarchs) srcpkglist reduced_deps_files in
  BGE.remove_build_deps reduced_deps_ht bg;

  let scc = List.filter_map (function [] | [_] -> None | s -> Some (BuildGraph.Oper.subgraph bg s)) (BuildGraph.Comp.scc_list bg) in

  let tnvt pkg = ((CudfAdd.decode pkg.Cudf.package), (CudfAdd.string_of_version pkg)) in
  let tnvtl = List.map tnvt in

  let result = {
    srcgraph = {
      snr_vertex = SG.nb_vertex sg;
      snr_edges = SG.nb_edges sg;
      type1cycles = List.map (fun (pkg, d1, d2) -> (tnvt pkg, tnvtl d1, tnvtl d2)) type1;
      type2cycles = List.map (fun (pkg, d1, d2) -> (tnvt pkg, tnvtl d1, tnvtl d2)) type2;
      type3cycles = List.map (fun (pkg, deps) -> (tnvt pkg, tnvtl deps)) type3;
    };
    buildgraph = {
      bnr_vertex = BG.nb_vertex bg;
      bnr_edges = BG.nb_edges bg;
      sccs = List.map (fun sg ->
        let cycles, cycleedge = if maxlength > 0 then begin
          let cycles = BuildGraph.Cycles.johnson ~maxlength sg in
          cycles, (BGS.edges_in_most_cycles sg cycles)
        end else [], [] in
        let fas = if maxlengthfas > 0 then begin
          let ht_fas = Hashtbl.create (BG.nb_edges sg) in
          BuildGraph.EdgeSet.iter (fun (src,_,pkg) ->
            Hashtbl.replace ht_fas src (pkg::(Hashtbl.find_default ht_fas src []))
          ) (BGE.calculate_fas ~maxlength:maxlengthfas sg);
          Hashtbl.fold (fun k v acc -> (k,v)::acc) ht_fas []
        end else [] in
        let sap, sb = if OptParse.Opt.get Options.sapsb then begin
          let sap = BuildGraph.Utils.find_strong_articulation_points sg in
          let sb = BuildGraph.Utils.find_strong_bridges sg in
          sap, sb
        end else [], [] in
        {
          sg_vertex = BG.nb_vertex sg;
          sg_edges = BG.nb_edges sg;
          cycles = List.map BGE.variantlist_of_vlist cycles;
          cycleedge = List.map (fun ((v1,_,v2),c) -> ((BGE.variant_of_vertex v1), (BGE.variant_of_vertex v2)), c) cycleedge;
          builddeps = List.map (fun (p,deps) -> tnvt p, tnvtl deps) (BGS.min_builddep sg);
          ratio_source = List.map (fun (s,b,n,o) -> tnvt s, b, tnvtl n, tnvtl o) (BGS.ratio_source sg);
          ratio_binary = List.map (fun (b,s,o) -> tnvt b, s, tnvtl o) (BGS.ratio_binary sg);
          weak = List.map (fun (pkg, deps) -> tnvt pkg, tnvtl deps) (BGS.only_weak_missing weak_deps_set sg);
          srcbinstats = List.map (fun (p,(a,b)) -> tnvt p, a, b) (BGS.get_src_bin_stats sg);
          fas = List.map (fun (p,deps) -> tnvt (BGE.pkg_of_vertex p), tnvtl (BGE.pkglist_of_vlist deps)) fas;
          sap = List.map (fun (p,i) -> (BGE.variant_of_vertex p), i) sap;
          sb = List.map (fun ((v1,_,v2),i) -> (BGE.variant_of_vertex v1, BGE.variant_of_vertex v2), i) sb;
        }
      ) scc;
    };
  } in
  
  (*print_endline (Yojson.Safe.prettify (Datatypes_j.string_of_stats result));*)
  print_endline (Datatypes_j.string_of_stats result);
;;

main ();;
