#ifndef _EVENT_H_
#define _EVENT_H_

#include "atom.h"
#include "position.h"

/**
 * Instances of classes derived from the abstract class Event are called events.
 * Events are atoms with a well defined position and duration. The position is specified by the Position class, containing basically a long int
 * and a lot of methods converting between the linear and the musical presentation of the position:
 * The linear position presents the total number of ticks by one number.
 * The musical presentation is made of the numbers denoting the bar, the beat and the number of ticks within that beat. E.g. 3.1.0 denotes the
 * first beat (the beginning) of the third bar.
 **/
class Event : public Atoma
{
 protected:
  Position _pos;
  long     _duration;

 public:
  /**
   * Default constructor
  **/
  Event();

  /**
   * Constructor used by copy()
  **/
  Event(const Event&);

  /**
   * Constructor at a position
  **/
  Event(Position);

  /**
   * Constructor with position and duration
  **/
  Event(Position,long);

  /**
   * Returns the start position (relative to the part) of an event. For the absolute start position of an event ev (within the track) use
   * the part's method
   * <pre>
   *
   * Position pos = part->start(ev);
   * </pre>
   * The following is always true:
   * <pre>
   *
   * (part->start(ev) == part->start() + ev->internalStart())
   * </pre>
  **/
  const Position & internalStart() const { return _pos; }

  /**
   * Returns the end position (relative to the part) of an event. For the absolute end position of an event ev (within the track) use
   * the part's method
   * <pre>
   *
   * Position pos = part->end(ev);
   * </pre>
   * The following is always true:
   * <pre>
   *
   * (part->end(ev) == part->start() + ev->internalEnd())
   * </pre>
  **/
  long internalEnd() const;

  /**
   * Returns the duration of an event.
  **/
  long duration() const { return _duration; }

  /**
   * Sets the start position within the part. To set the absolute position pos of an event ev, use the part's method:
   * <pre>
   *
   * part->setStart(ev, pos);
   * </pre>
  **/
  void setInternalStart(Position);

  /**
   * Quantizes the position, by making use of the positions snap() method.
  **/
  void snapStart(int i);

  /**
   * Sets the duration of an event.
  **/
  void setDuration(long dur);

  /**
   * Print has to be implemented by all classes inheriting from Event.
  **/
  virtual ostream & print(int,ostream&) const = 0;

  /**
   * Flush has to be implemented by all classes inheriting from Event.
  **/
  virtual void flush(const char*) const = 0;

  /**
   * Copy has to be implemented by all classes inheriting from Event.
  **/
  virtual Element * copy() const = 0;

  /**
   * Objects of classes derived from this are events
   **/
  virtual bool isEvent() const { return true; }

  /**
   * If new events are implemented (within a new addon), this may be implemented
   * in order to be shown in the Info box of the Score editor (other editors to follow)
   **/
  virtual const char * value();
};

#endif
