
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nickycopeland@hotmail.com> 1996,2008
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*#define DEBUG */

#include <pthread.h>
#include "bristol.h"

int midiHandle, controlHandle;
bristolMidiMsg midiMsg;

extern int midiExitReq;

extern int midiCheck();
extern void initMidiRoutines();

bristolMidiHandler bristolMidiRoutines;

int
midiMsgHandler(bristolMidiMsg *msg, audioMain *audiomain)
{
	int event = (msg->command & ~MIDI_STATUS_MASK) >> 4;
#ifdef DEBUG
	printf("midiMsgHandler(%x, %x)\n", msg, audiomain);
	printMidiMsg(msg);
#endif

	/*
	 * Depending on the message type, handle the message. If this is a note
	 * event, channel or poly pressure then we will have to apply the value
	 * mapping here, as early as possible.
	 */
	switch(event) {
		case 0:
		case 1:
			/* 
			 * This does key mapping as well, but note on and off events use
			 * the same table, otherwise we would have issues. The key mapping
			 * table defaults to linear (1 to 1) and takes the System map as
			 * we will not be remapping system events.
			 */
			msg->params.key.key
				= bristolMidiRoutines.bmr[7].map[msg->params.key.key];
			msg->params.key.velocity
				= bristolMidiRoutines.bmr[event].map[msg->params.key.velocity];
			break;
		case 2:
			msg->params.pressure.pressure
				= bristolMidiRoutines.bmr[event].map[msg->params.pressure.pressure];
			break;
		case 5:
			msg->params.channelpress.pressure
				= bristolMidiRoutines.bmr[event].map[
					msg->params.channelpress.pressure];
			break;
	}

	bristolMidiRoutines.bmr[(msg->command & ~MIDI_STATUS_MASK) >> 4].callback
		(audiomain, msg);

	return(0);
}

extern int exitReq;
static char *bSMD = "128.1";

void *
midiThread(audioMain *audiomain)
{
	int flags = BRISTOL_CONN_MIDI, exitstatus = -1, i;
#if (BRISTOL_HAS_ALSA == 1)
	char *device = bAMD;
#else
	char *device = bOMD;
#endif

#ifdef DEBUG
	printf("starting midi thread\n");
#endif

	audiomain->mtStatus = BRISTOL_WAIT;

	initMidiRoutines(audiomain, bristolMidiRoutines.bmr);
	/*
	 * bmr[7].floatmap may now contain a microtonal map, we should converge this
	 * onto the bmr.freq table.
	 */
	for (i = 0; i < 128; i++)
	{
		if (bristolMidiRoutines.bmr[7].floatmap[i] == 0)
			continue;

		bristolMidiRoutines.freq[i].freq
			= bristolMidiRoutines.bmr[7].floatmap[i];
		bristolMidiRoutines.freq[i].step = ((float) BRISTOL_BUFSIZE) /
			(bristolMidiRoutines.bmr[7].floatmap[i] * audiomain->samplerate);
	}

	/*
	 * Get a couple of semaphores to separate the midi from the audio 
	 * processing. These are destroyed in the inthandler() in bristol.c as
	 * the midi signalling is closed down. The semaphores are shared in the
	 * note on/off code in midinote.c and the audioEngine.c which need to 
	 * track the same voicelists.
	 *
	 * The semaphores are only for note events. There are two lockout sections
	 * to minimise delays for most key events. That is not possible for all
	 * events, see midinote.c for details, it regards voice starvation.
	 */
	if (sem_init(&audiomain->sem_long, 0, 1) < 0)
	{
		printf("could not get the long semaphore\n");
		audiomain->atReq = BRISTOL_REQSTOP;
		exitReq = 1;
		pthread_exit(&exitstatus);
	}
	if (sem_init(&audiomain->sem_short, 0, 1) < 0)
	{
		printf("could not get the short semaphore\n");
		audiomain->atReq = BRISTOL_REQSTOP;
		exitReq = 1;
		pthread_exit(&exitstatus);
	}

	if ((audiomain->flags & BRISTOL_MIDIMASK) == BRISTOL_MIDI_OSS)
	{
printf("midi oss\n");
		flags = BRISTOL_CONN_OSSMIDI;
	}

	if ((audiomain->flags & BRISTOL_MIDIMASK) == BRISTOL_MIDI_SEQ)
	{
printf("midi sequencer\n");
		flags = BRISTOL_CONN_SEQ;
		device = bSMD;
		device = "bristol";
	} else if (audiomain->mididev != (char *) NULL)
		device = audiomain->mididev;

	if ((audiomain->flags & BRISTOL_MIDIMASK) == BRISTOL_MIDI_JACK)
	{
		device = bSMD;
printf("midi jack: %s\n",device);
		flags = BRISTOL_CONN_SEQ;
	}

	/*
	 * Get a control port.
	 */
	if ((controlHandle = bristolMidiOpen("localhost",
		BRISTOL_DUPLEX|BRISTOL_CONN_TCP|BRISTOL_CONN_PASSIVE|
			(audiomain->flags & BRISTOL_MIDI_WAIT),
			audiomain->port,
		BRISTOL_REQ_SYSEX, midiMsgHandler, audiomain)) < 0)
	{
		printf("Error opening control device, exiting midi thread\n");
		audiomain->atReq = BRISTOL_REQSTOP;
		exitReq = 1;
		pthread_exit(&exitstatus);
	}

printf("midiOpen: %s(%x)\n", device, flags);
	if ((midiHandle = bristolMidiOpen(device, flags|BRISTOL_DUPLEX,
		-1, BRISTOL_REQ_NSX, midiMsgHandler, audiomain)) < 0)
	{
		printf("Error opening midi device %s, exiting midi thread\n", device);
		audiomain->atReq = BRISTOL_REQSTOP;
		exitReq = 1;
		bristolMidiClose(controlHandle);
		pthread_exit(&exitstatus);
	}

#ifdef DEBUG
	printf("opened midi device\n");
#endif
printf("opened midi device: %i/%i\n", controlHandle, midiHandle);

	audiomain->mtStatus = BRISTOL_OK;

	/*
	 * This will become blocking midi code, now that we are threaded. In short,
	 * midiCheck should not return. This can be an issue to ensure that the
	 * midi device is closed.
	 */
	midiCheck();

	bristolMidiClose(midiHandle);
	bristolMidiClose(controlHandle);

	printf("midiThread exiting\n");

	exitstatus = 0;
	pthread_exit(&exitstatus);
}

void
printMidiMsg(bristolMidiMsg *msg)
{
	if (msg->command == 0x00f0) {
		printf("	handle:	%i\n", msg->midiHandle);
		printf("	command:	SYSEX\n");
	} else {
		printf("	handle:	%i\n", msg->midiHandle);
		printf("	chan:	%i\n", msg->channel);
		printf("	command:	%x\n", msg->command);
		printf("	p1:	%i, p2:	%i\n",
			msg->params.key.key, msg->params.key.velocity);
	}
}

