/********************************************************************
 * Copyright (C) 2005 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BSCommander; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/
 
/*------- include files:
-------------------------------------------------------------------*/
#include "ConfigFnBtn.h"
#include "Shared.h"
#include "Config.h"
#include "Settings.h"
#include <qlayout.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qfontdialog.h>

/*------- local constants:
-------------------------------------------------------------------*/
const QString ConfigFnBtn::DescriptionDispMode = QT_TR_NOOP( "Description display mode" );
const QString ConfigFnBtn::OnlyFunctionKey     = QT_TR_NOOP( "&Only function key" );
const QString ConfigFnBtn::F11                 = QT_TR_NOOP( "F11" );
const QString ConfigFnBtn::OnlyDescription     = QT_TR_NOOP( "Only &description" );
const QString ConfigFnBtn::Description         = QT_TR_NOOP( "description" );
const QString ConfigFnBtn::FullDescription     = QT_TR_NOOP( "&Full description" );
const QString ConfigFnBtn::F11Description      = QT_TR_NOOP( "F11 description" );
const QString ConfigFnBtn::TwoLinesDescription = QT_TR_NOOP( "&Two line description" );
const QString ConfigFnBtn::F11NlDescription    = QT_TR_NOOP( "F11\ndescription" );
const QString ConfigFnBtn::ButtonPreview       = QT_TR_NOOP( "Button preview" );
const QString ConfigFnBtn::Font                = QT_TR_NOOP( "Font" );


//*******************************************************************
// ConfigFnBtn                                           CONSTRUCTOR
//*******************************************************************
ConfigFnBtn::ConfigFnBtn( QWidget* const in_parent )
: QFrame( in_parent )
, d_main_layout( new QHBoxLayout( this ))
, d_lft_layout( new QVBoxLayout )
, d_rgt_layout( new QVBoxLayout )
, d_radio_grp( new QButtonGroup( 2, Qt::Horizontal, tr( DescriptionDispMode ), this ))
, d_only_fn_lbl( new QLabel( tr( F11 ), d_radio_grp ))
, d_only_fn_rb( new QRadioButton( tr( OnlyFunctionKey ), d_radio_grp ))
, d_only_text_lbl( new QLabel( tr( Description ), d_radio_grp ))
, d_only_text_rb( new QRadioButton( tr( OnlyDescription ), d_radio_grp ))
, d_fn_text_lbl( new QLabel( tr( F11Description ), d_radio_grp ))
, d_fn_text_rb( new QRadioButton( tr( FullDescription ), d_radio_grp ))
, d_fn_nl_text_lbl( new QLabel( tr( F11NlDescription ), d_radio_grp ))
, d_fn_nl_text_rb( new QRadioButton( tr( TwoLinesDescription ), d_radio_grp ))
, d_example_gb( new QGroupBox( 1, Qt::Horizontal, tr( ButtonPreview ), this ))
, d_example_btn( new QPushButton( d_example_gb ))
, d_font_btn( new QPushButton( tr( Font ), this ))
, d_apply_btn( new QPushButton( tr(Shared::ApplyBtnLabel), this ))
{
	Shared::add_icon( d_font_btn, "fonts.png" );
	Shared::add_icon( d_apply_btn, Shared::ApplyIcon );
	
	d_lft_layout->addWidget( d_radio_grp );
	d_rgt_layout->addWidget( d_example_gb );
	d_rgt_layout->addStretch( Shared::OverStretch );
	d_rgt_layout->addWidget( d_font_btn );
	d_rgt_layout->addWidget( d_apply_btn );

	d_main_layout->setSpacing( Shared::LayoutSpacing );
	d_main_layout->setMargin( Shared::LayoutMargin );
	d_main_layout->addLayout( d_lft_layout );
	d_main_layout->addLayout( d_rgt_layout );
	
	connect( d_font_btn , SIGNAL( clicked()      ), this, SLOT( font_selection() ));
	connect( d_apply_btn, SIGNAL( clicked()      ), this, SLOT( apply() ));
	connect( d_radio_grp, SIGNAL( clicked( int ) ), this, SLOT( clicked( int )   ));
	connect( this       , SIGNAL( looks_refresh()), in_parent, SIGNAL( looks_refresh() ));
}

//*******************************************************************
// show                                            PRIVATE inherited
//*******************************************************************
void ConfigFnBtn::show()
{
	QFrame::show();
	read_config();
}
// end of show

//*******************************************************************
// read config                                               PRIVATE
//*******************************************************************
void ConfigFnBtn::read_config()
{
	Config::instance()->refresh();

	d_font = Config::instance()->fn_font();
	d_fn_disp_mode = Config::instance()->fn_disp_mode();
	d_radio_grp->setButton( d_fn_disp_mode );

	update_view();
}
// end of read_config

//*******************************************************************
// save config                                               PRIVATE
//*******************************************************************
void ConfigFnBtn::save_config()
{
	Settings::save( Config::AppConfigGroup, Config::FnDispModeKey, d_fn_disp_mode );
	Settings::save( Config::AppConfigGroup, Config::FnFontKey, d_font.toString() );
}
// end of save_config

//*******************************************************************
// update_view                                               PRIVATE
//*******************************************************************
void ConfigFnBtn::update_view()
{
	QString button_label;
	switch( d_fn_disp_mode ) {
		case Config::ONLY_FN:
			button_label = F11;
			break;
		case Config::ONLY_TEXT:
			button_label = Description;
			break;
		case Config::FULL_DESC:
			button_label = F11Description;
			break;
		case Config::TWO_LINES:
			button_label = F11NlDescription;
			break;
	}
	d_example_btn->setFont( d_font );
	d_example_btn->setText( tr( button_label ) );
}
// end of update_view

//*******************************************************************
// font_selection                                       PRIVATE slot
//-------------------------------------------------------------------
// Uzytkownik nacisnal przycisk zmiany fontu.
//*******************************************************************
void ConfigFnBtn::font_selection()
{
	bool ok;
	const QFont new_font = QFontDialog::getFont( &ok, d_font, this );
	if ( ok ) {
		if( new_font != d_font ) {
			d_font = new_font;
			update_view();
		}
	}
}
// end of font_selection

//*******************************************************************
// apply                                                PRIVATE slot
//*******************************************************************
void ConfigFnBtn::apply()
{
	d_fn_disp_mode = d_radio_grp->selectedId();
	save_config();
	Config::instance()->refresh();
	emit looks_refresh();
}
// end of apply

//*******************************************************************
// clicked                                              PRIVATE slot
//-------------------------------------------------------------------
// Uzytkownik wybral 'radio button'.
//*******************************************************************
void ConfigFnBtn::clicked( const int in_idx )
{
	d_fn_disp_mode = in_idx;
	update_view();
}
// end of clicked
