/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/
 
/*------- include files:
-------------------------------------------------------------------*/
#include "DirSynch.h"
#include "CmpFiles.h"
#include "Shared.h"
#include <qmessagebox.h>

/*------- local constants:
-------------------------------------------------------------------*/
const QString DirSynch::Caption          = QT_TR_NOOP( "Directories synchronisation" );
const QString DirSynch::InfoCaption      = QT_TR_NOOP( "Synchronised directories" );
const QString DirSynch::NotReadableFile  = QT_TR_NOOP( "You can't read from this file: %1.\nCheck permission access." );
const QString DirSynch::OpenForReadError = QT_TR_NOOP( "Can't open this file for reading: %1." );
const QString DirSynch::OpenForWriteError= QT_TR_NOOP( "Can't open this file for writing; %1." );


//*******************************************************************
// DirSynch                                              CONSTRUCTOR
//*******************************************************************
DirSynch::DirSynch( QWidget* const in_parent, const QString& in_src_dir, const QString& in_dst_dir )
: DirsDialog      ( in_parent, in_src_dir, in_dst_dir, Synchronize )
{
	setCaption( tr(Caption) );
	set_info_title( tr(InfoCaption) );

	connect( this, SIGNAL( file_do_it( const QString&, bool, bool, const QString&, const QString& ) ),
				this, SLOT  ( file_do_it( const QString&, bool, bool, const QString&, const QString& ) ));
	connect( this, SIGNAL( dir_do_it ( const QString&, bool, bool, const QString&, const QString& ) ),
				this, SLOT  ( dir_do_it ( const QString&, bool, bool, const QString&, const QString& ) ));
}
// end of DirSynch

//*******************************************************************
// dir_do_it                                            PRIVATE slot
//-------------------------------------------------------------------
// Funkcja jest wywolywana tylko i wylacznie jesli podkatalog o
// podanej nazwie nie istnieje we wskazanych katalogach.
// Nalezy nie istniejacy podkatalog utworzyc i zsynchronizowac.
//*******************************************************************
void DirSynch::dir_do_it(	const QString& in_name,
									const bool     in_lft_flag,
									const bool     in_rgt_flag,
									const QString& in_lft_dir,
									const QString& in_rgt_dir )
{
	Shared::idle();
	if( d_break ) return;
	
	const QString lft_path = in_lft_dir + "/" + in_name;
	const QString rgt_path = in_rgt_dir + "/" + in_name;
	
	QString lft_disp_name = "";
	QString rgt_disp_name = "";
	QString mark     = "";
	
	// ####### KOPIOWANIE Z LEWEJ NA PRAWO #######
	if( in_lft_flag && !in_rgt_flag ) {
		lft_disp_name = Shared::get_subpath( d_src_dir, lft_path ).remove( 0, 1 );
		mark = DirMark + " " + L2R;
		//.............................................
		if( rgt_is_blocked() ) {
			rgt_disp_name = tr(BlockMark);
		}
		else {
			if( create_dir( lft_path, in_rgt_dir  ) ) {
				rgt_disp_name = tr(CopyMark);
				scanning( lft_path, rgt_path );
			}
			else {
				rgt_disp_name = tr(ErrorMark);
			}
		}
		//.............................................
		add_item( lft_disp_name, mark, rgt_disp_name, lft_path, rgt_path );
	}
	// ####### KOPIOWANIE Z PRAWEJ NA LEWO #######
	else if( !in_lft_flag && in_rgt_flag ) {
		rgt_disp_name = Shared::get_subpath( d_dst_dir, rgt_path ).remove( 0, 1 );
		mark = R2L + "DIR";
		//.............................................
		if( lft_is_blocked() ) {
			lft_disp_name = tr(BlockMark);
		}
		else {
			if( create_dir( rgt_path, in_lft_dir ) ) {
				lft_disp_name =  tr(CopyMark);
				scanning( lft_path, rgt_path );
			}
			else {
				lft_disp_name = tr(ErrorMark);
			}
		}
		//.............................................
		add_item( lft_disp_name, mark, rgt_disp_name, lft_path, rgt_path );
	}	
}
// end of dir_do_it

//*******************************************************************
// create_dir                                                PRIVATE
//-------------------------------------------------------------------
// Tworzy brakujacy podkatalog.
//*******************************************************************
bool DirSynch::create_dir( const QString& in_src_dir, const QString& in_dst_dir )
{
	Shared::idle();
	if( d_break ) return FALSE;
	
	bool retval = FALSE;
	
	const QFileInfo fi( in_src_dir );
	if( Shared::is_regular_file( fi.fileName() ) ) {
		if( fi.isDir() ) {
			const QDir dir( in_src_dir );
			if( dir.isReadable() ) {
				QDir dst( in_dst_dir );
				dst.mkdir( fi.fileName(), FALSE );
				retval = dst.cd( fi.fileName(), FALSE );
			}
		}
	}

	return retval;
}
// end of create_dir

//*******************************************************************
// file_do_it                                           PRIVATE slot
//*******************************************************************
void DirSynch::file_do_it( const QString& in_name,
									const bool     in_lft_flag,
									const bool     in_rgt_flag,
									const QString& in_lft_dir,
									const QString& in_rgt_dir )
{
	Shared::idle();
	if( d_break ) return;
	
	const QString lft_path = in_lft_dir + "/" + in_name;
	const QString rgt_path = in_rgt_dir + "/" + in_name;
	
	QString mark     = "";
	QString lft_name = "";
	QString rgt_name = "";
	QString lft_disp_name = "";
	QString rgt_disp_name = "";

	// ####### USTALAMY KIERUNEK KOPIOWANIA #######

	// ------- oba pliki istnieja - sprawdz czy sa takie same -------
	if( in_lft_flag && in_rgt_flag ) {
		CmpFiles cmp( this, lft_path, rgt_path );
		if( CmpFiles::NotEqual == cmp.compare() ) {
			// Pliki sie roznia zawartoscia - sa rozne
			const QFileInfo lfi( lft_path );
			const QFileInfo rfi( rgt_path );
			if     ( lfi.lastModified() == rfi.lastModified() ) mark = QuestMark;
			else if( lfi.lastModified() >  rfi.lastModified() ) mark = L2R;
			else                                                mark = R2L;
		}
		else {
			// Pliki maja taka sama zawartosc - sa takie same
			mark = Equal;
		}
	}
	// ------- kopiuj z lewej na prawo ------
	else if( in_lft_flag ) {
		mark = L2R;
	}
	// ------- kopiuj z prawej na lewo -------
	else if( in_rgt_flag ) {
		mark = R2L;
	}

	// ####### KOPIOWANIE (JESLI NIE MA BLOKADY) #######
	
	if( L2R == mark ) {
		lft_disp_name = Shared::get_subpath( d_src_dir, lft_path ).remove( 0, 1 );
		if( rgt_is_blocked() ) {
			rgt_disp_name = tr(BlockMark);
		}
		else {
			rgt_disp_name = copy_file( lft_path, rgt_path ) ? tr(CopyMark) : tr(ErrorMark);
		}
	}
	else if( R2L == mark ) {
		rgt_disp_name = Shared::get_subpath( d_dst_dir, rgt_path ).remove( 0, 1 );
		if( lft_is_blocked() ) {
			lft_disp_name = tr(BlockMark);
		}
		else {
			lft_disp_name = copy_file( rgt_path, lft_path ) ? tr(CopyMark) : tr(ErrorMark);
		}
	}
	else {
		lft_disp_name = Shared::get_subpath( d_src_dir, lft_path ).remove( 0, 1 );
		rgt_disp_name = Shared::get_subpath( d_dst_dir, rgt_path ).remove( 0, 1 );
	}

	add_item( lft_disp_name, mark, rgt_disp_name, lft_path, rgt_path );
}
// end of file_do_it

//*******************************************************************
// copy_file                                                 PRIVATE
//*******************************************************************
bool DirSynch::copy_file( const QString& in_src_path, const QString& in_dst_path )
{
	Shared::idle();
	if( d_break ) return FALSE;
	
	bool result = FALSE;
	
	const QFileInfo fi( in_src_path );
	QFile in ( in_src_path );
	QFile out( in_dst_path );
	
	if( FALSE == fi.isReadable() ) {
		QMessageBox::warning( this, Shared::ProgramName, tr(NotReadableFile).arg( in_src_path ));
	}
	else {
		if( in.open( IO_ReadOnly ) ) {
			if( out.open( IO_WriteOnly | IO_Truncate ) ) {
				result = TRUE;
				while( ( FALSE == d_break ) && ( FALSE == in.atEnd() )) {
					const Q_LONG n = in.readBlock( d_buffer, BufferSize );
					if( -1 == n ) {
						result = FALSE;
						break;
					}
					out.writeBlock( d_buffer, n );
					Shared::idle();
				}
				out.close();
			}
			else {
				QMessageBox::warning( this, Shared::ProgramName, tr(OpenForWriteError).arg( in_dst_path ) );
			}
			in.close();
		}
		else {
			QMessageBox::warning( this, Shared::ProgramName, tr(OpenForReadError).arg( in_src_path ) );
		}
	}

	return result;
}
// end of copy_file
