/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "ViewTable.h"
#include "Shared.h"
#include "ViewTableItem.h"
#include "Config.h"
#include <qapplication.h>
#include <qdragobject.h> 
#include <qinputdialog.h>
#include <qregexp.h>
#include <qpainter.h>
#include <qcolor.h>
#include <qpopupmenu.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qheader.h>
#include <qurlinfo.h>
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const ViewTable::ColumnDef ViewTable::COLDEF[] =
{
	{ QT_TR_NOOP("Name")  , Qt::AlignLeft    },
	{ QT_TR_NOOP("Ext")   , Qt::AlignLeft    },
	{ QT_TR_NOOP("Access"), Qt::AlignHCenter },
	{ QT_TR_NOOP("Size")  , Qt::AlignRight    },
	{ QT_TR_NOOP("Date")  , Qt::AlignHCenter  },
	{ QT_TR_NOOP("Time")  , Qt::AlignHCenter  },
	{ QT_TR_NOOP("Owner") , Qt::AlignHCenter  },
	{ QT_TR_NOOP("Group") , Qt::AlignHCenter  }
};
//...................................................................
const QString ViewTable::FOLDER_ICON              = "folder.png";
const QString ViewTable::IFOLDER_ICON             = "ifolder.png";
const QString ViewTable::DRAG_DROP_ICON           = "drag.png";
const QString ViewTable::DRAG_DROP_SUBTYPE        = "text/copy";
const QString ViewTable::DRAG_DROP_CMD            = "copy or move";
const QColor  ViewTable::CTXMENU_BACKGROUND_COLOR = qRgb( 0xdd, 0xdd, 0xdd );
//...................................................................
const QString ViewTable::FILE_SELECTION_CAPTION   = QT_TR_NOOP( "File selection" );
const QString ViewTable::FILE_SELECTION_PROMPT    = QT_TR_NOOP( "Enter selection mask" );
const QString ViewTable::FILE_DESELECTION_CAPTION = QT_TR_NOOP( "File deselection" );
const QString ViewTable::FILE_DESELECTION_PROMPT  = QT_TR_NOOP( "Enter deselection mask" );

//*******************************************************************
// ViewTable
//*******************************************************************
ViewTable::ViewTable( QWidget* const in_parent )
: QListView( in_parent )
, d_timer( this )
, d_sel_counter( 0 )
, d_fname_on_start( Shared::EmptyStr )
, d_selection_mode(( Config::NC == Config::instance()->selection_mode()) ? Multi : Extended )
{
	viewport()->setAcceptDrops( TRUE );
	setAcceptDrops( TRUE );
	setShowSortIndicator( TRUE );
	setSelectionMode( d_selection_mode );
	setAllColumnsShowFocus( TRUE );

	for( int i = 0; i < COLUMNS; ++i ) {
		addColumn( tr(COLDEF[i].name) );
		setColumnAlignment ( i, COLDEF[i].align );
	}
	
	connect( this     , SIGNAL( selectionChanged() ),
				this     , SLOT  ( selection_changed() ));
	connect( this     , SIGNAL( rightButtonClicked( QListViewItem*, const QPoint&, int )),
				this     , SLOT  ( context_menu( QListViewItem*, const QPoint&, int )));
	connect( this     , SIGNAL( F2() ),
				in_parent, SLOT  ( F2() ));
	connect( this     , SIGNAL( F3() ),
				in_parent, SLOT  ( F3() ));
	connect( this     , SIGNAL( F4() ),
				in_parent, SLOT  ( F4() ));
	connect( this     , SIGNAL( F5() ),
				in_parent, SIGNAL( F5() ));
	connect( this     , SIGNAL( F6() ),
				in_parent, SLOT  ( F6() ));
	connect( this     , SIGNAL( F8() ),
				in_parent, SLOT  ( F8() ));
	connect( this     , SIGNAL( F9() ),
				in_parent, SIGNAL( F9() ));
	connect( this     , SIGNAL( empty() ),
				in_parent, SLOT  ( empty() ));
	connect( this     , SIGNAL( touch() ),
				in_parent, SLOT  ( touch() ));
	connect( this     , SIGNAL( disp_dir_size( ViewTableItem* ) ),
	         in_parent, SLOT  ( disp_dir_size( ViewTableItem* ) ));
	connect( &d_timer , SIGNAL( timeout()),
	         this     , SLOT  ( timeout()) );
}
// end of ViewTable

//*******************************************************************
// keyPressEvent                                   PRIVATE inherited
//*******************************************************************
void ViewTable::keyPressEvent( QKeyEvent* e )
{
	bool accepted = FALSE;
	
	if( Qt::ControlButton == e->state() ) {
		switch( e->key() ) {
			case Qt::Key_Home:
				e->accept();
				accepted = TRUE;
				emit cd_home();
				break;
			case Qt::Key_Prior:
				e->accept();
				accepted = TRUE;
				emit cd_up();
				break;
			case Qt::Key_N:
				e->accept();
				accepted = TRUE;
				emit touch();
				break;
			case Qt::Key_E:
				e->accept();
				accepted = TRUE;
				emit empty();
				break;
			case Qt::Key_Backspace:
				e->accept();
				accepted = TRUE;
				emit cd_root();
				break;
		}
	}
	else if( Qt::AltButton == e->state() ) {
		switch( e->key() ) {
			case Qt::Key_Space:
				ViewTableItem* const item = dynamic_cast<ViewTableItem*>( currentItem() );
				if( item ) {
					if( item->is_dir() && item->is_readable() ) {
						emit disp_dir_size( item );
					}
				}
				break;
		}
	}
	else {
		switch( e->key() ) {
			case Qt::Key_Backspace:
				e->accept();
				accepted = TRUE;
				emit cd_up();
				break;
			case Qt::Key_Delete:
				e->accept();
				accepted = TRUE;
				emit remove();
				break;
			case Qt::Key_Plus:
				e->accept();
				accepted = TRUE;
				select_plus();
				break;
			case Qt::Key_Minus:
				e->accept();
				accepted = TRUE;
				select_minus();
				break;
			case Qt::Key_Asterisk:
				e->accept();
				accepted = TRUE;
				invertSelection();
				break;
			case Qt::Key_F2:
				e->accept();
				accepted = TRUE;
				emit F2();
				break;
			case Qt::Key_Insert:
				if( Multi == d_selection_mode ) {
					e->accept();
					accepted = TRUE;
					select_this_one();
				}
				break;
			case Qt::Key_Space:
				if( Multi == d_selection_mode ) {
					e->accept();
					accepted = TRUE;
					// nic nie rob
				}
				break;
			
		}
	}
	
	if( FALSE == accepted ) {
		QListView::keyPressEvent( e );
	}
}
// end of keyPressEvent
/*
void ViewTable::contentsMousePressEvent ( QMouseEvent * e )
{
	qWarning( "ViewTable::QMouseEvent" );
	QListView::contentsMousePressEvent ( QMouseEvent * e );
}
*/
//*******************************************************************
// select_this_one                                           PRIVATE
//*******************************************************************
void ViewTable::select_this_one()
{
	ViewTableItem* const item = dynamic_cast<ViewTableItem*>( currentItem() );
	if( item ) {
		const bool new_state = ( FALSE == isSelected( item ) );
		setSelected( item, new_state );
		
		QListViewItem* const next_item = item->itemBelow();
		setCurrentItem( next_item );
		ensureItemVisible( next_item );
	}
}
// end of select_this_one

//*******************************************************************
// select_item_after_start                                    PUBLIC
//*******************************************************************
void ViewTable::select_item_after_start()
{
	if( FALSE == d_fname_on_start.isEmpty() ) {
		select_by_name( d_fname_on_start );
		d_fname_on_start = Shared::EmptyStr;
	}
}
// end of select_item_after_start

//*******************************************************************
// showEvent                                       PRIVATE inherited
//*******************************************************************
void ViewTable::showEvent( QShowEvent* e )
{
	QListView::showEvent( e );
	select_item_after_start();
	adjust();
}
// end of showEvent

//*******************************************************************
// setSorting                                      PRIVATE inherited
//*******************************************************************
void ViewTable::setSorting( int in_column, bool in_ascending )
{
	QListView::setSorting( in_column, in_ascending );
	ensureItemVisible( currentItem() );
}
// end of setSorting

//*******************************************************************
// adjust                                                     PUBLIC
//*******************************************************************
void ViewTable::adjust()
{
	static bool first_time = TRUE;
	bool adjust = TRUE;
	
	if( 1 == Config::instance()->disable_columns_resize()	) {
		if( first_time ) first_time = FALSE;
		else             adjust = FALSE;
	}

	if( adjust ) {
		for( int col = COL_FNAME; col < COLUMNS; ++col ) {
			adjustColumn( col );
		}
	}
}
// end of adjust

//*******************************************************************
// reset                                                      PUBLIC
//*******************************************************************
void ViewTable::reset()
{
	QListViewItem* item = firstChild();
	while ( item ) {
		takeItem( item );
		item = firstChild();
   }
}
// end of add_new_item

//*******************************************************************
// add_new_item                                               PUBLIC
//*******************************************************************
// wersja nr. 1 dla lokalnego systemu plikow
//-------------------------------------------
void ViewTable::add_new_item( const QFileInfo& in_fi, const bool in_is_root )
{
	if( can_display( in_fi.fileName(), in_is_root )) {
		display_item_info( in_fi );
	}
}
// wersja nr. 2 dla ftp
//---------------------
void ViewTable::add_new_item( const QUrlInfo& in_ui, const bool in_is_root )
{
	if( can_display( in_ui.name(), in_is_root )) {
		display_item_info( in_ui );
	}
}
// end of add_new_item

//*******************************************************************
// get_fname_fext                                             PUBLIC
//*******************************************************************
void ViewTable::get_fname_fext( const ViewTableItem* const in_item, QString& out_fname, QString& out_fext )
{
	out_fname = in_item->text( COL_FNAME );
	out_fext  = in_item->text( COL_FEXT );
}
// end of get_current_name

//*******************************************************************
// get_fname                                                  PUBLIC
//*******************************************************************
void ViewTable::get_fname( const ViewTableItem* const in_item, QString& out_fname )
{
	QString fext;
	get_fname_fext( in_item, out_fname, fext );
	if( !fext.isEmpty() ) {
		out_fname += Shared::Point;
		out_fname += fext;
	}
}
// end of get_fname

//*******************************************************************
// get_current_fname_fext                                     PUBLIC
//*******************************************************************
void ViewTable::get_current_fname_fext( QString& out_fname, QString& out_fext )
{
	out_fname = Shared::EmptyStr;
	const ViewTableItem* const item = dynamic_cast<ViewTableItem*>( currentItem() );
	if( item ){
		get_fname_fext( item, out_fname, out_fext );
	}
}
// end of get_current_fname_fext

//*******************************************************************
// get_current_info                                          PUBLIC
//*******************************************************************
bool ViewTable::get_current_info( QString& out_fname, bool& out_isdir )
{
	bool retval = FALSE;
	const ViewTableItem* const item = current_item();
	if( item ) {
		get_fname( item, out_fname );
		out_isdir = item->is_dir();
		retval = TRUE;
	}
	return retval;
}
// end of get_current_info

//*******************************************************************
// get_current_fname                                          PUBLIC
//*******************************************************************
void ViewTable::get_current_fname( QString& out_fname )
{
	out_fname = Shared::EmptyStr;
	const ViewTableItem* const item = current_item();
	if( item ){
		get_fname( item, out_fname );
	}
}
// end of get_current_fname

//*******************************************************************
// check_parent_mark                                          PUBLIC
//-------------------------------------------------------------------
// To jest funkcja reagujaca na rozne dzialanie serwerow ftp.
// Niektore przy odczycie katalogu zwracaja "..", a inne nie.
// My znak katalogu ".." potrzebujemy.
// Ta funkcja ma sprawdzic, czy jest on czy nie. Jesli go nie,
// ma go dodac.
//*******************************************************************
void ViewTable::check_parent_mark()
{
	if( 0 == findItem( Shared::ParentDir, COL_FNAME ) ) {
		QUrlInfo ui;
		ui.setName( Shared::ParentDir );
		ui.setDir( TRUE );
		display_item_info( ui );
	}
}
// end of check_parent_mark

//*******************************************************************
// set_dir_size                                               PUBLIC
//*******************************************************************
void ViewTable::set_dir_size( ViewTableItem* const in_item, const Q_ULLONG in_value )
{
	in_item->set_is_size_computed();
	in_item->setText( COL_SIZE, Shared::num2str( in_value ));
	adjust();
}
// end of set_dir_size


//###################################################################
//#                                                                 #
//#              OPERACJE ZWIAZANE Z ZAZNACZANIEM                   #
//#                                                                 #
//###################################################################


//*******************************************************************
// select_by_name                                             PUBLIC
//*******************************************************************
void ViewTable::select_by_name( const QString& in_fname, const QString& in_fext )
{
	bool found = FALSE;
	
	QListViewItemIterator it( this );
	while ( it.current() ) {
    	if( in_fname == (*it)->text( COL_FNAME ) ) {
			if( in_fext == (*it)->text( COL_FEXT ) ) {
//				setSelected( *it, TRUE );
				setCurrentItem( *it );
				ensureItemVisible( *it );
				found = TRUE;
				break;
			}
		}
		++it;
   }
   
	if( FALSE == found ) {
		select_first_item();
	}
}
void ViewTable::select_by_name( const QString& in_fname )
{
	const QFileInfo fi( in_fname );
	const QString fname = fi.baseName( TRUE );
	const QString fext = fi.extension( FALSE );
	select_by_name( fname, fext );
}
// end of select_by_name

void ViewTable::select_first_item()
{
	QListViewItem* const item = firstChild();
	if( item ) {
		setCurrentItem( item );
		ensureItemVisible( item );
	}
}

//*********************************************************
// select_dir_by_name                               PUBLIC
//*********************************************************
void ViewTable::select_dir_by_name( const QString& in_dname )
{
	QListViewItemIterator it( this );
	while ( it.current() ) {
    	if( in_dname == (*it)->text( COL_FNAME ) ) {
//			setSelected( *it, TRUE );
			setCurrentItem( *it );
			ensureItemVisible( *it );
			break;
		}
		++it;
   }
}
// end of select_dir_by_name

//*******************************************************************
// select_plus                                               PRIVATE
//-------------------------------------------------------------------
// Uzytkownik wcisnal klawisz '+'.
// Oznacza to, ze chce dokonac zaznaczenia plikow.
// W tym celu musi oczywiscie podac maske wyszukiwania.
//*******************************************************************
void ViewTable::select_plus()
{
	if( childCount() > 0 ) {
		bool ok = TRUE;
		const QString mask = QInputDialog::getText(
										tr(FILE_SELECTION_CAPTION), tr(FILE_SELECTION_PROMPT),
										QLineEdit::Normal, QString::null,
										&ok, this );
		if( TRUE == ok ) {										
			QString fname = Shared::EmptyStr;
			QRegExp regexp( mask, TRUE, TRUE );

   		QListViewItemIterator it( this );
    		while ( it.current() ) {
    			ViewTableItem* const item = dynamic_cast<ViewTableItem*>( *it );
    			get_fname( item, fname );
    			if( fname != Shared::ParentDir ) {
    				if( !item->isSelected() ) {
    					if( regexp.search( fname, 0 ) != -1 ) {
    						setSelected( item, TRUE );
    					}
    				}
    			}
  	      	++it;
   	 	}
		}
	}
}
// end of select_plus

//*******************************************************************
// select_minus                                              PRIVATE
//-------------------------------------------------------------------
// Uzytkownik wcisnal klawisz '-'.
// Oznacza to, ze chce dokonac zniesienia zaznaczen.
// W tym celu musi oczywiscie podac maske wyszukiwania.
//*******************************************************************
void ViewTable::select_minus()
{
	if( childCount() > 0 ) {
		bool ok = TRUE;
		const QString mask = QInputDialog::getText(
										tr(FILE_DESELECTION_CAPTION), tr(FILE_DESELECTION_PROMPT),
										QLineEdit::Normal, QString::null,
										&ok, this );
		if( TRUE == ok ) {										
			QString fname = "";
			QRegExp regexp( mask, TRUE, TRUE );

   		QListViewItemIterator it( this );
    		while ( it.current() ) {
    			ViewTableItem* const item = dynamic_cast<ViewTableItem*>( *it );
    			get_fname( item, fname );
    			if( fname != Shared::ParentDir ) {
    				if( item->isSelected() ) {
    					if( regexp.search( fname, 0 ) != -1 ) {
    						setSelected( item, FALSE );
    					}
    				}
    			}
  	      	++it;
   	 	}
		}
	}
}
// end of select_minus

//*******************************************************************
// selections                                                 PUBLIC
//-------------------------------------------------------------------
// Pierwszenstwo we wszelkich operacjach maja pozycje zaznaczone.
// Jesli takich nie ma przedmiotem operacji bedzie pozycja aktualna.
//*******************************************************************
const ViewTable::SelectedItems& ViewTable::selections()
{
	d_selections.clear();

	QListViewItemIterator it( this );
	while ( it.current() ) {
   	if( (*it)->isSelected() ) {
   		ViewTableItem* const item = dynamic_cast<ViewTableItem*>( *it );
   		if( FALSE == item->is_parent_dir() ) {
    			d_selections.push_back( item );
   		}
		}
		++it;
   }
	
	if( d_selections.empty() ) {
		ViewTableItem* const item = dynamic_cast<ViewTableItem*>( currentItem() );
		if( FALSE == item->is_parent_dir() ) {
    		d_selections.push_back( item );
   	}
	}

	return d_selections;
}
// end of selections

//*******************************************************************
// set_fname_on_start                                         PUBLIC
//*******************************************************************
void ViewTable::set_fname_on_start( const QString& in_fname )
{
	d_fname_on_start = in_fname;
}
// end of set_fname_on_start

//*******************************************************************
// get_selections_number                                     PRIVATE
//*******************************************************************
int ViewTable::get_selections_number()
{
	int counter = 0;
	
	if( childCount() > 0 ) {
   	QListViewItemIterator it( this );
    	while ( it.current() ) {
  			if( (*it)->isSelected() ) ++counter;
  	     	++it;
  		}
	}
	return counter;
}
// end of get_selections_number

//###################################################################
//#                                                                 #
//#            OBSLUGA ZDARZEN ZWIAZANYCH Z FOCUSEM                 #
//#                                                                 #
//###################################################################


//*******************************************************************
// focusInEvent                                    PRIVATE inherited
//-------------------------------------------------------------------
// Zmiana wygladu po otrzymaniu focusa.
//*******************************************************************
void ViewTable::focusInEvent( QFocusEvent* in_event )
{
	QListView::focusInEvent( in_event );
	updateContents();
	ensureItemVisible( currentItem() );
}
// end of focusInEvent

//*******************************************************************
// focusOutEvent                                   PRIVATE inherited
//-------------------------------------------------------------------
// Zmiana wygladu po utracie focusa.
//*******************************************************************
void ViewTable::focusOutEvent ( QFocusEvent* in_event )
{
	QListView::focusOutEvent( in_event );
	updateContents();
	
	const ViewTableItem* const item = current_item();
	d_fname_on_start = ( item ) ? item->name() : Shared::EmptyStr;
}
// end of focusOutEvent


//*******************************************************************
// viewportMousePressEvent                         PRIVATE inherited
//*******************************************************************
void ViewTable::viewportMousePressEvent( QMouseEvent* const in_event )
{
	if( Multi == d_selection_mode ) {
		QListViewItem* const item = itemAt( in_event->pos() );
		if( item ) {
			if( Qt::LeftButton == in_event->button() ) {
				setCurrentItem( item );
			}
			else if( Qt::RightButton == in_event->button() ) {
				setCurrentItem( item );
				setSelected( item, FALSE == item->isSelected() );
			}
			else {
				QListView::viewportMousePressEvent( in_event );
			}
		}
	}
	else {
		QListView::viewportMousePressEvent( in_event );
	}
}
// end of viewportMousePressEvent


//###################################################################
//#                                                                 #
//#                        DRAG & DROP                              #
//#                                                                 #
//###################################################################


//*******************************************************************
// contentsMousePressEvent                         PRIVATE inherited
//*******************************************************************
void ViewTable::contentsMousePressEvent( QMouseEvent* in_event )
{
	if( Extended == d_selection_mode ) {
		if( LeftButton == in_event->button() ) {
			d_drag_pos = in_event->pos();
		}
	}
	QListView::contentsMousePressEvent( in_event );
}
// end of contentsMousePressEvent

//*******************************************************************
// contentsMouseMoveEvent                          PRIVATE inherited
//*******************************************************************
void ViewTable::contentsMouseMoveEvent( QMouseEvent* in_event )
{
	if( Extended == d_selection_mode ) {
		if( in_event->state() & LeftButton ) {
			const int dx = ( in_event->pos() - d_drag_pos ).manhattanLength();
			if( dx > QApplication::startDragDistance() ) {
				start_drag();
			}
		}
	}
	QListView::contentsMouseMoveEvent( in_event );
}
// end of contentsMouseMoveEvent

//*******************************************************************
// contentsDragEnterEvent                          PRIVATE inherited
//*******************************************************************
void ViewTable::contentsDragEnterEvent( QDragEnterEvent* in_event )
{
	if( Extended == d_selection_mode ) {
		in_event->accept( in_event->provides( DRAG_DROP_SUBTYPE ));
	}
}
// end of contentsDragEnterEvent

//*******************************************************************
// contentsDropEvent                               PRIVATE inherited
//*******************************************************************
void ViewTable::contentsDropEvent( QDropEvent* in_event )
{
	if( Extended == d_selection_mode ) {
		QString fname;
		if( QTextDrag::decode( in_event, fname )) {
			const QWidget* const from_widget = in_event->source();
			if( from_widget ) {
				if( from_widget != this ) {
					if( from_widget->inherits( "ViewTable" )) {
						d_timer.start( 5, TRUE );
					}
				}
			}
		}
	}
}
// end of contentsDropEvent

//*******************************************************************
// start_drag                                                PRIVATE
//*******************************************************************
void ViewTable::start_drag()
{
	if( Extended == d_selection_mode ) {
		QTextDrag* const drag_item = new QTextDrag( DRAG_DROP_CMD, this );
		drag_item->setSubtype( DRAG_DROP_SUBTYPE.utf8() );
		drag_item->setPixmap( QPixmap::fromMimeSource( DRAG_DROP_ICON ));
		drag_item->drag();
	}
}
// end of start_drag

//*******************************************************************
// timeout                                              PRIVATE slot
//*******************************************************************
void ViewTable::timeout()
{
	select_item_after_start();
	emit F5();
}
// end of timeout

//###################################################################
//#                                                                 #
//#               WYSWIETLENIE DANYCH W TABLICY                     #
//#                                                                 #
//###################################################################


//*******************************************************************
// display_item_info                                         PRIVATE
//*******************************************************************
// wersja nr. 1 dla lokalnego systemu plikow.
//-------------------------------------------
void ViewTable::display_item_info( const QFileInfo& in_fi )
{
	d_item_as_text.clear();
	
	if( Shared::ParentDir == in_fi.fileName() ) {
		d_item_as_text.info[ COL_FNAME ] = Shared::ParentDir;
	}
	else {
		get_access( in_fi, d_item_as_text.info[ COL_ACCESS ] );
		if( 'd' == d_item_as_text.info[COL_ACCESS][0] ) d_item_as_text.info[ COL_FNAME ] = in_fi.fileName();
		else get_fname_fext( in_fi.fileName(), d_item_as_text.info[ COL_FNAME ], d_item_as_text.info[ COL_FEXT ] );
		get_size      ( in_fi.size(), d_item_as_text.info[ COL_SIZE ] );
		get_date_time ( in_fi.lastModified(), d_item_as_text.info[ COL_DATE ], d_item_as_text.info[ COL_TIME ] );

		d_item_as_text.info[ COL_OWNER ]  = in_fi.owner();
		d_item_as_text.info[ COL_GROUP ]  = in_fi.group();
		
		if( in_fi.isDir() ) emit next_dir();
		else emit next_file();
	}

	ViewTableItem* const item = create_tbl_item( d_item_as_text, FALSE );
	item->set_data( in_fi );
	add_dir_icon( item );
}
// wersja nr. 2 dla ftp
//----------------------
void ViewTable::display_item_info( const QUrlInfo& in_ui )
{
	d_item_as_text.clear();
	
	if( Shared::ParentDir == in_ui.name() ) {
		d_item_as_text.info[ COL_FNAME ] = Shared::ParentDir;
	}
	else {
		get_access( in_ui, d_item_as_text.info[ COL_ACCESS ] );
		if( 'd' == d_item_as_text.info[ COL_ACCESS ][0] ) d_item_as_text.info[ COL_FNAME ] = in_ui.name();
		else get_fname_fext( in_ui.name(), d_item_as_text.info[ COL_FNAME ], d_item_as_text.info[ COL_FEXT ] );
		get_size      ( in_ui.size()        , d_item_as_text.info[ COL_SIZE ] );
		get_date_time ( in_ui.lastModified(), d_item_as_text.info[ COL_DATE ], d_item_as_text.info[ COL_TIME ] );
		
		d_item_as_text.info[ COL_OWNER ]  = in_ui.owner();
		d_item_as_text.info[ COL_GROUP ]  = in_ui.group();
		
		if( in_ui.isDir() ) emit next_dir();
		else emit next_file();
	}

	ViewTableItem* const item = create_tbl_item( d_item_as_text, TRUE );
	item->set_data( in_ui );
	add_dir_icon( item );
}
// end of display_item_info

//*******************************************************************
// can_display                                               PRIVATE
//*******************************************************************
bool ViewTable::can_display( const QString& in_fname, const bool in_is_root )
{
	bool retval = TRUE;

	// Jednej kropki, THIS_DIR, nie wyswietlamy nigdy.
	if( Shared::ThisDir == in_fname ) {
		retval = FALSE;
	}
	// Normalnie dwukropek (PARENT_DIR) jest wyswietlamy.
	// Z wyjatkiem, gdy jestesmy w katalogu root (on nie ma katalogu parent).
	else if(( Shared::ParentDir == in_fname ) && in_is_root ) { 
		retval = FALSE;
	}

	return retval;
}
// end of can_display

//*******************************************************************
// get_fname_fext                                             PUBLIC
//*******************************************************************
void ViewTable::get_fname_fext( const QString& in_fname, QString& out_fname, QString& out_fext )
{
	if( Shared::ParentDir != in_fname ) {
		const int idx = in_fname.findRev( Shared::Point );
		if(( idx != -1 ) && ( idx != 0 )) {
			out_fname = in_fname.left( idx );
			out_fext = in_fname.mid( idx + 1 );
		}
		else {
			out_fname = in_fname;
			out_fext  = Shared::EmptyStr;
		}
	}
}
// end of get_fname_fext

//*******************************************************************
// get_sccess                                                PRIVATE
//*******************************************************************
// 1. lfs
//--------
void ViewTable::get_access( const QFileInfo& in_fi, QString& out_access )
{
	out_access = "----------";
	
	     if( in_fi.isDir()     ) out_access.at( 0 ) = 'd';
	else if( in_fi.isSymLink() ) out_access.at( 0 ) = 'l';

	if( in_fi.permission( QFileInfo::ReadUser   )) out_access.at( 1 ) = 'r';
	if( in_fi.permission( QFileInfo::WriteUser  )) out_access.at( 2 ) = 'w';
	if( in_fi.permission( QFileInfo::ExeUser    )) out_access.at( 3 ) = 'x';
	if( in_fi.permission( QFileInfo::ReadGroup  )) out_access.at( 4 ) = 'r';
	if( in_fi.permission( QFileInfo::WriteGroup )) out_access.at( 5 ) = 'w';
	if( in_fi.permission( QFileInfo::ExeGroup   )) out_access.at( 6 ) = 'x';
	if( in_fi.permission( QFileInfo::ReadOther  )) out_access.at( 7 ) = 'r';
	if( in_fi.permission( QFileInfo::WriteOther )) out_access.at( 8 ) = 'w';
	if( in_fi.permission( QFileInfo::ExeOther   )) out_access.at( 9 ) = 'x';
}
// 2. ftp
//--------
void ViewTable::get_access( const QUrlInfo& in_ui, QString& out_access )
{
	out_access = "----------";
	
	     if( in_ui.isDir()     ) out_access.at( 0 ) = 'd';
	else if( in_ui.isSymLink() ) out_access.at( 0 ) = 'l';

	if( in_ui.permissions() & QUrlInfo::ReadOwner  ) out_access.at( 1 ) = 'r';
	if( in_ui.permissions() & QUrlInfo::WriteOwner ) out_access.at( 2 ) = 'w';
	if( in_ui.permissions() & QUrlInfo::ExeOwner   ) out_access.at( 3 ) = 'x';
	if( in_ui.permissions() & QUrlInfo::ReadGroup  ) out_access.at( 4 ) = 'r';
	if( in_ui.permissions() & QUrlInfo::WriteGroup ) out_access.at( 5 ) = 'w';
	if( in_ui.permissions() & QUrlInfo::ExeGroup   ) out_access.at( 6 ) = 'x';
	if( in_ui.permissions() & QUrlInfo::ReadOther  ) out_access.at( 7 ) = 'r';
	if( in_ui.permissions() & QUrlInfo::WriteOther ) out_access.at( 8 ) = 'w';
	if( in_ui.permissions() & QUrlInfo::ExeOther   ) out_access.at( 9 ) = 'x';
}
// end of get_access

//*******************************************************************
// get_size                                                  PRIVATE
//*******************************************************************
inline void ViewTable::get_size( const unsigned int in_size, QString& out_size )
{
	out_size = Shared::num2str( in_size );
}
// end of get_size

//*******************************************************************
// get_date_time                                             PRIVATE
//*******************************************************************
void ViewTable::get_date_time( const QDateTime& in_dt, QString& out_date, QString& out_time )
{
	out_date = in_dt.date().toString( "yyyy-MM-dd" );
	out_time = in_dt.time().toString( "hh:mm:ss" );
}
// end of get_date_time


//###################################################################
//#                                                                 #
//#                           SLOTY                                 #
//#                                                                 #
//###################################################################


//*******************************************************************
// selection_changed                                    PRIVATE slot
//*******************************************************************
void ViewTable::selection_changed()
{
	static int number = 0;
	const int counter = get_selections_number();
	
	if( number != counter ) {
		emit selections_update( number = counter );
	}
}
// end of selection_changed

//*******************************************************************
// selection_mode_changed                               PRIVATE slot
//*******************************************************************
void ViewTable::selection_mode_changed()
{
	d_selection_mode = ( Config::NC == Config::instance()->selection_mode() ) ? Multi : Extended;
}
// end of selection_mode_changed

//*******************************************************************
// rightButtonClicked                              PRIVATE inherited
//*******************************************************************
void ViewTable::context_menu( QListViewItem*, const QPoint& in_pos, int )
{
	QPopupMenu* menu = new QPopupMenu( this );
	menu->setBackgroundMode( Qt::PaletteBackground );
	menu->setPaletteBackgroundColor( CTXMENU_BACKGROUND_COLOR );

	menu->insertItem( Shared::MenuAccess, this, SIGNAL( F2() ));
	menu->insertItem( Shared::MenuView  , this, SIGNAL( F3() ));
	menu->insertItem( Shared::MenuEdit  , this, SIGNAL( F4() ));
	menu->insertItem( Shared::MenuCopy  , this, SIGNAL( F5() ));
	menu->insertItem( Shared::MenuRename, this, SIGNAL( F6() ));
	menu->insertItem( Shared::MenuDelete, this, SIGNAL( F8() ));
	menu->insertItem( Shared::MenuPack  , this, SIGNAL( F9() ));
	menu->insertSeparator();
	menu->insertItem( Shared::MenuEmpty , this, SIGNAL( empty() ));

	menu->exec( in_pos );
	delete menu;
	menu = 0;
}
// end of rightButtonClicked

//*******************************************************************
// retranslate_strings                                  PRIVATE slot
//*******************************************************************
void ViewTable::retranslate_strings()
{
	for( int i = 0; i < COLUMNS; ++i ) {
		header()->setLabel( i, tr(COLDEF[i].name) );
	}
	adjust();
}
// end of retranslate_strings
