/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "Viewer.h"
#include "ViewTableItem.h"
#include "Config.h"
#include "Shared.h"
#include "Busy.h"
#include "SystemCall.h"
#include <qfile.h>
#include <qfont.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qfontmetrics.h>
#include <qapplication.h>
#include <qtextedit.h>
#include <qdesktopwidget.h>
#include <vector>
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const QString Viewer::TextSign  = "text/plain";
const QString Viewer::TarSign   = "x-tar";
const QString Viewer::GzipSign  = "x-gzip";
const QString Viewer::Bzip2Sign = "x-bzip2";
//...................................................................
const QString Viewer::Caption        = QT_TR_NOOP( "Viewer: " );


//*******************************************************************
// Viewer
//*******************************************************************
Viewer::Viewer( QWidget* const in_parent, const ViewTableItem* const in_item )
: QDialog        ( in_parent )
, d_main_layout  ( new QVBoxLayout( this ))
, d_btn_layout   ( new QHBoxLayout )
, d_browser      ( new QTextEdit( this ))
, d_btn_return   ( new QPushButton( tr(Shared::CloseBtnLabel), this ))
, d_btn_break    ( new QPushButton( tr(Shared::BreakBtnLabel), this ))
, d_path         ( in_item->path() )
, d_what         ( CHECK )
, d_syscall      ( new SystemCall( this ))
{
	setCaption( tr( Caption ) + in_item->name() );
	Shared::add_icon( d_btn_return, Shared::CloseIcon );
	setFont( Config::instance()->lfs_default_font() );
	
	d_browser->setReadOnly( TRUE );
	d_browser->setTextFormat( Qt::PlainText );
	d_browser->setPaletteBackgroundColor( Config::instance()->lfs_default_bkg_color() );
	d_browser->setWordWrap( QTextEdit::NoWrap );
	const QFont fnt( "Monospace", 9 );
	d_browser->setFont( fnt );
	const QFontMetrics fm( fnt );
	d_browser->setTabStopWidth( 3 * fm.width( Shared::WidthMarker ));
	
	d_btn_break->hide();
		
	d_main_layout->setMargin( Shared::LayoutMargin  );
	d_main_layout->setSpacing( Shared::LayoutSpacing );
		
	d_btn_layout->addStretch( Shared::OverStretch );
	d_btn_layout->addWidget( d_btn_break );
	d_btn_layout->addWidget( d_btn_return );

	d_main_layout->addWidget( d_browser );
	d_main_layout->addLayout( d_btn_layout );

	connect( d_btn_return, SIGNAL( clicked() ), this, SLOT( accept() ));
	connect( d_syscall  , SIGNAL( finished( int ) ), this, SLOT( finished( int ) ));
	connect( d_syscall  , SIGNAL( stdout_line( const QString& )), this, SLOT( stdout_line( const QString& )));
	
	vector<QString> parameters;
	parameters.push_back( "file" );
	parameters.push_back( "-b" );
	parameters.push_back( "-i" );
	parameters.push_back( d_path );
	d_syscall->run( parameters );
}
// end of Viewer

//*******************************************************************
// ~Viewer
//*******************************************************************
Viewer::~Viewer()
{
	if( d_syscall ) {
		delete d_syscall;
		d_syscall = 0;
	}
	Busy::set_busy( FALSE );
}
// end of ~Viewer

//*******************************************************************
// finished                                             PRIVATE slot
//*******************************************************************
void Viewer::finished( int in_status )
{
	vector<QString> stdout_lines;
	vector<QString> stderr_lines;
	d_syscall->result( stdout_lines, stderr_lines );

	if( in_status >= 0 ) {
		if( CHECK == d_what ) {
			vector<QString> parameters;
			const QString line = stdout_lines[0];
				
			d_what = DISPLAY;
			if( line.find( TextSign ) != -1 ) {
				parameters.push_back( "cat" );
				parameters.push_back( "-n" );
				parameters.push_back( d_path );
			}
			else if(
					( line.find( GzipSign )  != -1 ) ||
					( line.find( TarSign )   != -1 ) || 
					( line.find( Bzip2Sign ) != -1 )
				) {
				parameters.push_back( "tar" );
				parameters.push_back( "-t" );
				parameters.push_back( "-f" );
				parameters.push_back( d_path );
			}
			else {
				parameters.push_back( "xxd" );
				parameters.push_back( "-g" );
				parameters.push_back( "1" );
				parameters.push_back( d_path );
			}
			Busy::set_busy( TRUE );
			d_syscall->run( parameters, SystemCall::ON_LINE );
		}
		else {
			if( FALSE == stdout_lines.empty() ) {
				vector<QString>::const_iterator it = stdout_lines.begin();
				++it;
				while( it != stdout_lines.end() ) {
					d_browser->append( *it );
					++it;
				}
			}
			Busy::set_busy( FALSE );
			d_browser->setCursorPosition( 0, 0 );
		}
	}
}
// end of finished

//*******************************************************************
// polish
//*******************************************************************
void Viewer::polish()
{
	Shared::polish( this, 50, 70 );
}
// end of polish

//*******************************************************************
// stdout_line                                          PRIVATE slot
//*******************************************************************
void Viewer::stdout_line( const QString& in_line )
{
	d_browser->append( in_line );
	Shared::idle();
}
// end of stdout_line

//*******************************************************************
// stderr_line                                          PRIVATE slot
//*******************************************************************
void Viewer::stderr_line( const QString& )
{
}
// end of stderr_line
