/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "DateTime.h"
#include "ViewTableItem.h"
#include "Shared.h"
#include "InfoField.h"
#include "Config.h"
#include <qlayout.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qdatetimeedit.h>

/*------- local constants:
-------------------------------------------------------------------*/
const QString DateTime::DATE_TIME_FORMAT    = "yyyy/MM/dd  hh:mm:ss";
const QString DateTime::TOUCH_CMD           = "touch -c -t";
const QString DateTime::TOUCH_FORMAT        = " yyyyMMddhhmm.ss ";
//...................................................................
const QString DateTime::CAPTION             = QT_TR_NOOP( "Date/time of last file modyfication." );
const QString DateTime::BTN_ACCEPT_LABEL    = QT_TR_NOOP( "&Accept" );
const QString DateTime::CB_MUSTER_LABEL     = QT_TR_NOOP( "&Use '%1' like muster" );
const QString DateTime::GBOX_EDIT_CAPTION   = QT_TR_NOOP( "Date and Time" );
const QString DateTime::GBOX_MUSTER_CAPTION = QT_TR_NOOP( "Date/time muster" );

//*******************************************************************
// DateTime                                              CONSTRUCTOR
//*******************************************************************
DateTime::DateTime( QWidget* const in_parent, const ViewTable::SelectedItems& in_items, const ViewTableItem* const in_muster )
: QDialog( in_parent )
, d_items( in_items )
, d_muster( in_muster )
, d_accept_btn( new QPushButton( tr(BTN_ACCEPT_LABEL), this ))
, d_cancel_btn( new QPushButton( tr(Shared::CloseBtnLabel), this ))
, d_muster_cb( new QCheckBox( tr(CB_MUSTER_LABEL).arg(in_muster->name()), this ))
, d_edit_dt( 0 )
, d_muster_if( 0 )
{
	setCaption( tr(CAPTION) );
	Shared::add_icon( d_accept_btn, tr(Shared::AcceptIcon) );
	Shared::add_icon( d_cancel_btn, tr(Shared::CloseIcon) );
	
	QHBoxLayout* const main_layout = new QHBoxLayout( this );
	main_layout->setMargin( Shared::LayoutMargin );
	main_layout->setSpacing( Shared::LayoutSpacing );
	
	QVBoxLayout* const data_layout = new QVBoxLayout;
	data_layout->addWidget( make_edit_area() );
	data_layout->addWidget( d_muster_cb );
	data_layout->addWidget( make_muster_area() );
	main_layout->addLayout( data_layout );
	
	QVBoxLayout* const btn_layout = new QVBoxLayout;
	btn_layout->addStretch( Shared::OverStretch );
	btn_layout->addWidget( d_cancel_btn );
	btn_layout->addWidget( d_accept_btn );
	main_layout->addLayout( btn_layout );
	
	if( !d_muster ) d_muster_cb->setEnabled( FALSE );

	connect( d_accept_btn, SIGNAL( clicked     ()                   ),
				this        , SLOT  ( accept      ()                   ));
	connect( d_cancel_btn, SIGNAL( clicked     ()                   ),
				this        , SLOT  ( cancel      ()                   ));
	connect( d_muster_cb , SIGNAL( toggled     ( bool )             ),
				this        , SLOT  ( use_muster  ( bool )             ));
	connect( d_edit_dt   , SIGNAL( valueChanged( const QDateTime& ) ),
				this        , SLOT  ( no_muster   ( const QDateTime& ) ));
}
// end of DateTime

//*******************************************************************
// make_edit_area                                            PRIVATE
//*******************************************************************
QWidget* DateTime::make_edit_area()
{
	QGroupBox* const gbox = new QGroupBox( 1, Qt::Horizontal, tr(GBOX_EDIT_CAPTION), this );
	d_edit_dt = new QDateTimeEdit( d_items[0]->last_modified(), gbox );
	d_edit_dt->dateEdit()->setOrder( QDateEdit::YMD );
	return gbox;
}
// end of make_edit_area

//*******************************************************************
// make_muster_area                                         PRIVATE
//*******************************************************************
QWidget* DateTime::make_muster_area()
{
	QGroupBox* const gbox = new QGroupBox( 1, Qt::Horizontal, tr(GBOX_MUSTER_CAPTION), this );
	QString muster = Shared::EmptyStr;

	if( d_muster ) {
		const QDateTime dt( d_muster->last_modified() );
		muster = dt.toString( tr(DATE_TIME_FORMAT) );
	}
	d_muster_if = new InfoField( muster, gbox );

	return gbox;
}
// end of make_muster_area

//*******************************************************************
// accept                                               PRIVATE slot
//*******************************************************************
void DateTime::accept()
{
	ViewTable::SelectedItems::const_iterator it = d_items.begin();
	while( it != d_items.end() ) {
		QString cmd = TOUCH_CMD;
		cmd += d_edit_dt->dateTime().toString( TOUCH_FORMAT );
		cmd += (*it)->path();
		const int retval = system( cmd );
		if( -1 == retval ) {
		}
		++it;
	}
	
	QDialog::accept();
}
// end of accept

//*******************************************************************
// cancel                                               PRIVATE slot
//*******************************************************************
void DateTime::cancel()
{
	reject();
}
// end of close

//*******************************************************************
// use_muster                                           PRIVATE slot
//*******************************************************************
void DateTime::use_muster( bool in_checked )
{
	if( in_checked ) {
		d_edit_dt->setDateTime( d_muster->last_modified() );
		d_muster_cb->setChecked( TRUE );
	}
}
// end of use_muster

//*******************************************************************
// no_muster                                            PRIVATE slot
//*******************************************************************
void DateTime::no_muster( const QDateTime& in_dt )
{
	if( in_dt != d_muster->last_modified() ) {
		if( d_muster_cb->isChecked() ) {
			d_muster_cb->setChecked( FALSE );
		}
	}
}
// end of no_muster
