#!/usr/bin/env python
# -*- coding: utf-8 -*-
### BEGIN LICENSE
# Copyright (C) 2009 Rick Spencer rick.spencer@canonical.com
#This program is free software: you can redistribute it and/or modify it 
#under the terms of the GNU General Public License version 3, as published 
#by the Free Software Foundation.
#
#This program is distributed in the hope that it will be useful, but 
#WITHOUT ANY WARRANTY; without even the implied warranties of 
#MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
#PURPOSE.  See the GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License along 
#with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

"""LaunchpadUtils: Helper functions for bug-hugger project.
Wraps typical sequences of LaunchLib calls or provides similar services.
These functions are not designed to be called directly, except from within
classes created as part of bug-hugger.

Note that these functions all assume that the Ubuntu distribution is the
project of interest.

"""


import pygtk
pygtk.require('2.0')
import gtk
import gobject
import os
import re
import urllib
from launchpadlib.launchpad import Launchpad

def get_tasks_with_tag(params):
    lp = params["lp"]
    tag = params["tag"]
    bug_tasks = lp.distributions["ubuntu"].searchTasks(tags=tag)
    bug_dicts = []
    for b in bug_tasks:
        bug_dicts.append(bug_task_dict(lp, b))
    return (tag, bug_dicts)

def get_team(team_text, launchpad):
 """get_team: retrieve a team object from launchpad.

 arguments:
 team_text -- exact string to match for the team
 launchpad -- an instantiated launchpad object

 """
 print "retrieving team members for " + team_text
 teams = launchpad.people.findTeam(text = team_text)
 team = None
 for t in teams:
  if t.name == team_text:
   team = t
 if team is None:
  print "ERROR: Team " + team_text + " not found"
 return team

def get_team_bug_task_list(params):
 """get_team_assigned_list: a helper function for threading 
 retrieving a ListStore of bug_tasks for a specified team.

 arguments:
 params -- normal params object for using with AsynchProgressBox
 params["team"] -- a team object
 params["lp"] -- a launchpad instance

 """
 bug_task_list = []
 team = params["team"]
 lp = params["lp"]
 get_bug_tasks_assigned(team, team, bug_task_list, lp, params)
 print "all bug_tasks retrieved"
 return  (params["team"].name,bug_task_list)

def get_user_assigned_list(params):
 bug_task_list = []
 user = params["user"]
 lp = params["lp"]
 get_bug_tasks_assigned(user, user, bug_task_list, lp, params)
 print "all bug_tasks retrieved"
 return (params["user"].name, bug_task_list, )

def get_user_reported_list_store(params):
 bug_task_list = []
 user = params["user"]
 lp = params["lp"]
 get_bug_tasks_reported(user, user, bug_task_list, lp, params)
 print "all bug_tasks retrieved"
 return (params["user"].name, bug_task_list, )

def get_release_bug_tasks(params):
 """get_release_bug_tasks: a helper funtion for threading retrieving a 
 all current release targeted bug_tasks.

 arguments:
 params -- normal params object for using with AsynchProgressBox
 params["lp"] -- a launchpad instance

 """

 bug_task_list = []
 lp = params["lp"]
 bug_tasks = lp.distributions["ubuntu"].current_series.searchTasks(omit_targeted  = False)
 for bug_task in bug_tasks:
  bug_task_list.append(bug_task_dict(lp, bug_task, None))
 return bug_task_list

def get_bug_tasks_assigned(assigned_to, team, bug_task_list, lp, params):
 """get_bug_tasks_assigned: helper function for filling a gtk.ListStore
 with all bug_tasks assigned to a team and every member thereof.
 Called recursively, designed to be called as a task for a  AsynchProgressBox.

 arguments:
 assigned_to -- a person or team object that may have assigned bug_tasks
 team -- the top level team being queried
 bug_task_list -- the gtk.ListStore that is accumulating the bug_tasks
 lp -- the launchpad instance to use
 params -- normal params object for using with AsynchProgressBox
 check params["kill"] to see if the user has cancelled the query
 
 """
 if params["kill"] == True:
  return
 #get bug_tasks targetted for releases
 for series in lp.distributions["ubuntu"].series:
  if series.active:
   print "retrieving release targetted bug_tasks for " + assigned_to.name + " in " + series.name
   bug_tasks = series.searchTasks(assignee = assigned_to, milestone = None, omit_targeted = False)
   for bug_task in bug_tasks:
    bug_task_list.append(bug_task_dict(lp, bug_task, None, series.name))

 print "retrieving non-targetted bug_tasks for " + assigned_to.name
 #get bug_tasks that are just in the distro
 bug_tasks = lp.distributions["ubuntu"].searchTasks(assignee = assigned_to, milestone = None, omit_targeted  = False)
 for bug_task in bug_tasks:
  bug_task_list.append(bug_task_dict(lp, bug_task, team))

 if assigned_to.is_team:
  t = assigned_to
  for m in assigned_to.members:
   get_bug_tasks_assigned(m, t, bug_task_list, lp, params)

def get_bug_tasks_reported(reported_by, team, bug_task_list, lp, params):
 """get_bug_tasks_reported: helper function for filling a gtk.ListStore
 with all bug_tasks reported by a team and every member thereof.
 Called recursively, designed to be called as a task for a  AsynchProgressBox.

 arguments:
 reported_by -- a person or team object that may have reported bug_tasks
 team -- the top level team being queried
 bug_task_list -- the gtk.ListStore that is accumulating the bug_tasks
 lp -- the launchpad instance to use
 params -- normal params object for using with AsynchProgressBox
 check params["kill"] to see if the user has cancelled the query
 
 """
 if params["kill"] == True:
  return
 #get bug_tasks targetted for releases
 for series in lp.distributions["ubuntu"].series:
  if series.active:
   print "retrieving release targetted bug_tasks for " + reported_by.name + " in " + series.name
   bug_tasks = series.searchTasks(bug_reporter = reported_by, milestone = None, omit_targeted = False)
   for bug_task in bug_tasks:
    bug_task_list.append(bug_task_dict(lp, bug_task, None, series.name))

 print "retrieving non-targetted bug_tasks for " + reported_by.name
 #get bug_tasks that are just in the distro
 bug_tasks = lp.distributions["ubuntu"].searchTasks(bug_task_reporter = reported_by, milestone = None, omit_targeted  = False)
 for bug_task in bug_tasks:
  bug_task_list.append(bug_task_dict(lp, bug_task, team))

 if reported_by.is_team:
  t = reported_by
  for m in reported_by.members:
   get_bug_tasks_reported(m, t, bug_task_list, lp, params)

def get_bug_tasks_subscribed(subscribed_to, team, bug_task_list, lp, params):
 """get_bug_tasks_subscribed: helper function for filling a gtk.ListStore
 with all bug_tasks that a team and every member thereof has subscribed to.
 Called recursively, designed to be called as a task for a  AsynchProgressBox.

 arguments:
 subscribed_to -- a person or team object that may be subscribed to bug_tasks
 team -- the top level team being queried
 bug_task_list -- the gtk.ListStore that is accumulating the bug_tasks
 lp -- the launchpad instance to use
 params -- normal params object for using with AsynchProgressBox
 check params["kill"] to see if the user has cancelled the query
 
 """
 if params["kill"] == True:
  return
 #get bug_tasks targetted for releases
 for series in lp.distributions["ubuntu"].series:
  if series.active:
   print "retrieving release targetted bug_tasks for " + subscribed_to.name + " in " + series.name
   bug_tasks = series.searchTasks(bug_subscriber = subscribed_to, milestone = None, omit_targeted = False)
   for bug_task in bug_tasks:
    bug_task_list.append(bug_task_dict(lp, bug_task, None, series.name))

 print "retrieving non-targetted bug_tasks for " + subscribed_to.name
 #get bug_tasks that are just in the distro
 bug_tasks = lp.distributions["ubuntu"].searchTasks(bug_subscriber = subscribed_to, milestone = None, omit_targeted  = False)
 for bug_task in bug_tasks:
  bug_task_list.append(bug_task_dict(lp, bug_task, team))

 if subscribed_to.is_team:
  t = subscribed_to
  for m in subscribed_to.members:
   get_bug_tasks_subscribed(m, t, bug_task_list, lp, params)

def get_team_subscribed_list_store(params):
 """get_team_subscribed_list: a helper function for threading 
 retrieving a ListStore of bug_tasks for a specified team.

 arguments:
 params -- normal params object for using with AsynchProgressBox
 params["team"] -- a team object
 params["lp"] -- a launchpad instance

 """
 bug_task_list = []
 team = params["team"]
 lp = params["lp"]
 get_bug_tasks_team_subscribed(team, team, bug_task_list, lp, params)
 print "all bug_tasks retrieved"
 return  (params["team"].name,bug_task_list)

def get_bug_tasks_team_subscribed(subscriber, team, bug_task_list, lp, params):
 """get_bug_tasks_team_subscribed: helper function for filling a gtk.ListStore
 with all bug_tasks about packages that a team and every member thereof has 
 subscribed to. Called recursively, designed to be called as a task for a  
 AsynchProgressBox.
 """
 if params["kill"] == True:
  return
 # maybe someday it'll be possible to go from people to a structural subscription in the API
 # pattern used to extract the list of packages subscribed to by the team
 p = re.compile('>([^\s]+)\sin\subuntu')
 subscribed_packages = []
 (f_name, h) = urllib.urlretrieve("https://bugs.launchpad.net/~%s/+packagebugs" % subscriber.name)
 for l in open(f_name):
  m = p.search(l)
  if m:
   subscribed_packages.append(m.group(1))
 for package_name in subscribed_packages:
  package = lp.distributions['ubuntu'].getSourcePackage( name = package_name )
  bug_task_tasks = package.searchTasks()
  for bug_task in bug_task_tasks:
   bug_task_list.append(bug_task_dict(lp, bug_task, team))

 # if more teams start subscribing to packages directly this loop would be unnecessary
 if subscriber.is_team:
  t = subscriber
  for m in subscriber.members:
   get_bug_tasks_team_subscribed(m, t, bug_task_list, lp, params)

def numerical_status(status):
    '''                                                                                               
    Prefix status strings with number (for sorting).                                                  
    Throws a KeyError exception if status is unmappable                                               
    '''
    statuses = {
        "Won't Fix":     "1 - Won't Fix",
        "New":           "2 - New",
        "Incomplete":    "3 - Incomplete",
        "Confirmed":     "4 - Confirmed",
        "Triaged":       "5 - Triaged",
        "In Progress":   "6 - In Progress",
        "Fix Committed": "7 - Fix Committed",
        "Fix Released":  "8 - Fix Released"
        }
    return statuses[status]

def numerical_importance(importance):
    '''                                                                                               
    Prefix importance strings with number (for sorting).                                              
    Throws a KeyError exception if importance is unmappable                                           
    '''
    importances = {
        'Critical':   '1 - Critical',
        'High':       '2 - High',
        'Medium':     '3 - Medium',
        'Low':        '4 - Low',
        'Wishlist':   '5 - Wishlist',
        'Undecided':  '6 - Undecided',
        'Unknown':    '7 - Unknown'
        }
    return importances[importance]

def bug_task_dict(lp, bug_task, team = None, target_release = None):
 """bug_task_dict: helper function for creating a list for using in the 
 gtk.ListStore object of a BugPane.

 arguments:
 bug_task -- the Launchpad BugTask object to parse and add to the gtk.ListStore

 keyword arguments:
 team: a team who's name can be included in the team column
 target_release: a release that can be added to the release column
 extra: an extra data that can be accessed as the last column in the
 gtk.ListStore model

 returns a list object that can be added to a gtk.ListStore
 """

 bdict = {}

 bdict["id"] = bug_task.bug.id

 tf = bug_task.title.split("#")
 tf1 = tf[1].split(" in ")
 tt = ""
 for i in range(1,len(tf1)):
  tt += tf1[i]
  if i < len(tf1) - 1:
   tt += " in "

 tp = tt.split(":")
 title = ""
 for i in range(1, len(tp)):
  title += tp[i]
  if i < len(tp) - 1:
   title += ":"
 title = title.replace("\"","").strip()
 bdict["title"] = title

 if target_release is not None:
  bdict["release"] = target_release

 milestone = ""
 if bug_task.milestone_link is not None:
  bdict["milestone"] = bug_task.milestone_link.split("/")[-1]
 

 assignee = ""
 if bug_task.assignee != None:
  bdict["assignee"] = bug_task.assignee.name

 if team is not None:
  bdict["team name"] = team.name

 # this fetches a new object and is slow; LP #392432 asks to expose it in
 # bug_task
 b = lp.load(bug_task.bug_link)
 tags = " ".join(b.tags)
 bdict["tags"] = tags

 # if a bug has a value for latest_patch_uploaded then the bug likely has a patch
 # in the event of false positives this may not be getting unset in Launchpad when
 # the attachment or flag is removed
 patch = False
# if b.latest_patch_uploaded:
#    patch = True
# bdict["patch?"] = patch

 # bug gravity 
 gravity = 0
 dupe_count  = len(b.duplicates)
 sub_count = len(b.subscriptions)
 user_count = b.users_affected_count

 #TODO: add in gravity when it is reasonably fast
# tag_weights = { 'apport-bug':50, 'apport-package':100, 'apport-crash':100, 
#                 'regression-potential':150, 'regression-release':200,
#                 'regression-proposed':250, 'regression-updates':300 }
# tag_set = set()
# for tag in tag_weights:
#  tag_set.add(tag)
#  bug_tag_set = set()
# for tag in tags: 
#  bug_tag_set.add(tag)
# if tag_set.intersection(bug_tag_set):
#  for tag in tag_set.intersection(bug_tag_set):
#   gravity += tag_weights[tag]
# if b.private:
#  gravity += 151
# gravity += ( 6*dupe_count + 4*sub_count + 2*user_count )
# bdict["gravity"] = gravity
 bdict["status"] = numerical_status(bug_task.status)
 bdict["importance"] = numerical_importance(bug_task.importance)
 bdict["target"] = bug_task.bug_target_display_name
 bdict["date confirmed"] = str(bug_task.date_confirmed)
 bdict["affected users"] = b.users_affected_count
 bdict["__description"] = b.description
 bdict["__bug_task"] = bug_task

 return bdict

