package buoy.widget;

import buoy.event.*;
import buoy.xml.*;
import buoy.xml.delegate.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/**
 * A BRadioButton is a Widget for selecting between several options.  It is normally used as part of a
 * {@link RadioButtonGroup RadioButtonGroup}.  Each BRadioButton represents a single option.  Clicking on it selects it, and
 * deselects all other members of its RadioButtonGroup.
 * <p>
 * In addition to the event types generated by all Widgets, BRadioButtons generate the following event types:
 * <ul>
 * <li>{@link buoy.event.ValueChangedEvent ValueChangedEvent}</li>
 * </ul>
 *
 * @author Peter Eastman
 */

public class BRadioButton extends Widget
{
  private RadioButtonGroup group;
  
  static
  {
    WidgetEncoder.setPersistenceDelegate(BRadioButton.class, new EventSourceDelegate(new String [] {"group"}));
  }

  /**
   * Create a new BRadioButton with no label, which is initially deselected.
   *
   * @param group     the RadioButtonGroup this radio button is part of
   */
  
  public BRadioButton(RadioButtonGroup group)
  {
    this(null, false, group);
  }

  /**
   * Create a new BRadioButton.
   *
   * @param text     the text to display on the BRadioButton
   * @param state    the initial selection state of the BRadioButton
   * @param group    the RadioButtonGroup this radio button is part of
   */
  
  public BRadioButton(String text, boolean state, RadioButtonGroup group)
  {
    component = createComponent(text, state);
    getComponent().addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent ev)
      {
        dispatchEvent(new ValueChangedEvent(BRadioButton.this));
      }
    });
    this.group = group;
    group.add(this);
  }
  
  /**
   * Create the JRadioButton which serves as this Widget's Component.  This method is protected so that
   * subclasses can override it.
   *
   * @param text     the text to display on the BRadioButton
   * @param state    the initial selection state of the BRadioButton
   */
  
  protected JRadioButton createComponent(String text, boolean state)
  {
    return new JRadioButton(text, state);
  }

  public JRadioButton getComponent()
  {
    return (JRadioButton) component;
  }

  /**
   * Get the selection state of this radio button.
   */
  
  public boolean getState()
  {
    return getComponent().isSelected();
  }
  
  /**
   * Set the selection state of this radio button.  If you set the selection state to true,
   * the state of every other member of its RadioButtonGroup will be set to false.
   */
  
  public void setState(boolean selected)
  {
    if (selected)
      group.setSelection(this);
    else
      getComponent().setSelected(selected);
  }
  
  /**
   * Get the text which appears on this radio button.
   */
  
  public String getText()
  {
    return getComponent().getText();
  }
  
  /**
   * Set the text which appears on this radio button.
   */
  
  public void setText(String text)
  {
    getComponent().setText(text);
    invalidateSize();
  }
  
  /**
   * Get the RadioButtonGroup this button is part of.
   */
  
  public RadioButtonGroup getGroup()
  {
    return group;
  }
  
  /**
   * Set the RadioButtonGroup this button is part of.
   */
  
  public void setGroup(RadioButtonGroup newGroup)
  {
    for (int i = 0; i < group.getRadioButtonCount(); i++)
      if (group.getRadioButton(i) == this)
      {
        group.remove(i);
        break;
      }
    group = newGroup;
    group.add(this);
  }
  
  /**
   * Get the largest size at which this Widget can reasonably be drawn.  When a WidgetContainer lays out
   * its contents, it will attempt never to make this Widget larger than its maximum size.
   */
  
  public Dimension getMaximumSize()
  {
    return new Dimension(32767, 32767);
  }
}
