# Copyright (C) 2005, 2006 by Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""Submit an email to a Patch Queue Manager"""

import smtplib

import bzrlib
from bzrlib.branch import Branch
import bzrlib.config
from bzrlib.errors import BzrCommandError
import bzrlib.gpg
import bzrlib.osutils
from bzrlib.trace import note
import bzrlib.ui
import bzrlib.urlutils


_default_smtp_server = 'localhost'


def _signed_submission(branch, config, target_base):
    """Get the signed text that we would submit to the pqm"""

    merge_target = config.get_user_option('pqm_branch')
    if not merge_target:
        raise BzrCommandError('No PQM target branch specified '
                              'in configuration')

    unsigned_text = 'star-merge %s %s' % (target_base, merge_target)

    strategy = bzrlib.gpg.GPGStrategy(config)

    return strategy.sign(unsigned_text)


_default_headers = '''From: %s
To: %s
Subject: %s

''' 


def _setup_headers(message, user_from, user_to):
    """Get all of the important headers filled out."""

    return _default_headers % (user_from
            , user_to, message)


def public_branch(branch, config):
    target_base = config.get_user_option('public_branch')
    if target_base is None:
        repo_loc = branch.repository.bzrdir.root_transport.local_abspath('.')
        repo_config = bzrlib.config.LocationConfig(repo_loc)
        public_repo = repo_config.get_user_option("public_repository")
        if public_repo is not None:
            branch_relpath = bzrlib.osutils.relpath(repo_loc,
                branch.bzrdir.root_transport.local_abspath('.'))
            target_base = bzrlib.urlutils.join(public_repo, branch_relpath)
    return target_base


def submit(branch, message, dry_run=False, public_location=None):
    """Submit the given branch to the pqm."""
    config = branch.get_config()

    if message is None:
        rev = branch.repository.get_revision(branch.last_revision())
        message = rev.message
    message = message.encode('utf8')

    user_from = config.get_user_option('pqm_user_email')
    if not user_from:
        user_from = config.username()
    user_from = user_from.encode('utf8') # Make sure this isn't unicode
    user_to = config.get_user_option('pqm_email')
    if not user_to:
        raise BzrCommandError('No PQM submission address specified '
                              'in configuration')
    user_to = user_to.encode('utf8') # same here

    # All the entries must be utf-8 so the message creation doesn't fail
    msg = _setup_headers(message, user_from, user_to)

    server = config.get_user_option('smtp_server')
    if not server:
        server = _default_smtp_server
    smtp_username = config.get_user_option('smtp_username')
    smtp_password = config.get_user_option('smtp_password')

    if public_location is None:
        target_base = public_branch(branch, config)
    else:
        target_base = public_location
    note('Checking that the public branch is up to date ...')
    public_revision = Branch.open(target_base).last_revision()
    if public_revision != branch.last_revision():
        raise BzrCommandError(
            'Local last revision does not match public last revision: %s' % 
            (target_base,))
    # Get the signed text, this should be done just before
    # emailing, so that we don't run gpg --cl for nothing
    msg += _signed_submission(branch, config, target_base)

    if dry_run:
        return print_info(msg, server, user_from, user_to)

    smtp = smtplib.SMTP()
    smtp.connect(server)

    try:
        # The world can always use a little bit more encryption :)
        smtp.starttls()
    except smtplib.SMTPException:
        pass

    if smtp_username is not None:
        if smtp_password is None:
            smtp_password = bzrlib.ui.ui_factory.get_password(
                'Please enter SMTP password for %(host)s', host=server)
        try:
            smtp.login(smtp_username, smtp_password)
        except smtplib.SMTPHeloError, e:
            raise BzrCommandError('SMTP server refused HELO: %d %s'
                                  % (e.smtp_code, e.smtp_error))
        except smtplib.SMTPAuthenticationError, e:
            raise BzrCommandError('SMTP server refused authentication: %d %s'
                                  % (e.smtp_code, e.smtp_error))
        except smtplib.SMTPException, e:
            raise BzrCommandError(str(e))

    smtp.sendmail(user_from, [user_to], msg)


def print_info(msg, server, user_from, user_to):
    print "Server: %s" % server
    print "%s" % msg
