(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Protocol for communicating, a class
   implementing send and receive with 
   Unix unconnected sockets and a class to implement it with connected sockets.

   The protocol is the following. A message is:
   <version>\n
   <source_id>\n
   <source_host>\n
   <source_port>\n
   <dest_id>\n
   <message_type>\n
   [<message_room_name>]\n
   [<message_room_people_number>]\n
   [(<message_room_people_id>\n
     <message_room_people_host>\n
     <message_room_people_port>\n)+]
   [<message_body_length>]\n
   [<message_body>]
   [<file_length>]\n
   [<file_name>]
   [<id>\n
    <host>\n
    <port>]

   where <message_type> can be
   - HELLO
   - HELLO_OK
   - BYEBYE
   - MESSAGE
   - ADD_OPEN
   - ROOM_MESSAGE
   - FILE

   <message_body_length> appears only when the <message_type> is MESSAGE or ROOM_MESSAGE.
   <message_body> appears only after the <message_body_length>.
   <file_length> appears only when the <message_type> is FILE.
   <file_name> appears only after the <file_length>.

   [<id>\n
    <host>\n
    <port>] appears only for a ADD_OPEN <message_type>
 *)

type port = int
type host = string
type address = host * port

type message = string
type version = string
type id = string

type source = version * id * address
type dest = id 


type proto = 
  | Hello
  | HelloOk
  | Byebye  (** to signal that we are disconnecting *)
  | Message of message
  | AddOpen of id * address (** to remotely add a user *)
  | RoomMessage of id * (id * host * port) list * message 
      (** id of the room, people in the room and message *)
  | File of int * string (** size and name of the file proposed *)


type info = source * dest * proto (** source, destination, proto *)

let il input = 
  let l = input () in
  Chat_misc.remove_blanks l

(** read a info with the given functions. 
   @raise Failure if an error occurs (bad format, enf of file, ...). *)
let read_info getline input =
  try
    let v = il getline in
    let source_id = getline () in
    let source_host = getline () in
    let source_port = int_of_string (il getline) in
    let dest_id = getline () in
    let message_type = il getline in
    let proto = 
      match String.uppercase message_type with
      | "HELLO" -> Hello
      | "HELLO_OK" -> HelloOk
      | "BYEBYE" -> Byebye
      | "MESSAGE" ->
	  let length = int_of_string (il getline) in
	  let s = String.create length in
	  let n = input s 0 length in
	  Message s
      | "FILE" ->
	  let length = int_of_string (il getline) in
	  let name = getline () in
	  File (length,name)

      |	"ROOM_MESSAGE" ->
	  let name = getline () in
	  let n = int_of_string (il getline) in
	  let rec iter acc m = 
	    if m < n then
	      iter 
		(
		 let source_id = getline () in
		 let source_host = getline () in
		 let source_port = int_of_string (il getline) in
		 (source_id, source_host, source_port) :: acc
		)
		(m+1)
	    else
	      List.rev acc
	  in
	  let people = iter [] 0 in
	  let length = int_of_string (il getline) in
	  let s = String.create length in
	  let n = input s 0 length in
	  RoomMessage (name, people, s)

      | "ADD_OPEN" ->
	  let id = il getline in
	  let host = il getline in
	  let port = int_of_string (il getline) in
	  AddOpen (id, (host, port))
      |	_ ->
	  raise (Failure "Bad message type")
    in
    let source = (v, source_id, (source_host, source_port)) in
    (source, dest_id, proto)
  with
    End_of_file -> raise (Failure "End_of_file")
  | Invalid_argument "int_of_string" -> raise (Failure "Bad format")

let read_info_channel inch =
  read_info (fun () -> input_line inch) (input inch)

let read_info_buffer buf =
  read_info (fun () -> Chat_misc.buf_get_line buf) (Chat_misc.buf_input buf)


let ol buf s = Printf.bprintf buf "%s\n" s

(** write the given proto to the given buffer. *)
let write_info buf info =
  let (source, dest_id, proto) = info in
  let (v, source_id, (source_host, source_port)) = source in
  let p = ol buf in
  let p2 s = p (Chat_misc.remove_newlines s) in
  p2 v;
  p2 source_id ;
  p2 source_host ;
  p2 (string_of_int source_port) ;
  p2 dest_id ;
  match proto with
  | Hello -> p "HELLO"
  | HelloOk -> p "HELLO_OK"
  | Byebye -> p "BYEBYE"
  | Message s -> 
      p "MESSAGE"; 
      let l = String.length s in 
      p (string_of_int l);
      p s
  | AddOpen (id, (h, port)) ->
      p "ADD_OPEN";
      p2 id ;
      p2 h ;
      p2 (string_of_int port)
  | RoomMessage (name, people, s) ->
      p "ROOM_MESSAGE";
      p name ;
      p (string_of_int (List.length people));
      List.iter
	(fun (i,h,port) ->
	  p2 i ;
	  p2 h ;
	  p2 (string_of_int port)
	)
	people;
      let l = String.length s in 
      p (string_of_int l);
      p s
  | File (len,name) ->
      p "FILE";
      p (string_of_int len);
      p2 name

(** write the given proto to the given channel. *)
let write_info_channel ouch info =
  let buf = Buffer.create 256 in
  write_info buf info;
  output_string ouch (Buffer.contents buf);
  Pervasives.flush ouch

(*

(** The classes used to send and receive messages. *)
class type com =
  object
    (** Add a connection to the table of connections. *)
    method add_connection : address -> Unix.file_descr -> unit

    (** Free all what must be freed when the app is closed. *)
    method close : unit

    (** [send adr mes] sends the message [mes] to the
       application at address [adr].
       Should raise Failure with an error message if an
       error occurs.*)
    method send : id -> address -> proto -> unit

    (** [receive] returns an optional message, if
       one was pending.
       Should raise Failure with an error message if an
       error occurs.*)
    method receive : (info, Unix.file_descr) option
  end

let cpt = ref 1 

(** This class implements the com interface,
   with Unix connected sockets.
   It needs a {!Chat_config.config} class to
   access config options.*)
class tcp conf =
  let localhost = Unix.gethostname () in
  let sock_addr = Unix.ADDR_INET (inet_addr_any, conf#port) in
  let iptable = Hashtbl.create 13 in
  let socket_server = Unix.socket Unix.PF_INET Unix.SOCK_STREAM 0 in
  let _ =  Unix.set_nonblock sock in
  object (self)
    val instance = (incr cpt; !cpt)

    val mutable connections = Hashtbl.create 13

    method add_connection adr sock =
      Hashtbl.remove connections adr;
      Hashtbl.add connections adr sock;

    method private source =
      (version, conf#id, (localhost, conf#port))

    method close = Unix.close socket_server

    (** Return [true] if we got connected. *)
    method private connect sock sockaddr =
      Unix.set_nonblock sock ;
      try
	Unix.connect sock sockaddr; 
	Unix.clear_nonblock sock ;
	true
      with
	Unix.Unix_error (e, _,_ ) ->
	  match e with 
	    Unix.EWOULDBLOCK | Unix.EINPROGRESS ->
	      let res = 
		match Unix.select [] [sock] [] conf#second_send_timeout with
		| (_, [s], _) when s = sock-> 
		    (
		     try ignore(Unix.getpeername s) ; true
		     with Unix.Unix_error (_, _,_ ) -> false
		    )
		| _ -> false
	      in
	      Unix.clear_nonblock sock ;
	      res
	  | _ ->
	      false

    method private create_connection adr =
      let (host, port) = adr in
      let sock = Unix.socket AF_INET Unix.SOCK_STREAM 0 in
      let ip =
	try
	  Some (Hashtbl.find iptable host)
	with Not_found ->
	  try
	    let h = Unix.gethostbyname host in
	    Some (h.Unix.h_addr_list.(0))
	  with
	    Not_found -> raise (Failure (M.unknown_host host))
      in
      let sockaddr = Unix.ADDR_INET (ip, port) in
      try
	if self#connect sock sockaddr then
	  sock
	else
	  raise (Failure (M.could_not_connect host port))
      with
      | Unix.Unix_error (e,s1,s2) ->
	  let s = s1^" "^s2^" : "^(Unix.error_message e) in
	  raise (Failure s)

    method private new_connection ?(temp=false) adr =
      if not temp then
	try
	  Hashtbl.find connections adr
	with
	  Not_found ->
	    let sock = self#create_connection adr in
	    self#add_connection adr sock;
	    sock
      else
	self#create_connection adr
	
    (** Send a message.
       @param temp_connection indicate if we must create a new connection
       and not keep it in the table of connections. Used to transfer files.*)
    method send ?(temp_connection=false) id adr mes =
      let (host, port) = adr in
      if host = localhost && port = conf#port then
	raise (Failure Chat_messages.dest_is_source)
      else
	(
	 let domain = Unix.PF_INET in
	 let sock = Unix.socket domain Unix.SOCK_STREAM 0 in
	 let ip_opt =
	   try
	     Some (Hashtbl.find iptable host)
	   with Not_found ->
	     try
	       let h = Unix.gethostbyname host in
	       Some (h.Unix.h_addr_list.(0))
	     with
	       Not_found -> None
	 in
	 match ip_opt with
	   None -> ()
	 | Some ip ->
	     let sockaddr = Unix.ADDR_INET (ip, port) in
	     try
	       let paq = (self#source, id, mes) in
	       if self#connect sock sockaddr then
		 (
		  let chanout = Unix.out_channel_of_descr sock in
		  write_info_channel chanout paq;
		  flush chanout;
		  close_out chanout
		 )
	       else
		 ()
	     with
	     | Unix.Unix_error (e,s1,s2) ->
		 let s = s1^" "^s2^" : "^(Unix.error_message e) in
		 raise (Failure s)
	)

    method receive =
      try
	let (desc, addr) = Unix.accept socket_server in
	let host =
	  match addr with
	    Unix.ADDR_INET (addr,_) -> Unix.string_of_inet_addr addr
	  | _ -> failwith "not receiving data from a socket" in
	let _ = Chat_messages.verbose ("receive from " ^ host) in
	let chanin = Unix.in_channel_of_descr desc in
	let chanout = Unix.out_channel_of_descr desc in
	let paq = 
	  match read_info_channel chanin with 
            ((v,id,(_,port)),iddest,pro) -> 
	      ((v,id,(host, port)),iddest,pro) in
	let ret = 
	  match paq with
            ((v,id,(host, port)),iddest,pro) ->
	      if v <> version then 
		(
		 Chat_messages.verbose (Chat_messages.incompatible_versions v version);
		 close desc;
		 None;
		)
	      else
		(
		 match addr with
		   Unix.ADDR_INET (a,_) ->
		     (
		      try
			let old_a = Hashtbl.find iptable host in
			if old_a = a then
			  ()
			else
			  (
			   Hashtbl.remove iptable host ;
			   raise Not_found
			  )
		      with
			Not_found ->
			  Hashtbl.add iptable host a
		     );
		     Some (paq, desc)
		 | _ -> 
		     close desc;
		     None
		)
	in
	ret
      with
	Unix.Unix_error (Unix.EWOULDBLOCK,_,_)
      | Unix.Unix_error (Unix.EAGAIN,_,_) ->
	  None
      | Unix.Unix_error (e,s1,s2) ->
	  let s = (Unix.error_message e)^" :"^s1^" "^s2 in
	  raise (Failure s)
      |	Failure s ->
	  raise (Failure s)
      |	e ->
	  let s = Printexc.to_string e in
	  raise (Failure s)

    initializer
      Unix.setsockopt socket_server Unix.SO_REUSEADDR true ;
      Unix.set_nonblock socket_server ;
      try
	Unix.bind socket_server sock_addr ;
	Unix.listen socket_server 15;
      with
      | Unix.Unix_error (e,s1,s2) ->
	  let s = (Unix.error_message e)^" :"^s1^" "^s2 in
	  raise (Failure s)
  end
*)
