(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Convenient functions to generate HTML. *)

(** {2 Documents} *)

(** You should not rely on the doc definition, but rather use the
   {!Toolhtml.doc_of_report} to get a [doc] from a [Report.report],
   and {!Toolhtml.report_of_doc} to get a [Report.report] from a [doc]. *)
type doc = int Report.report

(** Return a {!Toolhtml.doc} from a string. This string should
   valid HTML. It won't be changed and put as-is in the final document.*)
val doc_of_html : string -> doc

(** Homer Simpson's way to call {!Toolhtml.doc_of_html}. *)
val doh : string -> doc

(** Get a {!Toolhtml.doc} from a {!Report.report} description. *)
val doc_of_report : 'a Report.report -> doc

(** Get a {!Report.report} description from a {!Toolhtml.doc}.*)
val report_of_doc : doc -> int Report.report

(** The empty document.*)
val empty : doc

(** {2 Documents composition} *)

(** [seq [doc1 ; doc2 ; .. ; docn]] returs a document composed of
  the given documents one after another.*)
val seq : doc list -> doc

(** [concat sep l] returns a document composed of the given list of documents [l]
 separated by the [sep] document. *)
val concat : doc -> doc list -> doc

(** {2 Simple piece of documents} *)

(** [tag t [(attribute1, value1) ; (attribute2, value2) ; ...] doc]
   creates a document where the given [doc] is encapsultated in
   a new tag [t], with the given attributes, in the form
   [<t attribute1="value1" attribute2="value2" ...>doc</t>].
   Note that there is no closing tag if it is not needed in HTML 
   (for example : "br" tags).*)
val tag : string -> (string * string) list -> doc -> doc

(** [link url doc] creates a document representing a link
   in the form [<a href="url">doc</a>]. 
   @param target optional target value for the "target" attribute of the "a" tag.*)
val link : string -> ?target: string -> doc -> doc

val br : doc

val tr : ?atts: (string * string) list -> doc list -> doc

val td : ?w: string -> ?atts: (string * string) list -> doc -> doc
val td_100 : ?atts: (string * string) list -> doc -> doc
val td_top : ?w: string -> ?atts: (string * string) list -> doc -> doc

val table  : ?w: string -> ?atts: (string * string) list -> doc list -> doc
val table_100 : ?atts: (string * string) list -> doc list -> doc

val ul : ?atts: (string * string) list -> doc list -> doc
val ul_li : doc list -> doc
val ol : ?atts: (string * string) list -> doc list -> doc
val ol_li : doc list -> doc

val h1 : ?atts: (string * string) list -> doc -> doc
val h2 : ?atts: (string * string) list -> doc -> doc
val h3 : ?atts: (string * string) list -> doc -> doc
val h4 : ?atts: (string * string) list -> doc -> doc
val h5 : ?atts: (string * string) list -> doc -> doc
val h6 : ?atts: (string * string) list -> doc -> doc

val span : ?cl: string -> ?atts: (string * string) list -> doc -> doc
val div : ?cl: string -> ?atts: (string * string) list -> doc -> doc

val p : doc -> doc

(** {2 Complex documents} *)

(** The css names used by complex documents. *)
type  css_names =
    {
      section_table : string ;
      section_title : string ;
      subsection_table : string ;
      subsection_title : string ;
      elements : string ;
      row : string array;
    } 

(** Return the current css names. *)
val css_names : unit -> css_names

(** Change the current css names. *)
val set_css_names : css_names -> unit

(** Return the css code to use for the given css names, for
  the complex documents to look ok. *)
val css_code : css_names -> string

(** [frame_table rows] builds a document with the given rows
   in a table, using two tables to create a nive border.
   [rows] are a list of documents which must contain the [TD] tags.
   @param title optional title
   @param width optional width for the top table (default is ["100%"])
*)
val frame_table : ?title: string -> ?width: string -> doc list -> doc

(** [list_in_table f l] builds a document representing a list [l], 
   using HTML tables. [f] is used to get the list [[(size1, doc1) ; ...]] 
   of documents for the columns of each row. [size] is a string
   being put in the "width" attribute of the "td" tag of the column.
   @param title the optional title of the list (can be any valid HTML code).
   @param sep indicate if we must insert space rows.*)
val list_in_table : ?css: css_names -> ?title: string -> ?sep: bool ->
  ('a -> (string * doc) list) -> 'a list -> doc

(** [double_list_in_table [((title, span), (f, l)); ...]]
   builds a document representing a list of lists, using
   HTML tables.
   For each list we give the [title], on how many columns the title spans,
   and a pair [(f, l)] as in the {!Toolhtml.list_in_table} function.
   @param title the optional title for the list of lists.
*)
val double_list_in_table :
    ?css: css_names -> ?width: string -> ?title: string -> 
      ((string * int) * 
	 (('a -> (string * doc) list) * 'a list)) list -> doc

(** [vframeset [(size1,name1,url1); (size2,name2,url2); ...]] 
   generates a vertically framed doc, with [name1] and [page1] as name and page 
   for the first frame, [name2] and [doc2] the name and page
   for the second frame and so on. The [size] value for each frame is the size
   of the frame of the form ["20%"] or any form allowed in HTML.*)
val vframeset : (string * (string * string)) list -> doc

(** Same as {!Toolhtml.vframeset} but generate an horizontal framed doc. *)
val hframeset : (string * (string * string)) list -> doc

(** [tree l f_doc f_children] creates a document representing
   trees. The given list contains the roots of the trees. [f_doc]
   returns the document to insert fo each node, while [f_children]
   returns the list of the given node.*)
val tree : 'a list -> ('a -> doc) -> ('a -> 'a list) -> doc

(** Same as {!Toolhtml.tree} but put the tree in a table. 
   @param title the title passed to the {!Toolhtml.list_in_table} function
*)
val tree_in_table : ?css: css_names -> ?title: string -> 'a list -> ('a -> doc) -> ('a -> 'a list) -> doc

(** [page title body] creates a documents representing a HTML page, with
   the given title and body. The body does not contain the "body" tag
   so you can define a frameset, for example.
   @param style use the style referenced by the given url
   @param more_head can be used to add HTML code to the header *)
val page : ?style: string -> ?more_head: string -> string -> doc -> doc

(** {2 forms} *)

type form_method = Get | Post

(** [form ?method action body] creates a form with the
   given [method] (default is [Post]) and [action], 
   containing the given [body].*)
val form : ?met: form_method -> string -> doc -> doc

(** [submit_button label] create a form submit button with
   the given label for the button. *)
val submit_button : string -> doc

(** [reset_button label] create a form reset button with
   the given label for the button. *)
val reset_button : string -> doc

(** [select name choices default] creates a [select] tag, to create a HTML combo in a form. *)
val select : string -> (string * string) list -> string -> doc

type input_type = 
    Text | Checkbox | Radio | Password | Submit | Reset | Hidden

(** [text_field typ ?value varname] creates a form field with
   given [typ], using the given variable name [varname], eventually
   adding a value for the [value] attribute if specified. 
   The [value] string is escaped by the function, using
   {!Toolhtml.escape_quotes} and {!Toolhtml.escape_entities}.
*)
val input : input_type -> ?size: string -> ?checked: bool -> ?value: string -> string -> doc

(** {2 Useful functions} *)

(** Replace the quotes in a string by "&#34;" *)
val escape_quotes : string -> string

(** Replace '<', '>' and '&' characters by their HTML code
   ("&lt;", "&gt;", "&amp;").*)
val escape_entities : string -> string

(** {2 Generating code of documents} *)

val compute : Format.formatter -> doc -> unit
(** Compute a document and print it to the given formatter. *)

val compute_file : string -> doc -> unit
(** Compute a document and print it in a file.*)

val html_of_doc : doc -> string
(** Compute a document and return the resulting string. *) 
