(*********************************************************************************)
(*                Cameleon                                                       *)
(*                                                                               *)
(*    Copyright (C) 2004-2008 Institut National de Recherche en Informatique     *)
(*    et en Automatique. All rights reserved.                                    *)
(*                                                                               *)
(*    This program is free software; you can redistribute it and/or modify       *)
(*    it under the terms of the GNU Library General Public License as            *)
(*    published by the Free Software Foundation; either version 2 of the         *)
(*    License, or any later version.                                             *)
(*                                                                               *)
(*    This program is distributed in the hope that it will be useful,            *)
(*    but WITHOUT ANY WARRANTY; without even the implied warranty of             *)
(*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *)
(*    GNU Library General Public License for more details.                       *)
(*                                                                               *)
(*    You should have received a copy of the GNU Library General Public          *)
(*    License along with this program; if not, write to the Free Software        *)
(*    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA                   *)
(*    02111-1307  USA                                                            *)
(*                                                                               *)
(*    Contact: Maxence.Guesdon@inria.fr                                          *)
(*                                                                               *)
(*********************************************************************************)

let oug_filename_var = "ougsource";;

let oug_data = ref (Ouglib.Data.create(), 0.0)

module D = Ouglib.Data

type file_elts = (int * int * D.elt_id) list
  (* list of (start_pos, end_pos, elt_id) *)

type loc_table = (string, file_elts) Hashtbl.t;;
let (loc_table : loc_table) = Hashtbl.create 111;;

let rec add_to_file_elts start_pos end_pos elt_id = function
  [] -> [start_pos, end_pos, elt_id]
| (sp, ep, _) as x :: q ->
    if start_pos < sp || (start_pos = sp && end_pos > ep) then
      (start_pos, end_pos, elt_id) :: x :: q
    else
      x :: (add_to_file_elts start_pos end_pos elt_id q)
;;

let fill_loc_table oug_data =
  Hashtbl.clear loc_table ;
  let f elt_id elt =
    match elt.D.kind with
      D.Class -> ()
    | _ ->
        let fname = elt.D.loc.D.loc_start.Lexing.pos_fname in
        let fname = Filename.basename fname in
        let l =
          try Hashtbl.find loc_table fname
          with Not_found -> []
        in
        let start_pos = elt.D.loc.D.loc_start.Lexing.pos_cnum in
        let end_pos = elt.D.loc.D.loc_end.Lexing.pos_cnum in
        let l = add_to_file_elts start_pos end_pos elt_id l in
        Hashtbl.replace loc_table fname l
  in
  Ouglib.Earray.iteri f oug_data.D.elements
;;

let rec elt_id_by_loc pos = function
  [] -> raise Not_found
| (sp, ep, elt_id) :: q ->
    if sp <= pos && pos <= ep then
      elt_id
    else
      elt_id_by_loc pos q
;;

let on_cursor f filename pos =
  (* FIXME: handle mode display mapping *)
  let filename = Filename.basename filename in
  try
    let l = Hashtbl.find loc_table filename in
    let elt_id = elt_id_by_loc pos l in
    let data = fst !oug_data in
    let elt = Ouglib.Earray.get data.D.elements elt_id in
    f elt_id elt
  with
    Not_found ->
      let msg = Printf.sprintf
        "No element in file \"%s\", character %d" filename pos
      in
      prerr_endline msg
;;

let on_active_sourceview f args =
  match !Ed_sourceview.active_sourceview with
    None -> ()
  | Some v ->
      let b = v#source_buffer in
      let iter = b#get_iter `INSERT in
      let pos = iter#offset in
      on_cursor f v#filename pos
;;

let print =
  let f  _ elt =
    Ed_misc.display_message
      (Ed_misc.to_utf8 (D.string_of_absname elt.D.name))
  in
  on_active_sourceview f
;;

module L = Ouglib.Lang ;;
let goto_prev =
  let f (elt_id : D.elt_id) (_ : D.elt) =
    let data = fst !oug_data in
    let filter = L.Ref ([], L.Internal_set [elt_id], None, Some 1) in
    let elt_ids = L.filter_elements data filter in
    let elts = List.map (Ouglib.Earray.get data.D.elements) elt_ids in
    let entries = List.map
      (fun elt ->
         let label =
           Cam_misc.escape_menu_label
             (Ed_misc.to_utf8 (D.string_of_absname elt.D.name))
         in
         let fname = elt.D.loc.D.loc_start.Lexing.pos_fname in
         let go =
         match fname with
             "" ->
               (fun () ->
                  Ed_misc.display_message (Ed_misc.to_utf8 "No filename for this element.")
               )
           | _ ->
               fun () ->
                 let line = elt.D.loc.D.loc_start.Lexing.pos_lnum in
                 let char = elt.D.loc.D.loc_start.Lexing.pos_cnum -
                   elt.D.loc.D.loc_start.Lexing.pos_bol
                 in
                 let com = Printf.sprintf "open_file %s %d,%d"
                   fname (line-1) char
                 in
                 Cam_commands.eval_command com
         in
         `I (label, go)
      )
      elts
    in
    GToolbox.popup_menu ~entries ~button: 1 ~time: Int32.zero
  in
  (on_active_sourceview f : string array -> unit)
;;

let load_data file =
  let data = Ouglib.Dump.load_data file in
  oug_data := (data, Unix.time());
  fill_loc_table data;;

let rec watch_load_data file () =
  try
    let mod_date = (Unix.stat file).Unix.st_mtime in
    let (_,date) = !oug_data in
    if date < mod_date then
    (try load_data file with _ -> ());
    ignore(GMain.Timeout.add ~ms: 5000 ~callback: (watch_load_data file));
    false
  with
    Unix.Unix_error _ -> false
;;

let start_oug_watch ?file () =
  let file =
    match file with
      Some f -> Some f
    | None ->
        try Some (Cam_commands.get_global oug_filename_var)
        with Not_found -> None
  in
  match file with
    None -> ()
  | Some file -> ignore(watch_load_data file ())
;;

let com_start_oug args =
  let file = if Array.length args > 0 then Some args.(0) else None in
  start_oug_watch ?file ()
;;

let _ =
  let com = Cam_commands.create_com
    "oug_start_watch" [| "ougfile"|] com_start_oug
  in
  Cam_commands.register com;;

let _ =
  let com = Cam_commands.create_com
    "oug_lookup" [| |] print
  in
  Cam_commands.register com;;

let _ =
  let com = Cam_commands.create_com
    "oug_goto_prev" [| |] goto_prev
  in
  Cam_commands.register com;;



  