/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "CleanPolyData.h"

#include <Application.h>
#include <MeshComponent.h>
#include <Log.h>

#include <VtkMeshUtil.h>

#include <vtkCleanPolyData.h>
#include <vtkCallbackCommand.h>

using namespace camitk;


// -------------------- CleanPolyData --------------------
CleanPolyData::CleanPolyData(ActionExtension* extension) : Action(extension) {
    // Setting name, description and input component 
    setName("Clean Mesh");
    setDescription("Merge duplicate points, and/or remove unused points and/or remove degenerate cells.");
    setComponent("MeshComponent");
    
    // Setting classification family and tags
    setFamily("Mesh Processing");
    addTag("vtkCleanPolyData");
    addTag("Clean vtkPolyData");
    
    setProperty("Tolerance", QVariant(1.0)); // Tolerance is given in percentage of the bounding box, if points are closer than 1.0% of the bounding box dimensions, then they are merged
}


// --------------- apply -------------------
Action::ApplyStatus CleanPolyData::apply() {
    CAMITK_INFO("CleanPolyData", "apply", "Merge duplicate points " << getTargets().last()->getName().toStdString());

    // set waiting cursor and status bar
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    Application::showStatusBarMessage("Cleaning poly data...");
    Application::resetProgressBar();

    // use the last target
    MeshComponent *targetMesh = dynamic_cast<MeshComponent*>(getTargets().last());

    //-- check if this is a polydata and if this is an unstructured grid, extract the surface
    vtkSmartPointer<vtkPolyData> targetPolyData = VtkMeshUtil::vtkPointSetToVtkPolyData(targetMesh->getPointSet());
    
    //-- clean the polydata
    // remove redundant points
    vtkSmartPointer<vtkCleanPolyData> cleaner = vtkSmartPointer<vtkCleanPolyData>::New();
    cleaner->AddInput ( targetPolyData );
    cleaner->PointMergingOn();
    cleaner->SetTolerance ( property("Tolerance").toDouble() *0.01 );  // % of the Bounding Box
    cleaner->ConvertLinesToPointsOn ();
    cleaner->ConvertPolysToLinesOn ();
    cleaner->ConvertStripsToPolysOn ();
    
    vtkSmartPointer<vtkCallbackCommand> progressCallback = vtkSmartPointer<vtkCallbackCommand>::New();
    progressCallback->SetCallback(&Application::vtkProgressFunction);
    cleaner->AddObserver(vtkCommand::ProgressEvent, progressCallback);

    // lets do it!
    cleaner->Update();

    // Create a mesh Component
    vtkSmartPointer<vtkPointSet> resultPointSet = cleaner->GetOutput();
    new MeshComponent(resultPointSet, targetMesh->getName() + " clean");
    Application::refresh();    

    // restore the normal cursor and progress bar
    Application::resetProgressBar();
    QApplication::restoreOverrideCursor();

    return SUCCESS;
}
