/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/


#ifndef EXTENSIONS_MANAGER_H
#define EXTENSIONS_MANAGER_H

// -- Core stuff
#include "ComponentExtension.h"
#include "ActionExtension.h"
#include "CamiTKAPI.h"
#include "AbortException.h"

// -- QT stuff
#include <QtPlugin>
#include <QPluginLoader>
#include <QStringList>
#include <QMap>

namespace camitk {
/** This class is used to manage all plugins loaded by the application.
 *
 *  The current version is able to load dynamic library for
 *  - ComponentExtension
 *  - ActionExtension
 *
 *  This class is a contained for all the loaded extension. It contains only
 *  static members.
 */
class CAMITK_API ExtensionManager : public QObject {

public:

    /// Autoload component and action extensions
    static void autoload();
    
    /** @name Component extension list management
       */
    ///@{
    /// autoload all the ComponentExtension plugins (use applicationDirPath/components)
    static void autoloadComponentExtensions();

    /// load a ComponentExtension plugin using a .dll/.so/.dylib filename
    /// @param file the plugin filenamae (.dll/.so/.dylib)
    /// @return false if it cannot be loaded
    static bool loadComponentExtension(QString file);

    /// unload a plugin corresponding to a given extension or name
    /// @return false if the plugin could not be unloaded - still in use somewhere
    static bool unloadComponentExtension(QString);

    /// get the plugin corresponding to a given extension or name (const, you can only use this method to get information, use open to open a file using a specific plugin)
    static const ComponentExtension * getComponentExtension(QString);

    /** get all the registered component extensions that can manage file (default component extension)
      * This is the public method (return a const, the extensions map is private
      * and cannot be modified externally).
      */
    static const QMap<QString, ComponentExtension*> & getComponentExtensions();

    /** get all the registered component extensions that can manage entire directories
     * This is the public method (return a const, the extensions map is private
      * and cannot be modified externally).
      */
    static const QMap<QString, ComponentExtension*> & getDataDirectoryComponents();

    /// get the list of all the suffixes managed by registered component extensions (all possible file suffix)
    static QStringList getFileExtensions();

    /// get the list of all the name of the registered Component data directory
    static QStringList getDataDirectoryExtNames();

    ///@}

    
    /// @name Action extension list management
    ///@{

    /// autoload all the action extensions (use applicationDirPath/actions)
    static void autoloadActionExtensions();

    /// unload all action extensions and delete instanciated actions
    static void unloadAllActionExtensions();
    
    /// load an action extension using a .dll/.so/.dylib filename
    /// @return false if it cannot be loaded
    static bool loadActionExtension(QString);

    /// unload an action extension using its .dll/.so/.dylib filename
    static bool unloadActionExtension(QString);

    /** get all the registered action extension.
     * This is the public method (return a const, the extensions map is private
     * and cannot be modified externally).
     */
    static const QMap<QString, ActionExtension*> & getActionExtensions();
    //@}
    
private:
    /** get the singleton map of loaded component plugins for files (the key is the file extension)
    * This is the private (intern) method.
    * The component extension map is updated by loadExtension, unloadExtension and autoloadExtensions.
    * This method follows the "construct on first use" idiom/design-pattern.
    * It therefore avoids the infamous "static initialization order fiasco",
    * see http://www.parashift.com/c++-faq/ctors.html
    */
    static QMap<QString, ComponentExtension*> & getComponentExtensionMap();

    /** get the singleton map of loaded component plugins for data directory (the key is the name)
    * This is the private (intern) method.
    * The data directory component extension map is updated by loadExtension, unloadExtension and autoloadExtensions.
    * This method follows the "construct on first use" idiom/design-pattern.
    * It therefore avoids the infamous "static initialization order fiasco",
    * see http://www.parashift.com/c++-faq/ctors.html
    */
    static QMap<QString, ComponentExtension*> & getDataDirectoryComponentExtensionMap();

    /** get the singleton map of loaded component plugins for files (the key is the file extension)
     * This is the private (intern) method.
     * The component extension map is updated by loadExtension, unloadExtension and autoloadExtensions.
     * This method follows the "construct on first use" idiom/design-pattern.
     * It therefore avoids the infamous "static initialization order fiasco",
     * see http://www.parashift.com/c++-faq/ctors.html
     */
    static QMap<QString, ActionExtension*> & getActionExtensionMap();
};

}


#endif //EXTENSIONS_MANAGER_H
