/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// C++ stuff
#include <algorithm>

// -- Core stuff
#include "Slice.h"
#include "Log.h"
#include <InteractiveViewer.h>
#include <RendererWidget.h>

// -- vtk stuff
#include <vtkImageData.h>
#include <vtkProperty.h>
#include <vtkViewport.h>

using namespace std;


namespace camitk {
/* Constructor */
Slice::Slice(vtkSmartPointer<vtkImageData> volume, PossibleOrientation orientation, vtkSmartPointer<vtkWindowLevelLookupTable> lookupTable) {
    sliceOrientation = orientation;
    this->lut = lookupTable;
    init();

    setOriginalVolume(volume);

    setSlice(getNumberOfSlices() / 2);

}

void Slice::init() {
    currentSliceIndex = 0;

    extent[0] = 0;
    extent[1] = 0;
    extent[2] = 0;
    extent[3] = 0;
    extent[4] = 0;
    extent[5] = 0;

    reslicedDimensions[0] = 0;
    reslicedDimensions[1] = 0;
    reslicedDimensions[2] = 0;

    originalSpacing[0] = 0.0;
    originalSpacing[1] = 0.0;
    originalSpacing[2] = 0.0;

    resliceSpacing[0] = 0.0;
    resliceSpacing[1] = 0.0;
    resliceSpacing[2] = 0.0;

    originalDimensions[0] = 0;
    originalDimensions[1] = 0;
    originalDimensions[2] = 0;

    originalVolume = NULL;

    imgToMapFilter      = NULL;
    image3DActor    = NULL;
    image2DChangeInfo = NULL;
    image2DReslicer   = NULL;
    image2DActor    = NULL;

    sliceCoordsTransform    = NULL;
    transformReslice2Volume = NULL;
    coordsTransform         = NULL;

    pickPlane       = NULL;
    pickPlaneMapper = NULL;
    pickPlaneActor  = NULL;
	pixelActor = NULL;

    imgToMapFilter      = vtkSmartPointer<vtkImageMapToColors>::New();
    image3DActor    = vtkSmartPointer<vtkImageActor>::New();
    image2DReslicer   = vtkSmartPointer<vtkImageReslice>::New();

    image2DChangeInfo = vtkSmartPointer<vtkImageChangeInformation>::New();
    image2DActor    = vtkSmartPointer<vtkImageActor>::New();

    coordsTransform         = vtkSmartPointer<vtkMatrix4x4>::New();
    sliceCoordsTransform    = vtkSmartPointer<vtkMatrix4x4>::New();
    transformReslice2Volume = vtkSmartPointer<vtkTransform>::New();

    pickPlane       = vtkSmartPointer<vtkPlaneSource>::New();
    pickPlaneMapper = vtkSmartPointer<vtkPolyDataMapper>::New();
    pickPlaneActor  = vtkSmartPointer<vtkActor>::New();
}

void Slice::setOriginalVolume( vtkSmartPointer<vtkImageData> volume ) {
    // If there were already a referenced volume,
    //  de-reference the smart pointer.
    originalVolume = volume;

    // Get volume information
    originalVolume->GetDimensions (originalDimensions );
    originalVolume->GetSpacing        (originalSpacing );
    originalVolume->GetExtent     (extent);

    // In other methods, there will be divisions by originalSpacing[k]
    // so let's avoid division by 0
    if (originalSpacing[0] == 0.0)
        originalSpacing[0] = -1.0;

    if (originalSpacing[1] == 0.0)
        originalSpacing[1] = -1.0;

    if (originalSpacing[2] == 0.0)
        originalSpacing[2] = -1.0;

    // Sets the end of the pipeline
    initActors();
}


void Slice::setInitialSlicerTransformation() {
    transformReslice2Volume->Identity();
    currentSliceIndex = 0.0;

    switch (sliceOrientation) {
        default:
        case ARBITRARY_ORIENTATION:
            transformReslice2Volume->RotateX(currentXAngle);
            transformReslice2Volume->RotateY(currentYAngle);
            transformReslice2Volume->RotateZ(currentZAngle);
            transformReslice2Volume->Translate( 0.0, 0.0, currentSliceIndex );

            // Transformation from Reslice Volume to Original Volume
            //TODO
            break;
        case AXIAL_ORIENTATION:
            // Transformation from Reslice Volume to Original Volume
            transformReslice2Volume->Translate( 0.0, 0.0, currentSliceIndex );
            transformReslice2Volume->RotateZ(180.0);
            break;

        case CORONAL_ORIENTATION:
            // Transformation from Reslice Volume to Original Volume
            transformReslice2Volume->Translate( 0.0, 0.0, currentSliceIndex );
            transformReslice2Volume->RotateX( -90.0 );
            transformReslice2Volume->RotateZ(180.0);
            break;

        case SAGITTAL_ORIENTATION:
            // Transformation from Reslice Volume to Original Volume
            transformReslice2Volume->RotateX(90);
            transformReslice2Volume->RotateY(-90);
            transformReslice2Volume->Scale( 1.0, 1.0, -1.0 );
            transformReslice2Volume->Translate( 0.0, 0.0, currentSliceIndex );
            break;

    }

    // Save the basic tranformation from reslice to original.
    coordsTransform->DeepCopy(transformReslice2Volume->GetMatrix());


    // Set tranformation matrix in the vtkImageReslice
    image2DReslicer->SetResliceAxes( transformReslice2Volume->GetMatrix() );

    // Update reslicer
    updateLocalTransformation();

}

void Slice::updateLocalTransformation() {
    image2DReslicer->SetOutputDimensionality(3);
    image2DReslicer->UpdateInformation();
    sliceCoordsTransform = image2DReslicer->GetResliceAxes();

    image2DReslicer->SetOutputDimensionality(2);
    image2DReslicer->UpdateInformation();

}

// --------------------computeReslicedDimensions ------------------
void Slice::computeReslicedDimensions() {
    // Compute the number, size and spacing of slices
    double realDims[4];
    double localDims[4];
    double realSpacing[4];
    double localSpacing[4];

    realDims[0] = (double) originalDimensions[0];
    realDims[1] = (double) originalDimensions[1];
    realDims[2] = (double) originalDimensions[2];
    realDims[3] = 0.0;

    realSpacing[0] = originalSpacing[0];
    realSpacing[1] = originalSpacing[1];
    realSpacing[2] = originalSpacing[2];
    realSpacing[3] = 0.0;

    // Compute data from Original to Reslice.
    coordsTransform->Invert();
    coordsTransform->MultiplyPoint(realDims, localDims);
    coordsTransform->MultiplyPoint(realSpacing, localSpacing);
    coordsTransform->Invert();

    // Reslice dimensions are kept positive and rounded due to previous matrix inverse.
    reslicedDimensions[0] = (localDims[0] < 0) ? (int) (-(localDims[0] - 0.5)) : (int) (localDims[0] + 0.5);
    reslicedDimensions[1] = (localDims[1] < 0) ? (int) (-(localDims[1] - 0.5)) : (int) (localDims[1] + 0.5);
    reslicedDimensions[2] = (localDims[2] < 0) ? (int) (-(localDims[2] - 0.5)) : (int) (localDims[2] + 0.5);

    // Reslice spacings kept positive.
    resliceSpacing[0] = (localSpacing[0] < 0) ? -localSpacing[0] : localSpacing[0];
    resliceSpacing[1] = (localSpacing[1] < 0) ? -localSpacing[1] : localSpacing[1];
    resliceSpacing[2] = (localSpacing[2] < 0) ? -localSpacing[2] : localSpacing[2];
}


void Slice::initActors() {
    imgToMapFilter->SetInput(originalVolume);

    if (NULL==lut) {
        imgToMapFilter->SetLookupTable(0);
    }
    else
        imgToMapFilter->SetLookupTable(lut);

    /* 3D Actor case: directly pluged to the output of imgToMapFilter */
    image3DActor->SetInput(imgToMapFilter->GetOutput());
    image3DActor->InterpolateOn();

    /* 2D Actor: pluged to the output of the reslicer */
    // Put the origin to 0, 0, 0 so that the reslicer could cut at the right place
    //   it also helps to display 2D actor at the right place in 2D viewers.
    image2DChangeInfo->SetInput(imgToMapFilter->GetOutput());
    image2DChangeInfo->SetOutputOrigin(0.0, 0.0, 0.0);
    image2DReslicer->SetInputConnection(image2DChangeInfo->GetOutputPort());
    image2DReslicer->SetInformationInput(imgToMapFilter->GetOutput());
    image2DReslicer->AutoCropOutputOn();
    image2DReslicer->SetOutputOriginToDefault();
    image2DReslicer->SetOutputExtentToDefault();
    image2DReslicer->SetOutputSpacingToDefault();

    image2DActor->SetInput(image2DReslicer->GetOutput());
    image2DActor->InterpolateOn();

    setInitialSlicerTransformation();
    computeReslicedDimensions();

    // Pick plane
    updatePickPlane();

    pickPlaneMapper->SetInputConnection(pickPlane->GetOutputPort());
    pickPlaneActor->SetMapper(pickPlaneMapper);
    pickPlaneActor->GetProperty()->SetRepresentationToWireframe();

    switch (sliceOrientation) {
        default:
        case ARBITRARY_ORIENTATION:
            pickPlaneActor->GetProperty()->SetColor(1.0, 1.0, 1.0);
            pickPlaneActor->GetProperty()->SetEdgeColor(1.0, 1.0, 1.0);
            break;
        case AXIAL_ORIENTATION:
            pickPlaneActor->GetProperty()->SetColor(0.0, 0.0, 1.0);
            pickPlaneActor->GetProperty()->SetEdgeColor(0.0, 0.0, 1.0);
            break;
        case CORONAL_ORIENTATION:
            pickPlaneActor->GetProperty()->SetColor(0.0, 1.0, 0.0);
            pickPlaneActor->GetProperty()->SetEdgeColor(0.0, 1.0, 0.0);
            break;
        case SAGITTAL_ORIENTATION:
            pickPlaneActor->GetProperty()->SetColor(1.0, 0.0, 0.0);
            pickPlaneActor->GetProperty()->SetEdgeColor(1.0, 0.0, 0.0);
            break;
    }

    pickPlaneActor->GetProperty()->SetOpacity(1.0);
    pickPlaneActor->GetProperty()->SetLineWidth(2.0);
}




/* Return the displayed images as a vtkImageData. */
vtkSmartPointer<vtkImageData> Slice::getImageData() const {
    return image2DReslicer->GetOutput();
}

/* Return the vtkImageActor (vtkProp) representing a slice to be displayed in 2D. */
vtkSmartPointer<vtkImageActor> Slice::get2DImageActor() const {
    return image2DActor;
}

/* Return the vtkImageActor (vtkProp) representing a slice to be displayed in 2D. */
vtkSmartPointer<vtkImageActor> Slice::get3DImageActor() const {
    return image3DActor;
}


/* Return the vtkActor used to pick pixels in the slices. */
vtkSmartPointer<vtkActor> Slice::getPickPlaneActor() const {
    return pickPlaneActor;
}



/* This method is called when the associated plane has been picked in the InteractiveViewer,
 *  the given coordinates is position where the plane was picked.
 */
void Slice::pixelPicked(double x, double y, double z) {
    return;
}


void Slice::reslicedToVolumeCoords(const double ijk[3], double xyz[3]) {
    // At this point, coordinates are expressed in reslice frame NOT in real frame.
    double localPoint[4];
    double realPoint[4];

    localPoint[0] = ijk[0];
    localPoint[1] = ijk[1];
    localPoint[2] = ijk[2];
    localPoint[3] = 1.0;

    // From RESLICE to ORIGINAL
    coordsTransform->MultiplyPoint(localPoint, realPoint);

    xyz[0] = realPoint[0];
    xyz[1] = realPoint[1];
    xyz[2] = realPoint[2];
}

void Slice::volumeToReslicedCoords(const double xyz[3], double ijk[3]) {
    // At this point, coordinates are expressed in original frame NOT in real frame.
    double realPoint[4];
    double localPoint[4];

    realPoint[0] = xyz[0];
    realPoint[1] = xyz[1];
    realPoint[2] = xyz[2];
    realPoint[3] = 1.0;

    // From ORIGINAL to RESLICE
    coordsTransform->Invert();
    coordsTransform->MultiplyPoint(realPoint, localPoint);
    coordsTransform->Invert();

    ijk[0] = localPoint[0];
    ijk[1] = localPoint[1];
    ijk[2] = localPoint[2];
}

void Slice::updatePickPlane() {
    double pickPlaneBounds[6];

    //--- Update pick plane
    image2DReslicer->GetOutput()->ComputeBounds();
    image2DReslicer->GetOutput()->Update();
    image2DReslicer->GetOutput()->GetBounds(pickPlaneBounds);

    // At this point, pick plane will match Image2DActor bounds.
    // Be carefull, The center of the first voxel (0,0,0) is displayed at coordinates (0.0, 0.0, 0.0).
    // Pixels winthin borders are represented (in 2D) only by their half, quarter or eigth depending on their coordinates.
    pickPlane->SetOrigin(pickPlaneBounds[0], pickPlaneBounds[2], pickPlaneBounds[4]);
    pickPlane->SetPoint1(pickPlaneBounds[1], pickPlaneBounds[2], pickPlaneBounds[4]);
    pickPlane->SetPoint2(pickPlaneBounds[0], pickPlaneBounds[3], pickPlaneBounds[4]);
    pickPlane->UpdateWholeExtent();
}

/* Return the number of slices in the image data set. */
int Slice::getNumberOfSlices() const {
    int nbSlices = 0;

    switch (sliceOrientation) {
        default:
        case ARBITRARY_ORIENTATION:
            // Set manually the slice to the max of the 3 "normal" slices
            // Vtk correclty handles the "out of bounds" cases.
            nbSlices = max ( max(extent[1] - extent[0], extent[3] - extent[2]), extent[5] - extent[4]) + 1;
            break;
        case AXIAL_ORIENTATION:
            nbSlices = extent[5] - extent[4] + 1;
            break;
        case CORONAL_ORIENTATION:
            nbSlices = extent[3] - extent[2] + 1;
            break;
        case SAGITTAL_ORIENTATION:
            nbSlices = extent[1] - extent[0] + 1;
            break;
    }

    return nbSlices;
}


/* Set the current arbitrary angle.*/
void Slice::setRotationX(double angle) {
    // Check if s is inside the volume bounds
    if ( angle < 0 || angle > 360) {
        angle = 0.0;
    }

    //-- Keep track of current the slice index
    currentXAngle = angle;
    applyRotation();
}

/* Set the current arbitrary angle.*/
void Slice::setRotationY(double angle) {
    // Check if s is inside the volume bounds
    if ( angle < 0 || angle > 360) {
        angle = 0.0;
    }

    //-- Keep track of current the slice index
    currentYAngle = angle;
    applyRotation();

}

/* Set the current arbitrary angle.*/
void Slice::setRotationZ(double angle) {
    // Check if s is inside the volume bounds
    if ( angle < 0 || angle > 360) {
        angle = 0.0;
    }

    //-- Keep track of current the slice index
    currentZAngle = angle;
    applyRotation();

}


void Slice::applyRotation() {
    switch (sliceOrientation) {
        case ARBITRARY_ORIENTATION:
            transformReslice2Volume->Identity();
            transformReslice2Volume->Translate(originalDimensions[0]/2, originalDimensions[1]/2,  originalDimensions[2]/2);
            transformReslice2Volume->RotateZ(currentZAngle);
            transformReslice2Volume->RotateX(currentXAngle);
            transformReslice2Volume->RotateY(currentYAngle);
            transformReslice2Volume->Translate( -originalDimensions[0]/2, -originalDimensions[1]/2, -originalDimensions[2]/2);
            transformReslice2Volume->Translate( 0.0, 0.0, currentSliceIndex );

            image2DReslicer->SetResliceAxes( transformReslice2Volume->GetMatrix() );
            image3DActor->SetUserMatrix(transformReslice2Volume->GetMatrix());
            image3DActor->SetInput(image2DReslicer->GetOutput());
            break;
        default:
            break;
    }

    updateLocalTransformation();
    updatePickPlane();
}



/* Return the index of the current displayed slice. */
int Slice::getSlice() const {
    return currentSliceIndex;
}

/* Set the current slice index.
 * If the slice index is less than the first slice index, the first slice is displayed.
 * If the slice index is more than the last slice index, the last slice is displayed.
 * @param s the index of the slice to display (base 0). */
void Slice::setSlice(int s) {
    if(ARBITRARY_ORIENTATION == sliceOrientation) {
        currentSliceIndex = s;
        applyRotation();
    }
    else {

        // The origin of the reslicer should be set at the top corner of the
        // volume, thus only the z components should be updated.
        double orgInAxial[3] = { 0.0, 0.0, (double)s };
        double orgInReal[3];

        // Check if s is inside the volume bounds
        if ( s < 0 ) {
            s = 0.0;
        }
        else
            if ( s >= reslicedDimensions[2] ) {
                s = reslicedDimensions[2] - 1;
            }

        //-- Keep track of current the slice index
        currentSliceIndex = s;

        /* 3D Actors                                   */
        switch (sliceOrientation) {
            default:
            case ARBITRARY_ORIENTATION:
                //TODO
                break;
            case AXIAL_ORIENTATION:
                image3DActor->SetDisplayExtent(
                    extent[0], extent[1],
                    extent[2], extent[3],
                    s, s);
                break;
            case CORONAL_ORIENTATION:
                image3DActor->SetDisplayExtent(
                    extent[0], extent[1],
                    s, s,
                    extent[4], extent[5]);
                break;
            case SAGITTAL_ORIENTATION:
                image3DActor->SetDisplayExtent(
                    s, s,
                    extent[2], extent[3],
                    extent[4], extent[5]);
                break;
        }

        /* 2D Actors                                    */
        // Passing from index coordinates to real coordinates without spacing.
        reslicedToVolumeCoords(orgInAxial, orgInReal);

        // Real coordinates are rounded and spacing is taken into account.
        orgInReal[0] = (double)(int)(orgInReal[0] + 0.5) * originalSpacing[0];
        orgInReal[1] = (double)(int)(orgInReal[1] + 0.5) * originalSpacing[1];
        orgInReal[2] = (double)(int)(orgInReal[2] + 0.5) * originalSpacing[2];

        // Set the real position of the slice.
        image2DReslicer->SetResliceAxesOrigin( orgInReal );

        updateLocalTransformation();
        updatePickPlane();
    }
}

// Set the slice corresponding to the given world coordinates
void Slice::setSlice(double x, double y, double z) {
    // At this point, coordinates are expressed in REAL coordinate system (WORLD)

    double ijk[3] = {0.0, 0.0, 0.0};
    double xyz[3] = {x, y, z};

    // Passing from real coordinates to index coordinates
    volumeToReslicedCoords(xyz, ijk);

    // Set pixel position in current slice.
    setPixelRealPosition( ijk[0], ijk[1], 0.0 );

    // Round the slice number due to matrix inverse. It will be troncated in setSlice( int) procedure.
    ijk[2] += 0.5;

    // Set the slice number.
    setSlice(ijk[2]);
}



/* Return the number of colors in the images.
 * If color is coded on 1 byte, the images are on 256 grey level.
 * If color is coded on 2 bytes, the images are on 4096 grey level (not 65536). */
int Slice::getNumberOfColors() const {
    return originalVolume->GetScalarTypeMax() - originalVolume->GetScalarTypeMin();
}


// move the pixel selection green indicator (pixelActor) to the given real position
void Slice::setPixelRealPosition(double x, double y, double z) {
    // Truncate the pixel coordinates
    int intPixelX = x;
    int intPixelY = y;
    int intPixelZ = z;

    // Compute pixel REAL coordinates. Add an offset to the Z coordinate to make the actor visible in TOP lEFT origin.
    // TODO : Handle actor's visibility in BOTTOM RIGHT origin.
    double matchPixelX = (double)intPixelX * resliceSpacing[0];
    double matchPixelY = (double)intPixelY * resliceSpacing[1];
    double matchPixelZ = (double)intPixelZ * resliceSpacing[2];

    updatePixelActorPosition(matchPixelX, matchPixelY, matchPixelZ);
}

void Slice::initPixelActor() {
    // Get the bounding box coordinates within the slice
    double pickPlaneBounds[6];
    image2DReslicer->GetOutput()->ComputeBounds();
    image2DReslicer->GetOutput()->Update();
    image2DReslicer->GetOutput()->GetBounds(pickPlaneBounds);
    double xMin = pickPlaneBounds[0];
    double xMax = pickPlaneBounds[1];
    double yMin = pickPlaneBounds[2];
    double yMax = pickPlaneBounds[3];

    computeReslicedDimensions();

    // Instanciate the actor
    pixelActor = vtkSmartPointer<vtkActor>::New();

    // Draw the bounding box around the center of the slice
    updatePixelActorPosition(xMin + (xMax - xMin)/2,  yMin + (yMax - yMin)/2, 0.01);
}


void Slice::updatePixelActorPosition(double x, double y, double z) {
	// if the interactive viewer is not refreshed/displayed, the pixelActor is null. In this case, no point to update it
	if (pixelActor) {
		// The coordinates of the bounding box around the picked pixel
		double xMin, xMax , yMin, yMax;

		//--- Update pick plane
		double pickPlaneBounds[6];
		image2DReslicer->GetOutput()->ComputeBounds();
		image2DReslicer->GetOutput()->Update();
		image2DReslicer->GetOutput()->GetBounds(pickPlaneBounds);
		xMin = pickPlaneBounds[0];
		xMax = pickPlaneBounds[1];
		yMin = pickPlaneBounds[2];
		yMax = pickPlaneBounds[3];

		computeReslicedDimensions();

		// Compute the bounding box according to the picked pixel
		double xBorderMin, xBorderMax, yBorderMin, yBorderMax;
		xBorderMin = - x;
		xBorderMax = - x + xMin - xMax;

		switch (sliceOrientation) {
			case AXIAL_ORIENTATION:
				yBorderMin =  - y;
				yBorderMax =  - y + yMin - yMax;
				break;
			default:
				yBorderMin = - y;
				yBorderMax = - y - yMin + yMax;
				break;
		}

		// Check the orientation of the slice to deduce the depth to draw the pixel actor with its bounding box
		RendererWidget::CameraOrientation orientation;

		switch(sliceOrientation) {
			case AXIAL_ORIENTATION:
				orientation = InteractiveViewer::getAxialViewer()->getRendererWidget()->getCameraOrientation();
				break;
			case CORONAL_ORIENTATION:
				orientation = InteractiveViewer::getCoronalViewer()->getRendererWidget()->getCameraOrientation();
				break;
			case SAGITTAL_ORIENTATION:
				orientation = InteractiveViewer::getSagittalViewer()->getRendererWidget()->getCameraOrientation();
				break;
			default:
				break;
		}

		switch(orientation) {
			case RendererWidget::LEFT_UP:
			case RendererWidget::RIGHT_DOWN: // camera position has a negative depth
				z = -0.01;
				break;
			case RendererWidget::RIGHT_UP:
				z = +0.01;
				break;
			default:
				break;
		}

		// Create the cloup points used to describe the bounding box around
		vtkSmartPointer<vtkPoints> pixelPoints = vtkSmartPointer<vtkPoints>::New();
		pixelPoints->SetNumberOfPoints(4);
		// Those points describes the (red / blue / green) cross around the picked pixel.
		pixelPoints->InsertPoint(0, xBorderMin, 0, z );
		pixelPoints->InsertPoint(1, xBorderMax, 0, z );
		pixelPoints->InsertPoint(2, 0, yBorderMin, z );
		pixelPoints->InsertPoint(3, 0, yBorderMax, z );

		// Create the lines crossing the picked pixel
		vtkSmartPointer<vtkLine> hLine = vtkSmartPointer<vtkLine>::New();
		hLine->GetPointIds()->SetId(0, 0);
		hLine->GetPointIds()->SetId(1, 1);
		vtkSmartPointer<vtkLine> vLine = vtkSmartPointer<vtkLine>::New();
		vLine->GetPointIds()->SetId(0, 2);
		vLine->GetPointIds()->SetId(1, 3);

		// Create the unstructured grid according to these 2 lines and cloup points
		vtkSmartPointer<vtkUnstructuredGrid> aPixelGrid = vtkSmartPointer<vtkUnstructuredGrid>::New();
		aPixelGrid->Allocate(2, 2);
		aPixelGrid->InsertNextCell( hLine->GetCellType(), hLine->GetPointIds());
		aPixelGrid->InsertNextCell( vLine->GetCellType(), vLine->GetPointIds());
		aPixelGrid->SetPoints(pixelPoints);

		// Create the corresponding mapper
		vtkSmartPointer<vtkDataSetMapper> aPixelMapper = vtkSmartPointer<vtkDataSetMapper>::New();
		aPixelMapper->SetInput(aPixelGrid);
		
		pixelActor->SetMapper(aPixelMapper);
		pixelActor->GetProperty()->SetAmbient(1.0);
		pixelActor->GetProperty()->SetDiffuse(1.0);

		// Update pixel actor properties
		switch (sliceOrientation) {
			default:
			case ARBITRARY_ORIENTATION:
				pixelActor->GetProperty()->SetColor( 1.0, 1.0, 0.0);
				break;
			case AXIAL_ORIENTATION:
				pixelActor->GetProperty()->SetColor( 0.0, 0.0, 1.0);
				break;
			case CORONAL_ORIENTATION:
				pixelActor->GetProperty()->SetColor(0.0, 1.0, 0.0);
				break;
			case SAGITTAL_ORIENTATION:
				pixelActor->GetProperty()->SetColor(1.0, 0.0, 0.0);
				break;
		}

		pixelActor->GetProperty()->SetLineWidth( 1.0 );
		pixelActor->SetPosition( x, y, z );
		//-- pixelActor can not be picked
		pixelActor->PickableOff();
	}
	// else no point to update the pixelActor
}


/* Create the pixel grid (bounding box) around the pixel picked.
 * Returns the actors that has been calculated */
vtkSmartPointer<vtkActor> Slice::getPixelActor() {
    if (!pixelActor)
        initPixelActor();
    else {
        // Refresh the drawing of the pixel actor from its current position
        // Needed if slice plane are flipped.
        double crtPos[3];
        pixelActor->GetPosition(crtPos);
        updatePixelActorPosition(crtPos[0], crtPos[1], crtPos[2]);
    }

    return pixelActor;
}

// TODO : put those methods into a dedicated interface
//------------------------------- addProp ----------------------------------------
bool Slice::addProp(const QString& name, vtkSmartPointer< vtkProp > prop ) {
    if (!extraProp.contains(name)) {
        extraProp.insert(name, prop);
        return true;
    }
    else
        return false;
}

//------------------------------- getProp ----------------------------------------
vtkSmartPointer< vtkProp > Slice::getProp(const QString& name) {

    if (extraProp.contains(name))
        return extraProp.value(name);
    else
        return NULL;
}

//------------------------------- getNumberOfProp ----------------------------------------
unsigned int Slice::getNumberOfProp() const {
    return extraProp.values().size();
}

//------------------------------- getProp ----------------------------------------
vtkSmartPointer< vtkProp > Slice::getProp(unsigned int index) {
    return extraProp.values().at(index);
}

//------------------------------- removeProp ----------------------------------------
bool Slice::removeProp(const QString& name) {
    if (extraProp.contains(name)) {
        // remove the prop from any renderer/consummers
        vtkSmartPointer<vtkProp> prop = extraProp.value(name);
        prop->VisibilityOff();

        for (int i = 0; i < prop->GetNumberOfConsumers(); i++) {
            vtkViewport *viewer = vtkViewport::SafeDownCast(prop->GetConsumer(i));

            if (viewer)
                viewer->RemoveViewProp(prop);
        }

        // remove it from the maps
        extraProp.remove(name);
        return true;
    }
    else
        return false;
}




}

