/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
/**
 * @file convertMml
 * @brief Manage Qapp convertMMl
 * @author Johan Sarrazin
 * @date November 27 2012
 */


#include <iostream>
// qt
#include <qmessagebox.h>
#include <qfiledialog.h>

#include "convertMml.h"
#include "../csv2Sthing/csv2StatsVal/csv2StatsVal.h"
#include "../mml2Csv/mml2Csv.h"
#include "../csv2Sthing/csv2Plot/csv2Plot.h"
#include "help.h"
#include "ui_convertMml.h"


/* LIST OF INDICATORS */
const QStringList convertMml::monitors = QStringList() << "ComputingTime" << "DistanceX" << "DistanceY" << "DistanceZ" << "Force" << "GeomDeviation" << "Ren" << "Surface" << "Volume";
const QStringList convertMml::statistics = QStringList() << "RMS" << "RMSe" << "MAX" << "MEAN" << "MIN" << "STD";
const QStringList convertMml::plot = QStringList() << "BARCHART" << "BOX";


//----------convertMml--------------------------------------------------------------
convertMml::convertMml ( QWidget* parent ) : QMainWindow ( parent ), ui ( new Ui::convertMml )
{
    ui->setupUi ( this );
    //Some initialisation
    ui->mmlRadioButton->setChecked ( true );
    // maj combo box
    ui->monitorMmlComboBox->addItems ( monitors );
    ui->monitorComboBox->addItems ( monitors );
    ui->plotComboBox->addItems ( plot );
    ui->statsComboBox->addItems ( statistics );

    //flat list widget
    ui->resultTextEdit->viewport()->setAutoFillBackground ( false );

    initSLOT();
}

//----------destructor-----------------------------------------------------------------
convertMml::~convertMml()
{
    delete ( ui );
}

//---------------initApp---------------------------------------------------------------
void convertMml::initSLOT()
{
    // add/delete some file(s) to study
    QObject::connect ( ui->addMmlButton,SIGNAL ( clicked() ),SLOT ( addFile() ) );
    QObject::connect ( ui->delMmlButton,SIGNAL ( clicked() ),SLOT ( deleteFile() ) );
    // generate result asked
    QObject::connect ( ui->genComLinesButton,SIGNAL ( clicked() ),SLOT ( generateCommandLines() ) );
    QObject::connect ( ui->genCsvButton,SIGNAL ( clicked() ),SLOT ( generateCsvFile() ) );
    // where to save csv
    QObject::connect ( ui->saveStatsButton,SIGNAL ( clicked() ),SLOT ( saveStatsAs() ) );
    QObject::connect ( ui->saveMonitorsButton,SIGNAL ( clicked() ),SLOT ( saveMonitorsAs() ) );
    // change between csv or mml files
    QObject::connect ( ui->mmlRadioButton,SIGNAL ( toggled ( bool ) ),SLOT ( changeListItem ( bool ) ) );
    // add delete from stats tab
    QObject::connect ( ui->addMonitorButton,SIGNAL ( clicked() ),SLOT ( addMonitor() ) );
    QObject::connect ( ui->addStatButton,SIGNAL ( clicked() ),SLOT ( addStats() ) );
    QObject::connect ( ui->delMonitorButton,SIGNAL ( clicked() ),SLOT ( delMonitor() ) );
    QObject::connect ( ui->delStatButton,SIGNAL ( clicked() ),SLOT ( delStats() ) );
    // plot something
    QObject::connect ( ui->plotButton,SIGNAL ( clicked ( bool ) ),SLOT ( plotMonitor() ) );
    QObject::connect ( ui->savePlotButton,SIGNAL ( clicked ( bool ) ),SLOT ( savePlot() ) );
    // help !!!
    QObject::connect ( ui->helpButton,SIGNAL ( clicked ( bool ) ),SLOT ( helpInfo() ) );
}


//---------------addFile-------------------------------------------------------------------
void convertMml::addFile()
{
    QStringList filename;
    QString filenameCSV;
    std::string ext = "*.csv";
    // case where mml radio button is checked
    if ( ui->mmlRadioButton->isChecked() ) {
        ext = "*.mml";
        filename = QFileDialog::getOpenFileNames ( this,tr ( "Select files" ),"",tr ( ext.c_str() ) );
        ui->mmlListWidget->addItems ( filename );
    } else {
        // case where a csv radio button
        filenameCSV = QFileDialog::getOpenFileName ( this,tr ( "Select files" ),"",tr ( ext.c_str() ) );
        if ( !filenameCSV.isEmpty() ) {
            if ( ui->mmlListWidget->count() !=0 ) {
                ui->mmlListWidget->item ( 0 )->setText ( filenameCSV );
            } else {
                ui->mmlListWidget->addItem ( filenameCSV );
            }
            QString temp = filenameCSV;
            temp.truncate ( ui->mmlListWidget->item ( 0 )->text().lastIndexOf ( "." ) );
            temp+="STATS.csv";
            ui->saveStatsLineEdit->setText ( temp );
        }
    }
}

//---------------deleteFile------------------------------------------------------------
void convertMml::deleteFile()
{
    delete ui->mmlListWidget->currentItem();
}

//---------------addMonitor-------------------------------------------------------------
void convertMml::addMonitor()
{
    if ( ! ( ui->monitorMmlComboBox->currentText() ).isEmpty() ) {
        ui->monitorListWidget->addItem ( ui->monitorMmlComboBox->currentText() );
    }
}

//---------------addStats-------------------------------------------------------------
void convertMml::addStats()
{
    if ( ! ( ui->statsComboBox->currentText() ).isEmpty() ) {
        ui->statsListWidget->addItem ( ui->statsComboBox->currentText() );
    }
}

//---------------delmonitor------------------------------------------------------------
void convertMml::delMonitor()
{
    delete ui->monitorListWidget->currentItem();
}

//---------------delStats---------------------------------------------------------------
void convertMml::delStats()
{
    delete ui->statsListWidget->currentItem();
}

//---------------changeListItem---------------------------------------------------------
void convertMml::changeListItem ( bool mml )
{
    ui->mmlListWidget->clear();
    // case CSV toggled
    if ( !mml ) {
        ui->monitorListWidget->clear();
        ui->addMonitorButton->setDisabled ( true );
        ui->delMonitorButton->setDisabled ( true );

        ui->saveMonitorsButton->setDisabled ( true );
        ui->saveMonitorslineEdit->setDisabled ( true );
        ui->saveMonitorslineEdit->clear();
    } else {
        // MML toggled
        ui->addMonitorButton->setEnabled ( true );
        ui->delMonitorButton->setEnabled ( true );
        ui->saveMonitorsButton->setEnabled ( true );
        ui->saveMonitorslineEdit->setEnabled ( true );
    }
}

//----------------plot------------------------------------------------------------------
void convertMml::plotMonitor()
{
    QString inputFile;
    //test if file are in input
    if ( ui->mmlListWidget->count() !=0 ) {
        if ( ui->mmlRadioButton->isChecked() ) {
            if ( ui->monitorListWidget->count() !=0 ) {
                // First stage is to compute a csv from ./mml2csv with files in the box above
                list <dataMonitoredList> monitors;
                for ( int i = 0; i< ui->monitorListWidget->count(); i++ ) {
                    monitors.push_back ( dataMonitoredList ( ui->monitorListWidget->item ( i )->text().toStdString() ) );
                }
                list<string> mmlFiles;
                for ( int j = 0; j< ui->mmlListWidget->count(); j++ ) {
                    mmlFiles.push_back ( ui->mmlListWidget->item ( j )->text().toStdString() );
                }
                string output= "untitled.csv";
                if ( !ui->saveMonitorslineEdit->text().isEmpty() ) {
                    output = ui->saveMonitorslineEdit->text().toStdString();
                } else {
                    ui->saveMonitorslineEdit->setText ( QString ( output.c_str() ) );
                }
                mml2Csv mml ( output,monitors,mmlFiles );
                inputFile = ui->saveMonitorslineEdit->text();
            } else {
                QMessageBox msgBox;
                msgBox.setText ( "Error !\nSome monitors must be added\nin 'MML to Stats' tab before to plot something !" );
                msgBox.exec();
                return;
            }
        } else {
            inputFile = ui->mmlListWidget->item ( 0 )->text();
        }
        //Now we can plotting with the qcustomplot library and ./csv2plot
        // csv2Plot plot ;
        list<string> liste;
        liste.push_back ( ui->plotComboBox->currentText().toStdString() );
        csv2Plot plot ( inputFile.toStdString(),liste );
        ui->customPlot->clearPlottables();
        if ( ui->plotComboBox->currentText().toStdString() =="BARCHART" ) {
            plot.drawBarChart ( ui->customPlot, ui->monitorComboBox->currentText().toStdString() );
        } else if ( ui->plotComboBox->currentText().toStdString() =="BOX" ) {
            plot.drawBoxChart ( ui->customPlot, ui->monitorComboBox->currentText().toStdString() );
        }
        ui->customPlot->replot();
    }
}

//------------------generateCommandLines------------------------------------------------
void convertMml::generateCommandLines()
{
    QString commandline="";
    QString saveMonitors="";
    if ( ui->mmlListWidget->count() != 0 ) {
        if ( ui->saveMonitorslineEdit->text().isEmpty() ) {
            saveMonitors += "untitled.csv ";
        } else {
            saveMonitors += ui->saveMonitorslineEdit->text() +" ";
        }
        /* case of mml radio button checked
         * ie => input are mml files so to compute statistic the mini-app mml2csv
         * must be called first
         */
        if ( ui->mmlRadioButton->isChecked() ) {
            commandline += "./mml2Csv ";
            for ( int i=0; i<ui->monitorListWidget->count(); i++ ) {
                commandline += "-" + ui->monitorListWidget->item ( i )->text() + " ";
            }
            for ( int j=0; j< ui->mmlListWidget->count(); j++ ) {
                commandline += ui->mmlListWidget->item ( j )->text() + " ";
            }
            commandline += ":"+saveMonitors+"\n\n";
        }
        commandline += "./csv2Stat ";
        for ( int i=0; i<ui->statsListWidget->count(); i++ ) {
            commandline += "-"+ui->statsListWidget->item ( i )->text() +" ";
        }
        commandline +=saveMonitors+" ";
        if ( ui->saveStatsLineEdit->text().isEmpty() ) {
            commandline += ":untitledSTATS.csv ";
        } else {
            commandline += ":"+ui->saveMonitorslineEdit->text() +"STATS";
        }
        ui->resultTextEdit->setText ( commandline );
    } else {
        ui->resultTextEdit->setText ( "Data to study are missing !" );
    }
}


//---------------generateCsvFile---------------------------------------------------------
void convertMml::generateCsvFile()
{
    string saveMMLFile= ui->saveMonitorslineEdit->text().toStdString();
    if ( ui->mmlListWidget->count() != 0 ) {
        // Case where ./mml2csv must be launch first
        if ( ui->mmlRadioButton->isChecked() ) {
            list <dataMonitoredList> monitors;
            for ( int i = 0; i< ui->monitorListWidget->count(); i++ ) {
                monitors.push_back ( dataMonitoredList ( ui->monitorListWidget->item ( i )->text().toStdString() ) );
            }
            list<string> mmlFiles;
            for ( int j = 0; j< ui->mmlListWidget->count(); j++ ) {
                mmlFiles.push_back ( ui->mmlListWidget->item ( j )->text().toStdString() );
            }
            saveMMLFile= "untitled.csv";
            if ( !ui->saveMonitorslineEdit->text().isEmpty() ) {
                saveMMLFile = ui->saveMonitorslineEdit->text().toStdString();
            } else {
                ui->saveMonitorslineEdit->setText ( QString ( saveMMLFile.c_str() ) );
            }
            mml2Csv mml ( saveMMLFile,monitors,mmlFiles );
            ui->resultTextEdit->setText ( "Executed : mml2Csv" );
        } else {
            saveMMLFile=ui->mmlListWidget->item ( 0 )->text().toStdString();
        }
        // get a list where stats name are saved
        list<string> statsName;
        for ( int i =0; i< ui->statsListWidget->count(); i++ ) {
            statsName.push_back ( ( ui->statsListWidget->item ( i )->text() ).toStdString() );
        }
        // compute statistics
        string output ="untitledSTATS.csv";
        if ( !ui->saveStatsLineEdit->text().isEmpty() ) {
            output =  ui->saveStatsLineEdit->text().toStdString();
        } else {
            ui->saveStatsLineEdit->setText ( QString ( output.c_str() ) );
        }
        if ( ui->statsListWidget->count() !=0 ) {
            csv2StatsVal c ( output,saveMMLFile,statsName );
            ui->resultTextEdit->setText ( c.getResultsAsText().c_str() );
        } else {
            ui->resultTextEdit->setText ( ui->resultTextEdit->toPlainText() +"\nNo stats to compute !" );
        }
    } else {
        ui->resultTextEdit->setText ( "Data to study are missing !" );
    }
}

//-----------------------------save As-----------------------------------
void convertMml::saveMonitorsAs()
{
    QFileDialog dialog ( this );
    dialog.setFileMode ( QFileDialog::AnyFile );
    dialog.setNameFilter ( tr ( "*.csv" ) );
    dialog.setViewMode ( QFileDialog::Detail );
    QStringList fileNames;
    if ( dialog.exec() ) {
        fileNames = dialog.selectedFiles();
        if ( !fileNames.first().contains ( ".csv" ) ) {
            fileNames.first() +=".csv";
        }
        ui->saveStatsLineEdit->setText ( fileNames.first() );
        ui->saveMonitorslineEdit->setText ( fileNames.first() );
        QString temp = ui->saveMonitorslineEdit->text();
        temp.truncate ( ui->saveMonitorslineEdit->text().lastIndexOf ( "." ) );
        temp += "STATS.csv";
        ui->saveStatsLineEdit->setText ( temp );
        ui->saveMonitorslineEdit->setText ( fileNames.first() );
    }
}

//---------------saveStatsAs----------------------------------------------
void convertMml::saveStatsAs()
{
    QFileDialog dialog ( this );
    dialog.setFileMode ( QFileDialog::AnyFile );
    dialog.setNameFilter ( tr ( "*.csv" ) );
    dialog.setViewMode ( QFileDialog::Detail );
    QStringList fileNames;
    if ( dialog.exec() ) {
        fileNames = dialog.selectedFiles();
        ui->saveStatsLineEdit->setText ( fileNames.first() );
    }
}

//---------------savePlot-----------------------------------------------------------
void convertMml::savePlot()
{
    QString fileName,filter="";
    fileName= QFileDialog::getSaveFileName ( this,tr ( "Save File As" ),"",tr ( "Images (*.png);; Pdf (*.pdf);; PostScript (*.ps)" ),&filter );

    if ( filter.contains ( ".png" ) ) {
        if ( !fileName.contains ( ".png" ) ) {
            //type of image is missing
            fileName+=fileName+".png";
        }
        QPixmap pm = QPixmap::grabWidget ( ui->customPlot );
        pm.save ( fileName );

    } else if ( filter.contains ( ".ps" ) ) {
        if ( !fileName.contains ( ".ps" ) ) {
            //type of image is missing
            fileName+=fileName+".ps";
        }
        ui->customPlot->savePdf ( fileName,true,0,0 );

    } else if ( filter.contains ( ".pdf" ) ) {
        if ( !fileName.contains ( ".pdf" ) ) {
            //type of image is missing
            fileName+=fileName+".pdf";
        }
        ui->customPlot->savePdf ( fileName,true,0,0 );

    }
}

//---------------help----------------------------------------------------------------
void convertMml::helpInfo()
{
    help *h= new help;
    h->show();
}


//--main---------------------------------
int main ( int argc, char* argv[] )
{
    QApplication app ( argc,argv );
    convertMml *c = new convertMml;
    c->show();
    return app.exec();
    //delete in memory
    c->~convertMml();
}


