/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef MULTICOMPONENT_H
#define MULTICOMPONENT_H

#include <vector>
#include <algorithm>

#include "Component.h"

/**
 * A multi-component stores other components, hence providing a way to have
 * an tree representation of components. Isn't that tricky?
 *
 * \note To delete and free the memory of all the sub components, you have to
 * call the deleteAllSubComponents() method.
 *
 *
 */
class MultiComponent : public Component {
public:
    /** Default Constructor */
    MultiComponent(PhysicalModel *);
    /// constructor that allows to name the structure (provides a name)
    MultiComponent(PhysicalModel *, std::string);
    /// delete all the subcomponents (call the deleteAllSubComponents method)
    ~MultiComponent();

    unsigned int getNumberOfSubComponents() const;
    Component * getSubComponent(const unsigned int) const;
    void addSubComponent(Component *);
    /**
     * Remove a component from the list.
     * Becareful: this method DOES NOT delete the object and/or free the memory.
      * This method ask the component c to remove this multicomponent from the list of its parent component
     * @param c the ptr to the structure to remove
     * @see removeAllSubComponent()
     */
    void removeSubComponent(Component *c);

    /**
     * this method free all the sub-components (i.e. delete all the sub component
     *  and clear the list).
     * After this methode getNumberOfSubComponents should return 0
     */
    void deleteAllSubComponents();

    /** return true only if the parameter is equal to "MultiComponent" */
    virtual bool isInstanceOf(const char *) const;

    /** print to an output stream in "pseaudo" XML format (do nothing if there are no sub components).
      */
    void xmlPrint(std::ostream &) const;

    /// get the total nr of cell of the component
    unsigned int getNumberOfCells() const;

    /// get cell by order number (not cell index)
    Cell * getCell(unsigned int) const;

    /// conveniant method to get the sub component of the name given in parameter
    Component * getComponentByName(const std::string);
    
    /// return the state of a visibility mode in all the sub component (if at least one sub component is visible for this mode, it will return true; if none are visible it will return false).
    virtual bool isVisible(const RenderingMode::Mode mode) const;

    /// set the state of a visibility mode in all the sub component.
    virtual void setVisible(const RenderingMode::Mode mode, const bool b);
    
    /// set the physical model (recursively)
    virtual void setPhysicalModel(PhysicalModel *);

protected:
    /**
     * List of sub component
     */
    std::vector <Component *> components;
};

// -------------------- inline methods -------------------
inline 	unsigned int MultiComponent::getNumberOfSubComponents() const {
    return components.size();
}
inline Component * MultiComponent::getSubComponent(const unsigned int i) const {
    if (i<components.size())
        return components[i];
    else
        return NULL;
}
inline void MultiComponent::addSubComponent(Component * c) {
    // add c in the list
    components.push_back(c);
    // add this in the list of c's composing component
    c->addParentMultiComponent(this);
}
inline void MultiComponent::removeSubComponent(Component *c) {
    std::vector <Component *>::iterator it = std::find(components.begin(), components.end(), c);
    if (it != components.end()) {
        components.erase(it);
        c->removeParentMultiComponent(this);
    }
}
inline Component * MultiComponent::getComponentByName(const std::string n) {
    std::vector <Component *>::iterator it=components.begin();
    Component *foundC=NULL;
    while (it!=components.end() && !foundC) {
        foundC = ((*it)->getName()==n)?(*it):NULL;
        // look inside the component if it is a MultiComponent
        if (!foundC && (*it)->isInstanceOf("MultiComponent")) {
            foundC = ((MultiComponent *) (*it))->getComponentByName(n);
        }
        it++;
    }
    return foundC;
}
inline bool MultiComponent::isInstanceOf(const char *className) const {
    return (std::string(className)==std::string("MultiComponent"));
}

#endif //MULTICOMPONENT_H
