/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "OpenFile.h"

// CamiTK
#include <Application.h>
#include <ExtensionManager.h>

// Qt
#include <QFileDialog>
#include <QBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QFrame>

using namespace camitk;

// --------------- constructor -------------------
OpenFile::OpenFile (ActionExtension * extension) : Action (extension) {
    // Setting name, description and input component
    setName("Open File");
    setDescription("Open data (component) from a given file");
    setComponent("");

    if (qApp->type() != QApplication::Tty)
        setIcon (QPixmap (":/fileOpen"));

    // Setting classification family and tags
    setFamily("Application");
    addTag("Open");

    // add a shortcut
    if (qApp->type() != QApplication::Tty) {
        getQAction()->setShortcut( QKeySequence::Open );
        getQAction()->setShortcutContext(Qt::ApplicationShortcut);
    }

    // the unique property is the filename
    setProperty("File Name", QVariant(""));

    // lazy instanciation
    myWidget = NULL;
}

// --------------- destructor -------------------
OpenFile::~OpenFile() {
}

// --------------- getWidget --------------
QWidget * OpenFile::getWidget() {
    if (!myWidget) {
        //-- the frame
        myWidget = new QFrame();
        myWidget->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
        myWidget->setLineWidth(3);

        //-- the vertical layout: put every GUI elements in it
        QVBoxLayout *myWidgetLayout = new QVBoxLayout();

        //-- build file browsing widget
        QLabel* fileBrowserLabel = new QLabel("File Name:");
        fileBrowserLineEdit = new QLineEdit();
        fileBrowserLineEdit->setText(property("File Name").toString()); 
        fileBrowserLineEdit->setReadOnly(true);
        QPushButton *fileBrowserButton = new QPushButton("Browse");
        connect(fileBrowserButton, SIGNAL(clicked()), this, SLOT(selectFile()));

        //-- the horizontal layout for file browsing
        QHBoxLayout * fileBrowserLayout = new QHBoxLayout();
        fileBrowserLayout->addWidget(fileBrowserLabel);
        fileBrowserLayout->addWidget(fileBrowserLineEdit);
        fileBrowserLayout->addWidget(fileBrowserButton);
        myWidgetLayout->addLayout(fileBrowserLayout);

        //-- build the open button
        QPushButton *openSelected = new QPushButton("Open Selected File");
        QObject::connect(openSelected, SIGNAL(clicked()), this, SLOT(apply()));

        //-- the horizontal layout for the open button
        QHBoxLayout *buttonLayout = new QHBoxLayout;
        buttonLayout->addWidget(openSelected);
        myWidgetLayout->addLayout(buttonLayout);

        //-- set the layout for the action widget
        myWidget->setLayout(myWidgetLayout);
    }

    return myWidget;
}

// --------------- apply -------------------
Action::ApplyStatus OpenFile::apply() {
    QString fileToOpen = property("File Name").toString();

    if (!fileToOpen.isEmpty()) {
        Application::showStatusBarMessage(tr(QString("Opening file " + fileToOpen + "...").toStdString().c_str()));
        Application::open(fileToOpen);
        return SUCCESS;
    }
    else {
        return ABORTED;
    }
}

// --------------- selectFile -------------------
void OpenFile::selectFile() {
    // filter: extension of known file format
    QString filter;
    QString allKnownFiles; //example: (*.xml *.vtk *.wrl)

    // first item = all known files
    filter += QString ("All known files (*.") + ExtensionManager::getFileExtensions().join(" *.") + ");;";

    // add the extension of loaded and valid plugins
    foreach(ComponentExtension * ext, ExtensionManager::getComponentExtensions().values().toSet()) {
        filter += ext->getName() + " (*." + ext->getFileExtensions().join(" *.") + ");;";
    }

    // Open only one file
    QString fileName = QFileDialog::getOpenFileName(NULL, "Select One File to Open", Application::getLastUsedDirectory().absolutePath(), filter);

    if (!fileName.isEmpty()) {
        fileBrowserLineEdit->setText(fileName);
        setProperty("File Name",fileName);
    }
}


