# CTest instructions to make a continuous build on a computer (Linux / Windows / MacOSX)
# Schedule a task running this script with ctest and launch it using the following command:
# "ctest -DCAMITK_CONTINUOUS_INTEGRATION -DCTEST_BINARY_DIRECTORY -S continuous.cmake -V"
# CAMITK_CONTINUOUS_INTEGRATION = COMPILER-ARCH-BUILDTYPE is the build name, providing build options
# CTEST_BINARY_DIRECTORY = path to the build directory where to compile the generated code of this test.

# What this script does ?
# * INFORMATION STEP
#       Configure SITE and BUILD information to be correctly display on dashboard
#       Loads information from the CTestConfig.cmake file.
# * UPDATE STEP
#       use svn to udpdate source code to the latest revision.
# * CONFIGURE STEP
#       configure the whole CamiTK project.
# * BUILD STEP 
#       build updated subprojects of CamiTK.
# NOTE : Reports are send tagged for each subproject
# For each step a report is sent to the dashboard. This allows any developer to be informed (mailing list) of any problem, even BEFORE the script ends !

# Need to be defined, for the build to run.
if(NOT DEFINED CTEST_SOURCE_DIRECTORY)
    message(FATAL_ERROR "Please provide the source directory of the continuous test with the CTEST_SOURCE_DIRECTORY argument")
endif()
if(NOT DEFINED CTEST_BINARY_DIRECTORY)
    message(FATAL_ERROR "Please provide the build directory of the continuous test with the CTEST_BINARY_DIRECTORY argument")
endif()

# Script configuration, depending of the build, computer running the script
# Update to feat each computer which runs this script

# Get VM compilation information given by ctest call command
if(CAMITK_CONTINUOUS_INTEGRATION)
    string(REGEX REPLACE "^(.*)-.*-.*" "\\1" COMPILER "${CAMITK_CONTINUOUS_INTEGRATION}")
    string(REGEX REPLACE "^.*-(.*)-.*" "\\1" ARCH "${CAMITK_CONTINUOUS_INTEGRATION}")
    string(REGEX REPLACE "^.*-.*-(.*)" "\\1" BUILDTYPE "${CAMITK_CONTINUOUS_INTEGRATION}")
else()
    message(FATAL_ERROR "CAMITK_CONTINUOUS_INTEGRATION value must be given as option of the ctest command calling this script.")
endif()

# Compose with those variables the CTest required ones.
site_name(CTEST_SITE)
set( CTEST_BUILD_NAME ${CAMITK_CONTINUOUS_INTEGRATION})
if(UNIX)
    set( CTEST_CMAKE_GENERATOR  "Unix Makefiles" )
elseif(WIN32)
    if(COMPILER MATCHES "MinGW" OR "MINGW")
        set( CTEST_CMAKE_GENERATOR  "MinGW Makefiles" )
    elseif(COMPILER MATCHES "MSVC2008")
        set( CTEST_CMAKE_GENERATOR "Visual Studio 9 2008" )
    elseif(COMPILER MATCHES "MSVC2010" AND ARCH MATCHES "32bits")
        set( CTEST_CMAKE_GENERATOR  "Visual Studio 10" )
        set( CMAKE_MAKE_PROGRAM "C:/Program Files (x86)/Microsoft Visual Studio 10.0/Common7/IDE/VCExpress.exe") # Do not use MSBuild.exe as it won't work
    elseif(COMPILER MATCHES "MSVC2010" AND ARCH MATCHES "64bits")
        set( CTEST_CMAKE_GENERATOR  "Visual Studio 10 Win64" )
        set( CMAKE_MAKE_PROGRAM "C:/Program Files (x86)/Microsoft Visual Studio 10.0/Common7/IDE/VCExpress.exe") # Do not use MSBuild.exe as it won't work
    elseif(COMPILER MATCHES "MSVC2012" AND ARCH MATCHES "32bits")
        set( CTEST_CMAKE_GENERATOR  "Visual Studio 11" )
        set( CMAKE_MAKE_PROGRAM "C:/Program Files (x86)/Microsoft Visual Studio 11.0/Common7/IDE/WDExpress.exe") # Do not use MSBuild.exe as it won't work
    elseif(COMPILER MATCHES "MSVC2012" AND ARCH MATCHES "64bits")
        set( CTEST_CMAKE_GENERATOR  "Visual Studio 11 Win64" )
        set( CMAKE_MAKE_PROGRAM "C:/Program Files (x86)/Microsoft Visual Studio 11.0/Common7/IDE/WDExpress.exe") # Do not use MSBuild.exe as it won't work
    elseif(COMPILER MATCHES "MSVC2013" AND ARCH MATCHES "32bits")
        set( CTEST_CMAKE_GENERATOR  "Visual Studio 12" )
    elseif(COMPILER MATCHES "MSVC2013" AND ARCH MATCHES "64bits")
        set( CTEST_CMAKE_GENERATOR  "Visual Studio 12 Win64" )
    else()
        message(FATAL_ERROR "CTEST COMPILER ERROR : No proper compiler found, please check ctest command syntax.")
    endif()
endif()

if(BUILDTYPE)
    set(CTEST_BUILD_CONFIGURATION ${BUILDTYPE})
    set(CTEST_CONFIGURATION_TYPE ${BUILDTYPE})
else()
    message(FATAL_ERROR "NO BUILD TYPE : Please provide a build type: Debug or Release")
endif()

# Update source code and send reports
# set( CTEST_UPDATE_COMMAND "svn")
find_program(CTEST_SVN_COMMAND svn)
set(CTEST_UPDATE_COMMAND ${CTEST_SVN_COMMAND})

# CMake configuration (put here all the configure flags)
set(CTEST_CONFIGURE_COMMAND "${CMAKE_COMMAND} -Wno-dev -G \"${CTEST_CMAKE_GENERATOR}\"")
set(CTEST_CONFIGURE_COMMAND "${CTEST_CONFIGURE_COMMAND} -DCMAKE_BUILD_TYPE:STRING=${CTEST_BUILD_CONFIGURATION}")
if(COMPILER MATCHES "MSVC2010" AND ARCH MATCHES "64bits") # Do not compile MML and PhysicalModel until libxml2 MSVC2010 x64 bug remains.
    set(CTEST_CONFIGURE_COMMAND "${CTEST_CONFIGURE_COMMAND} -DACTION_MML=FALSE -DCOMPONENT_MML=FALSE -DCOMPONENT_PHYSICALMODEL=FALSE")
endif()
set(CTEST_CONFIGURE_COMMAND "${CTEST_CONFIGURE_COMMAND} -DCEP_IMAGING=TRUE -DCEP_MODELING=TRUE ${CTEST_SOURCE_DIRECTORY}")

set(CTEST_USE_LAUNCHERS 1)
# -DCAMITK_INSTALL_ROOT=${CTEST_BINARY_DIRECTORY}

# to get CDash server configuration :
include("${CTEST_SOURCE_DIRECTORY}/CTestConfig.cmake")

# Continuous build parameters
# Update them regarding your project's compilation time
set(CONTINUOUS_BUILD_DURATION 43200) # Duration time of the main loop of the continuous build in seconds, here 12 hours.
set(CONTINUOUS_BUILD_SLEEP 600) # Step to perform each build <=> maximum time for the script to sleep between 2 builds.

# Clean binary directory each beginning of the day
ctest_empty_binary_directory(${CTEST_BINARY_DIRECTORY})

# Main loop
# During 10 hours from starting this script, CTest will continuously build every 10 min.
while (${CTEST_ELAPSED_TIME} LESS ${CONTINUOUS_BUILD_DURATION})
        # Get start time of the build
        set(START_TIME ${CTEST_ELAPSED_TIME})
        
        # Start CTest in Continuous mode
        ctest_start(Continuous)
        
        # Update source code and get update command result
        ctest_update(SOURCE ${CTEST_SOURCE_DIRECTORY} RETURN_VALUE count)
        
        if (count GREATER 0) # We have updated source code, thus configure and build modified files
                # show update reports if there is a change in the source code
                ctest_submit(PARTS Update Notes)
                
                # Configure the whole camitk project and send configure report on the server 
                ctest_configure()
                ctest_submit(PARTS Configure)
                
                # Get subprojects listing (automatically created at confituration step)
                include("${CTEST_BINARY_DIRECTORY}/Subprojects.cmake")
                
                # Update CDash configuration to the server
                # Project.xml file is automatically generated. If someone added an extension to CamiTK, CDash will automatically be updated according to it.
                # To do this, send this file to the server
                ctest_submit(FILES "${CTEST_BINARY_DIRECTORY}/Project.xml") 
                
                # Build each subprojects
                foreach(subproject ${CAMITK_SUBPROJECTS})

                        # tag sub project, tag are used to sort projects according to tests
                        set_property(GLOBAL PROPERTY SubProject ${subproject})
                        set_property(GLOBAL PROPERTY Label ${subproject})

                        # build each sub project
                        set(CTEST_BUILD_TARGET ${subproject})
                        ctest_build()
                        ctest_submit(PARTS Build)
                        # build binary for testing
#                         set(CTEST_BUILD_TARGET ${subproject}test)
#                         ctest_build()

                        # Run tests
                        ctest_test(BUILD "${CTEST_BINARY_DIRECTORY}" INCLUDE_LABEL "${subproject}")
                        ctest_submit(PARTS Test)
                        
                endforeach(subproject)
                        
        endif ()

        message(STATUS "Waiting for SVN change")
        
        # Wait maximum the CONTINUOUS_BUILD_SLEEP time (in seconds) to loop again and launch next build
        ctest_sleep(${START_TIME} ${CONTINUOUS_BUILD_SLEEP} ${CTEST_ELAPSED_TIME})
endwhile()