/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/


#ifndef EXTENSIONS_MANAGER_H
#define EXTENSIONS_MANAGER_H

// -- Core stuff
#include "ComponentExtension.h"
#include "ActionExtension.h"
#include "CamiTKAPI.h"
#include "AbortException.h"

// -- QT stuff
#include <QtPlugin>
#include <QPluginLoader>
#include <QStringList>
#include <QMap>
#include <QDir>

namespace camitk {
/** 
 * @ingroup group_sdk_libraries_core
 * 
 * @brief
 * This class is used to manage all plugins loaded by the application.
 *
 *  The current version is able to load dynamic library for
 *  - ComponentExtension
 *  - ActionExtension
 *
 *  This class is a contained for all the loaded extension. It contains only
 *  static members.
 */
class CAMITK_API ExtensionManager : public QObject {

public:
    /// 
    enum {
    };
    /** \enum ExtensionType describes the CamiTK Extension Type (Action, Component, Application...).
     * CamiTK follows Component-Based programming architecture. The Service Layer defines four types
     * of extensions.
     */
    enum ExtensionType {
        ACTION,         ///< Action extensions: manages the processing logic
        COMPONENT,      ///< Component extensions: manages the data logic
        APPLICATION,    ///< MainWindow extensions: manages the application logic
        VIEWER          ///< Viewer extensions: manages the presentation and user interaction logic.
    };    

    /// Autoload component and action extensions
    static void autoload();

    /** Autoload all the extension of a given type.
     *  This method try to load all the extensions of the given type that are found (in this specific order) :
     *  - in the current working directory (considered as an install directory), this should be the build directory (load extensions that are just being compiled, before their installation
     *  - in the user config directory (CamiTK user install)
     *  - in the system-wide install (CamiTK global install)
     *  - and in the list of user-registered extensions (stored in the configuration directory)
     * 
     * Duplicates found within the installation directory are removed (e.g. if you have the extension x.so in the build directory as
     * well as the user directory, only the extension in the build directory will be loaded (it takes precedence).
     * 
     * @param type the extension type
     */
    static void autoload(ExtensionType type);
    
    /** Load a specific extension from a file.
     *  This methods loads a ActionExtension, ComponentExtension,... plugin from a .dll/.so/.dylib filename
     *
     * @param type the extension type
     * @param file the plugin filename (.dll/.so/.dylib)
     * @return false if it cannot be loaded
     */
    static bool loadExtension(ExtensionType type, QString file);

    /** Returns a three char string that corresponds to the installation directory of a given extension.
     *
     *  It works for all type of extension.
     *
     *  The following string code are returned:
     *  - [G] for an extension installed in the global installation directory (where CamiTK SDK is installed)
     *  - [L] for an extension installed in the user configuration directory (local install)
     *  - [W] for an extension loaded directly from the current working directory (build installation tree)
     *  - [U] for an extension loaded manually by the user
     * 
     *  @return the string coding for the installation
     *  @param file the extension plugin file
     */
    static QString getInstallationString(QString file);
    
    /** @name Component extension list management
       */
    ///@{
    /// get the plugin corresponding to a given extension or name (const, you can only use this method to get information, use open to open a file using a specific plugin)
    static const ComponentExtension * getComponentExtension(QString);

    /** get all the registered component extensions that can manage file (default component extension)
      * This is the public method (return a const, the extensions map is private
      * and cannot be modified externally).
      */
    static const QMap<QString, ComponentExtension*> & getComponentExtensions();

    /** get all the registered component extensions that can manage entire directories
     * This is the public method (return a const, the extensions map is private
      * and cannot be modified externally).
      */
    static const QMap<QString, ComponentExtension*> & getDataDirectoryComponents();

    /// get the list of all the suffixes managed by registered component extensions (all possible file suffix)
    static QStringList getFileExtensions();

    /// get the list of all the name of the registered Component data directory
    static QStringList getDataDirectoryExtNames();
    
    /** Register the file extension with the current application for opening
     *  This function is called by registerFileExtensions
     *  @param fileExtension the file extension to associate with the current application for opening
     **/
    static void registerFileExtension(QString fileExtension);

    /// Prompt the user if he wants the current application to natively (by default) open the following file extensions
    /// @param fileExtensions The list of file extensions (without the prefixed '.').
    /// @return The decision of the user regarding the file association.
    static bool promptRegisterFileExtensions(QStringList fileExtensions);
    
    /** Unload a plugin corresponding to a given extension or component name.
     * @return false if the plugin could not be unloaded - still in use somewhere
     */
    static bool unloadComponentExtension(QString);
    ///@}

    
    /// @name Action extension list management
    ///@{

    /// unload all action extensions and delete instanciated actions
    static void unloadAllActionExtensions();

    /** get all the registered action extension (key is the share object/dll/dylib filename that contains the ActionExtension).
     * This is the public method (return a const, the extensions map is private
     * and cannot be modified externally).
     */
    static const QMap<QString, ActionExtension*> & getActionExtensions();
    
    /** Unload an action extension using its .dll/.so/.dylib filename
     * @return false if the plugin could not be unloaded - still in use somewhere
     */
    static bool unloadActionExtension(QString);
    ///@}

    ///@cond
    // TODO CAMITK_DEPRECATED. This section list all the methods marked as deprecated. They are to be removed in CamiTK 4.0    
    /** Deprecated.
     * load a ComponentExtension plugin using a .dll/.so/.dylib filename
     * @param file the plugin filenamae (.dll/.so/.dylib)
     * @return false if it cannot be loaded
     */
    static bool loadComponentExtension(QString file);

    /** Deprecated.
     * load an action extension using a .dll/.so/.dylib filename
     * @return false if it cannot be loaded
     */
    static bool loadActionExtension(QString);

    /** Deprecated.
     * autoload all the ComponentExtension plugins (use applicationDirPath/components)
     */
    static void autoloadComponentExtensions();

    /** Deprecated.
     *  autoload all the action extensions (use applicationDirPath/actions)
     */
    static void autoloadActionExtensions();

    ///@endcond
    
private:
    /** get the singleton map of loaded component plugins for files (the key is the file extension)
    * This is the private (intern) method.
    * The component extension map is updated by loadComponentExtension, unloadComponentExtension and autoloadComponentExtensions.
    * This method follows the "construct on first use" idiom/design-pattern.
    * It therefore avoids the infamous "static initialization order fiasco",
    * see http://www.parashift.com/c++-faq/ctors.html
    */
    static QMap<QString, ComponentExtension*> & getComponentExtensionMap();

    /** get the singleton map of loaded component plugins for data directory (the key is the name)
    * This is the private (intern) method.
    * The data directory component extension map is updated by loadExtension, unloadExtension and autoloadExtensions.
    * This method follows the "construct on first use" idiom/design-pattern.
    * It therefore avoids the infamous "static initialization order fiasco",
    * see http://www.parashift.com/c++-faq/ctors.html
    */
    static QMap<QString, ComponentExtension*> & getDataDirectoryComponentExtensionMap();

    /** get the singleton map of loaded action plugins (the key is the shared object/dll/dylib filename)
     * This is the private (intern) method.
     * The component extension map is updated by loadActionExtension, unloadActionExtension and autoloadActionExtensions.
     * This method follows the "construct on first use" idiom/design-pattern.
     * It therefore avoids the infamous "static initialization order fiasco",
     * see http://www.parashift.com/c++-faq/ctors.html
     */
    static QMap<QString, ActionExtension*> & getActionExtensionMap();

    /// return the extensions file filter
    static QStringList getExtensionFilter();

    /// return the list of shared objects in a directory considering debug/release on MSVC
    static QStringList getPluginFileNames(QDir);
};

}


#endif //EXTENSIONS_MANAGER_H
