/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef MESHCOMPONENT_H
#define MESHCOMPONENT_H

#include "Component.h"
#include "Geometry.h"
#include "MeshDataModel.h"
#include "MeshSelectionModel.h"

class vtkPointSet;
class vtkSelection;
class vtkSelectionNode;

class QTableView;

namespace camitk {

/**
 * @ingroup group_sdk_libraries_core_component_mesh
 *
 * @brief
 * Basic component to manage any kind of mesh.
 */
class CAMITK_API MeshComponent : public Component {
    Q_OBJECT
public:

    /** Creates a top-level MeshComponent from a file.
     *  \note this is only to be used from a Component Extension open(...) or from an Action that creates data from a filter or transformation of a vtkPointSet.
     *
     *  Please consider using MeshComponent(vtkSmartPointer<vtkPointSet> , const QString &)
     */
    MeshComponent( const QString & file ) throw( AbortException );

    /** Creates a top-level MeshComponent from a vtkPointSet (and instanciate its 3D representation).
     *
     * \note the filename is set to the empty string "".
     *
     * \note if aPointSet is NULL, the representation is not initialized, it is the responsability of the subclass to initialize it later
     *
     * @param aPointSet point set of the new MeshComponent
     * @param name name to be given to the Component
     */
    MeshComponent( vtkSmartPointer<vtkPointSet> aPointSet, const QString &name );

    /** Creates a MeshComponent as a sub component of another Component using a vtkPointSet (and instanciate its 3D representation).
     *
     * \note if aPointSet is NULL, the representation is not initialized, it is the responsability of the subclass to initialize it later
     *
     * @param parentComponent the parent component of the new MeshComponent
     * @param aPointSet point set of the new MeshComponent
     * @param name name to be given to the new MeshComponent
     */
    MeshComponent( Component *parentComponent, vtkSmartPointer<vtkPointSet> aPointSet, const QString &name );

    /// Destructor
    virtual ~MeshComponent();

    /// reimplemented to save the last pick point id
    void pointPicked( vtkIdType pointId, bool );

    /// reimplemented to save the last pick point id
    void cellPicked( vtkIdType cellId, bool );

    /// get the last pick point id, @return -1 if no point where picked
    vtkIdType getPickedPointId();

    /// get the last pick point id, @return -1 if no point where picked
    vtkIdType getPickedCellId();

    /// update property: no specific properties to manage, this is needed to avoid console warnings.
    void updateProperty( QString , QVariant ) {};

    /// there is more than one property widgets (to view as tabs in the property explorer)
    virtual unsigned int getNumberOfPropertyWidget();

    /// the proprety widgets are: default property widget, selection view and data view
    virtual QWidget * getPropertyWidgetAt( unsigned int i, QWidget* parent = 0 );

    /**
     * @name Selection management
     * @see http://www.kitware.com/media/html/SelectionsInVTK.html
     * @{
     */
    /**
     * Add a selection.
     */
    void addSelection( const QString& name, int fieldType, int contentType );

    /**
     * Remove a selection.
     */
    void removeSelection( const QString& name );

    /**
     * Get the number of selection
     */
    int getNumberOfSelections() const;

    /**
     * Get a selection.
     */
    vtkSmartPointer<vtkSelectionNode> getSelection( const QString& name ) const;

    /**
     * Get selection
     */
    vtkSmartPointer<vtkSelectionNode> getSelectionAt( int i ) const;

    /**
     * Get the selection list
     */
    const QMap< QString, vtkSmartPointer<vtkSelectionNode> >& getSelectionList() const;

    /**
     * Add to a selection.
     */
    void addToSelection( const QString& name, vtkSmartPointer<vtkAbstractArray> array );
    /**
     * @}
     */

    /**
     * @name Data management
     * @{
     */

    /**
     *  Add data array
     */
    void addDataArray( const QString& name, vtkSmartPointer<vtkDataArray> data );

    /**
     *  Add data array
     */
    void removeDataArray( const QString& name );

    /**
     * Link data array to points
     */
    void linkDataToPoints( const QString& name );

    /**
     * Link data array to cells
     */
    void linkDataToCells( const QString& name );

    /**
     * Add point data.
     */
    void addPointData( const QString& name, int dataType, vtkSmartPointer<vtkDataArray> data );

    /**
     * Add cell data.
     */
    void addCellData( const QString& name, int dataType, vtkSmartPointer<vtkDataArray> data );

    /// get the current data view model (model as the M in Qt MVC design pattern)
    MeshDataModel* getDataModel() {
        return dataModel;
    }
    /**
     * @}
     */

protected:
    /// build the instance of Geometry from the given vtkPointSet
    virtual void initRepresentation( vtkSmartPointer<vtkPointSet> );

    /// create and initialize dynamic properties
    virtual void initDynamicProperties();

protected slots :

    /// called when the selection is modified
    void changeSelectedSelection( const QModelIndex& current, const QModelIndex& previous );

private:

    /// the concrete building of the 3D objects (Slice/Geometry): none in this case, everything is done by initRepresentation(vtkPointSet)
    virtual void initRepresentation() {};

    /// initialisation of the mesh component members
    void init();

    /// the last picked point
    vtkIdType pickedPointId;

    /// the last picked cell
    vtkIdType pickedCellId;

    /// manages current selection using vtk
    vtkSmartPointer<vtkSelection> currentSelection;

    /// maps a name with a type of selection
    QMap< QString, vtkSmartPointer<vtkSelectionNode> > selectionList;

    /// maps a name with a data array
    QMap< QString, vtkSmartPointer<vtkDataArray> > dataList;

    /// selection GUI View (view as the V in Qt MVC design pattern)
    QTableView* selectionView;

    /// data GUI View (view as the V in Qt MVC design pattern)
    QTableView* dataView;

    /// selection model (model as the M in Qt MVC design pattern)
    MeshSelectionModel* selectionModel;

    /// data model (model as the M in Qt MVC design pattern)
    MeshDataModel* dataModel;

};

}

#endif // MESHCOMPONENT_H
