/*  audiodevs: Abstraction layer for audio hardware & samples
    Copyright (C) 2003-2004 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/

/** \class CAudioSample

  \brief Base class for audio related channels

  CAudioSample is a base class for anything that's linear stream of samples.

*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include "AudioSample.h"


CAudioSample::CAudioSample()
{
   Init();
}

CAudioSample::CAudioSample(const SoundAttributes &attrs, SamplePos buffer_length)
{
   Init();
   if (m_pData == 0)
     return;
   CreateBuffer(attrs, buffer_length);
}

CAudioSample::~CAudioSample()
{
   if (m_pData != 0 && m_pData->deref()) // If were the last reference, clean up
   {
     Reset();
     delete m_pData;
     m_pData = 0;
   }
}

/**
  \brief Copy constructor

*/
CAudioSample::CAudioSample(const CAudioSample &audio_sample)
{
   m_pData = audio_sample.m_pData;
   m_pData->ref();
}

// private

void CAudioSample::Init()
{
   m_pData = new SampleData;
   if (m_pData == 0)
   {
     qWarning("CAudioSample::Init() Failed to allocate SampleData structure.");
     return;
   }
   ReInit();
}

void CAudioSample::ReInit()
{
   m_pData->pMemBuffer = 0;
   m_pData->BufLength = 0;
   m_pData->BufStartOffset = 0;
   m_pData->Attributes.Reset();
   m_pData->BytesPerSample = 0;
   for (int i = 0; i < SoundAttributes::MaxChannel; i++)
      m_pData->PositionOffset[i] = -1;

}

void CAudioSample::Reset()
{
   if (m_pData == 0)
     return;
   if (m_pData->pMemBuffer != 0)
   {
     free(m_pData->pMemBuffer);
   }
   ReInit();
}


void CAudioSample::CalculateOffsets() const
{
   int Offset = 0, Pos;
   int i;

   m_pData->BytesPerSample = m_pData->Attributes.BytesPerSample();
   // Clear existing offsets
   for (i = 0; i < SoundAttributes::MaxChannel; i++)
      m_pData->PositionOffset[i] = -1;
   /* SoundAttributes lists the position in memory order. Here we determine
      the reverse mapping, at which offset a certain position is. This offset
      is in the sample-widths unit (thus not necessarely bytes).
    */
   for (i = 0; i < m_pData->Attributes.Channels; i++)
   {
      Pos = m_pData->Attributes.ChannelPosition[i];
      if (Pos >= 0 && Pos < SoundAttributes::MaxPosition) // check, check, double-check
        m_pData->PositionOffset[Pos] = Offset;
      Offset++;
   }
}


// protected

void CAudioSample::SetSoundAttributes(const SoundAttributes &attr) const
{
   if (m_pData == 0)
     return;
   m_pData->Attributes = attr;
   CalculateOffsets();
}

int CAudioSample::GetBytesPerSample() const
{
   if (m_pData == 0)
     return 0;
   return m_pData->BytesPerSample;
}

bool CAudioSample::CreateBuffer(const SoundAttributes &attrs, SamplePos buffer_length)
{
   unsigned long MemNeeded;
   void *pNewMem;

   //Reset();
   MemNeeded = attrs.BytesPerSample() * buffer_length;
   if (MemNeeded == 0)
     return false;
   pNewMem = realloc(m_pData->pMemBuffer, MemNeeded);
   if (pNewMem == 0)
     return false;
   m_pData->pMemBuffer = pNewMem;
   m_pData->BufLength = buffer_length;
   SetSoundAttributes(attrs);
   return true;
}

void CAudioSample::Detach()
{
   if (m_pData == 0 || m_pData->count == 1)
     return;

qDebug("CAudioSample::Detach() making copy.");
   SampleData *pNewData = new SampleData(*m_pData); // copy
   if (pNewData == 0)
   {
     qWarning("CAudioSample::Detach() Could not create SampleData struct.");
   }
   else
   {
     pNewData->pMemBuffer = malloc(m_pData->BufLength * m_pData->BytesPerSample);
     if (pNewData->pMemBuffer == 0)
     {
       qWarning("CAudioSample::Detach() Could not allocate memory byffer.");
     }
     else
     {
       memcpy(pNewData->pMemBuffer, m_pData->pMemBuffer, m_pData->BufLength * m_pData->BytesPerSample);
     }
   }
   m_pData->deref(); // deref old pointer
   m_pData = pNewData; // make the new copy our own (even if it's null)
}



// public

CAudioSample &CAudioSample::operator =(const CAudioSample &audio_sample)
{
   audio_sample.m_pData->ref(); // avoid x = x'
   if (m_pData->deref())
   {
     Reset();
     delete m_pData;
   }
   m_pData = audio_sample.m_pData;
   return *this;
}

#if 0
CAudioSample CAudioSample::Copy() const
{
   CAudioSample audio_sample;

   if (!IsNull())
   {
     audio_sample.m_pData->BufStartOffset = m_pData->BufStartOffset;
     if (audio_sample.CreateBuffer(m_pData->Attributes, m_pData->BufLength))
     {
       memcpy(audio_sample.m_pData->pMemBuffer, m_pData->pMemBuffer, m_pData->BufLength * m_pData->BytesPerSample);
     }
   }

   return audio_sample;
}
#endif

SoundAttributes CAudioSample::GetSoundAttributes() const
{
   if (m_pData == 0)
     return SoundAttributes();
   return m_pData->Attributes;
}

bool CAudioSample::IsNull() const
{
   return (m_pData == 0 || m_pData->pMemBuffer == 0);
}

/**
  \brief Get total number of samples

  This function will return the total number of samples; for file-based samples
  this can be very large, more than will fit in memory.
*/
SamplePos CAudioSample::TotalLength() const
{
   if (m_pData == 0)
     return 0;
   return m_pData->BufLength;
}

/**
  \brief Returns whether we can seek or not

  When this function returns \b true, we can jump through the sample back
  and forward randomly with \ref SeekTo. If false, we can't do this (for
  example, with streaming audio).
*/
bool CAudioSample::CanSeek() const
{
   return false;
}

/**
   \brief Set position to sample \ref offset
   \return True when succeeded, false when out-of-bounds

   This function will seek internally to the specified offset. If \ref CanSeek returns
   false, this function is a no-op.
*/
bool CAudioSample::SeekTo(SamplePos offset)
{
   return false;
}


/**
  \brief Get sample-value of a channel
  \param offset The offset within the sample
  \param The channel (left, right, center, etc).

   Return sample at offset and channel.

   Illegal offsets and channels will return 0 (silence).
*/
int CAudioSample::GetValue(SamplePos offset, SoundAttributes::Position channel) const
{
   int ChannelOffset, Ret;
   void *pPos;
   int8_t *pS8Pos;
   u_int8_t *pU8Pos;
   int16_t *pS16Pos;
   u_int16_t *pU16Pos;

   if (m_pData == 0 || m_pData->pMemBuffer == 0 || offset >= m_pData->BufLength)
     return 0;
   if (channel < 0 || channel >= m_pData->Attributes.MaxPosition)
     return 0;
   ChannelOffset = m_pData->PositionOffset[channel];
   if (ChannelOffset < 0)
     return 0;
   pPos = (char *)m_pData->pMemBuffer + offset * m_pData->BytesPerSample;
   Ret = 0;
   // this is the real stuff.
   switch(m_pData->Attributes.SampleFormat)
   {
     case SoundAttributes::Signed8:
       pS8Pos = (int8_t *)pPos;
       Ret = pS8Pos[ChannelOffset];
       break;

     case SoundAttributes::Unsigned8:
       pU8Pos = (u_int8_t *)pPos;
       Ret = pU8Pos[ChannelOffset] - 128;
       break;

     case SoundAttributes::Signed16:
       pS16Pos = (int16_t *)pPos;
       Ret = pS16Pos[ChannelOffset];
       break;

     case SoundAttributes::Unsigned16:
       pU16Pos = (u_int16_t *)pPos;
       Ret = pU16Pos[ChannelOffset] - 32768;
       break;

   }
   return Ret;
}

void CAudioSample::SetValue(SamplePos offset, SoundAttributes::Position channel, int value)
{
   int ChannelOffset;
   void *pPos;
   int8_t *pS8Pos;
   u_int8_t *pU8Pos;
   int16_t *pS16Pos;
   u_int16_t *pU16Pos;

   if (m_pData == 0 || m_pData->pMemBuffer == 0 || offset >= m_pData->BufLength)
     return;
   if (channel < 0 || channel >= m_pData->Attributes.Channels)
     return;
   ChannelOffset = m_pData->PositionOffset[channel];
   if (ChannelOffset < 0)
     return;
   Detach();
   pPos = (char *)m_pData->pMemBuffer + offset * m_pData->BytesPerSample;
   //m_pMemBuffer[offset + m_Attributes.Channels + ChannelOffset] = value;
   switch(m_pData->Attributes.SampleFormat)
   {
     case SoundAttributes::Signed8:
       pS8Pos = (int8_t *)pPos;
       if (value < -128) value = -128;
       if (value >  127) value =  127;
       pS8Pos[ChannelOffset] = value;
       break;

     case SoundAttributes::Unsigned8:
       pU8Pos = (u_int8_t *)pPos;
       value += 128;
       if (value <   0) value =   0;
       if (value > 255) value = 255;
       pU8Pos[ChannelOffset] = value;
       break;

     case SoundAttributes::Signed16:
       pS16Pos = (int16_t *)pPos;
       if (value < -32768) value = -32768;
       if (value >  32767) value =  32767;
       pS16Pos[ChannelOffset] = value;
       break;

     case SoundAttributes::Unsigned16:
       pU16Pos = (u_int16_t *)pPos;
       value += 32768;
       if (value <     0) value =     0;
       if (value > 65535) value = 65535;
       pU16Pos[ChannelOffset] = value;
       break;
   }
}



