/* The Cantus project.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "plugin.h"

//#define _DEBUG_


/******************************************************************************
 * Constructor/Destructor
 ******************************************************************************/
/* Constructor.
 */
Plugin::Plugin(void)
{
  refct      = 1;
  dlhandle   = NULL;
  plugindata = cantushash_create();
  cantushash_set_pointer(plugindata, "Cantus:AddListener", 
                                     (gpointer)&eventbus_add_listener);
  cantushash_set_pointer(plugindata, "Cantus:AddListenerSigC",
                                     (gpointer)&eventbus_add_listener_sigc);
  cantushash_set_pointer(plugindata, "Cantus:RemoveListener",
                                     (gpointer)&eventbus_remove_listener);
  cantushash_set_pointer(plugindata, "Cantus:Emit",
                                     (gpointer)&eventbus_emit);
  cantushash_set_pointer(plugindata, "Cantus:PrefSet_int",
                                     (gpointer)&preferences_set_int);
  cantushash_set_pointer(plugindata, "Cantus:PrefGet_int",
                                     (gpointer)&preferences_get_int);
  cantushash_set_pointer(plugindata, "Cantus:PrefSet_bool",
                                     (gpointer)&preferences_set_bool);
  cantushash_set_pointer(plugindata, "Cantus:PrefGet_bool",
                                     (gpointer)&preferences_get_bool);
  cantushash_set_pointer(plugindata, "Cantus:PrefSet_char",
                                     (gpointer)&preferences_set_char);
  cantushash_set_pointer(plugindata, "Cantus:PrefGet_char",
                                     (gpointer)&preferences_get_char);
  cantushash_set_pointer(plugindata, "Cantus:FileInfoGet",
                                     (gpointer)&fileinfomanager_get_info);
  cantushash_set_pointer(plugindata, "Cantus:FileInfoUnlock",
                                     (gpointer)&fileinfomanager_unlock_info);
}


/* Destructor.
 */
Plugin::~Plugin(void)
{
#ifdef _DEBUG_
    std::cout << "Destroying plugin '" << get_name() << "'.\n";
#endif
  DestroyFunc func = (DestroyFunc)cantushash_get_pointer(plugindata,
                                                         "Plugin:DestroyFunc");
  if (func) {
    gint err = func();
#ifdef _DEBUG_
    std::cout << "Plugin destroy func returned '" << err << "'.\n";
#endif
  }
  else
    g_warning("Plugin '%s' has no destructor!\n", get_name());
  
  if (dlhandle)
    dlclose(dlhandle);
  cantushash_destroy(plugindata);
}


/******************************************************************************
 * Public
 ******************************************************************************/
/* Loads a plugin and all its symbols into the object.
 */
gint Plugin::load(const gchar *filename)
{
  // Open the library.
#ifdef _DEBUG_
  std::cout << "Plugin::load(): Opening " << filename << "...\n";
#endif
  if (!(dlhandle = dlopen(filename, RTLD_LAZY))) {
    std::cerr << "Cannot open library: " << dlerror() << '\n';
    return -1;
  }
  
  Symbol symbols[] = {  // Specify which symbols to load.
    { "plugin_destroy",      "Plugin:DestroyFunc",  TRUE  },
    { "plugin_handles",      "Plugin:HandlesFunc",  FALSE },
    { "plugin_read",         "Plugin:ReadFunc",     FALSE },
    { "plugin_write",        "Plugin:WriteFunc",    FALSE },
    { "plugin_get_uiwidget", "Plugin:UIWidgetFunc", FALSE },
    { "plugin_get_uiprefs",  "Plugin:UIPrefsFunc",  FALSE },
    { "plugin_get_uiicon",   "Plugin:UIIconFunc",   FALSE },
    { NULL, NULL, 0 }
  };
  AnyFunc func;
  
  // Load the initializer symbol.
#ifdef _DEBUG_
  std::cout << "Plugin::load(): Loading symbol 'plugin_init'...\n";
#endif
  if (!(func = (AnyFunc)dlsym(dlhandle, "plugin_init"))) {
    std::cerr << "Cannot load symbol 'plugin_init': " << dlerror() << '\n';
    dlclose(dlhandle);
    return -11;
  }
  
  // Call the initializer.
#ifdef _DEBUG_
  std::cout << "Plugin::load(): Calling 'plugin_init'...\n";
#endif
  gint err = ((InitFunc)func)(plugindata);
  if (err != 0) {  // Init failed.
    std::cerr << "Error: Plugin initializer returned error " << err << ".\n";
    dlclose(dlhandle);
    return -31;
  }
  
  // Load all symbols.
  gint i = -1;
  while (symbols[++i].name) {
#ifdef _DEBUG_
    std::cout << "Loading symbol '" << symbols[i].name << "'...\n";
#endif
    func = (AnyFunc)dlsym(dlhandle, symbols[i].name);
    if (symbols[i].required && !func) {
      std::cerr << "Cannot load mandatory symbol '" << symbols[i].name << "': "
                << dlerror() << '\n';
      dlclose(dlhandle);
      return -51;
    }
    cantushash_set_pointer(plugindata, symbols[i].keyname, (void*)func);
#ifdef _DEBUG_
    if (func)
      std::cout << "Symbol '" << symbols[i].name << "' found.\n";
#endif
  }
  
  // Plugin validity check.
  if (!check_plugin()) {
    std::cerr << "Error: Plugin validity check failed.\n";
    dlclose(dlhandle);
    return -41;
  }
  
  return 0;
}


/* Increase a plugin's refcounter.
 */
void Plugin::ref(void)
{
#ifdef _DEBUG_
  std::cout << "Plugin::ref(): " << get_name() << " refct: " << refct << "\n";
#endif
  refct++;
}


/* Decrease a plugin's refcounter. When the refcounter is <= 0 the plugin will
 * be unloaded. On object creation, the refcounter is 1.
 */
void Plugin::unref(void)
{
  refct--;
#ifdef _DEBUG_
  std::cout << "Plugin::unref(): " << get_name() << " refct: " << refct << "\n";
#endif
  if (refct > 0)
    return;
  signal_plugin_deleted.emit(this);
  delete this;
}


/* Define the plugin's priority.
 */
void Plugin::set_priority(gint prio)
{
  priority = prio;
}


/* Returns the plugin's priority.
 */
gint Plugin::get_priority(void)
{
  return priority;
}


/* Returns a human readable plugin name.
 */
const gchar *Plugin::get_name(void)
{
  return cantushash_get_char(plugindata, "Plugin:Name");
}


/* Returns the name of the tags handled by this plugin.
 */
const gchar *Plugin::get_tagname(void)
{
  return cantushash_get_char(plugindata, "Plugin:TagName");
}


/* Returns the title for the plugin's notebook tab label.
 */
const gchar *Plugin::get_label(void)
{
  return cantushash_get_char(plugindata, "Plugin:Label");
}


/* Returns the the plugin description (should be 200 chars max.)..
 */
const gchar *Plugin::get_description(void)
{
  return cantushash_get_char(plugindata, "Plugin:Description");
}


/* Returns the plugin's major version number.
 */
gint Plugin::get_majorversion(void)
{
  return cantushash_get_int(plugindata, "Plugin:MajorVersion");
}


/* Returns the plugin's minor version number.
 */
gint Plugin::get_minorversion(void)
{
  return cantushash_get_int(plugindata, "Plugin:MinorVersion");
}


/* Returns a function pointer to the plugin's read() function.
 */
const ReadFunc Plugin::get_readfunc(void)
{
  return (ReadFunc)cantushash_get_pointer(plugindata, "Plugin:ReadFunc");
}


/* Returns a function pointer to the plugin's write() function.
 */
const WriteFunc Plugin::get_writefunc(void)
{
  return (WriteFunc)cantushash_get_pointer(plugindata, "Plugin:WriteFunc");
}


/* Given a file name, this function returns TRUE if the plugin is responsible
 * for handling this filetype, otherwise FALSE.
 */
gint Plugin::handles(const gchar *filename)
{
#ifdef _DEBUG_
  std::cout << "Plugin::handles(): Checking whether '"
            << get_name() << "' is responsible for " << filename << ".\n";
#endif
  HandlesFunc handles = (HandlesFunc)cantushash_get_pointer(plugindata,
                                                          "Plugin:HandlesFunc");
  if (!filename || *filename == '\0' || !handles)
    return FALSE;
#ifdef _DEBUG_
  std::cout << "Plugin::handles(): Checking whether the patterns from "
            << get_name() << " match " << filename << ".\n";
#endif
  const gchar **pattern = (const gchar **)cantushash_get_pointer(plugindata,
                                                              "Plugin:Pattern");
  if (!shellpattern_match_any(filename, pattern, FALSE))
    return FALSE;
  return handles(filename);
}


/* Calls the plugin's uiwidget() function.
 */
void *Plugin::get_uiwidget(gboolean vertical)
{
  UIWidgetFunc func = (UIWidgetFunc)cantushash_get_pointer(plugindata,
                                                         "Plugin:UIWidgetFunc");
  g_assert(func != NULL);
  return func(vertical);
}


/******************************************************************************
 * Protected
 ******************************************************************************/
/* Checks, whether the plugin meets all mandatory requirements. Returns TRUE
 * if the plugin is valid, otherwise FALSE.
 */
gboolean Plugin::check_plugin(void)
{
  gint compat = cantushash_get_int(plugindata, "Plugin:CompatibilityLevel");
  if (compat > PLUGIN_COMPAT_LEVEL) {           // Check compatibility.
    std::cerr << "Error: Incompatible plugin.\n";
    return FALSE;
  }
                                                // Pattern defined?
  if (!cantushash_get_pointer(plugindata, "Plugin:Pattern")) {
    std::cerr << "Error: Plugin did not specify a filename pattern.\n";
    return FALSE;
  }
  if (*(this->get_name()) == '\0') {            // Invalid plugin name.
    std::cerr << "Error: Initializer returned an emtpy name.\n";
    return FALSE;
  }
  if (strlen(this->get_description()) > 200)    // Invalid description length.
    g_warning("Plugin::get_description(): Description < 200 chars failed.\n");
                                                // Preferences icon defined?
  if (cantushash_get_pointer(plugindata, "Plugin:UIPrefsFunc")
    && !cantushash_get_pointer(plugindata, "Plugin:UIIconFunc")) {
    std::cerr << "Error: Plugin has preferences but did not specify an icon\n.";
    return FALSE;
  }
  
  return TRUE;
}
