/* The Cantus project.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "gui_preferences.h"

//#define _DEBUG_


/******************************************************************************
 * Constructor/Destructor
 ******************************************************************************/
/* Constructor.
 */
PreferencesWindow::PreferencesWindow(std::string gladefile,
                                     std::string gladedomain)
{
  glade_file   = gladefile;
  glade_domain = gladedomain;
  window = NULL;
  preftree.signal_selection_changed.connect(
          SigC::slot(*this, &PreferencesWindow::on_preftree_selection_changed));
}


/* Destructor.
 */
PreferencesWindow::~PreferencesWindow()
{
}


/******************************************************************************
 * Public
 ******************************************************************************/
void PreferencesWindow::show(void)
{
  if (!window)
    create();
  window->show();
}


void PreferencesWindow::hide(void)
{
  window->hide();
}


/* Updates one widget according to the given value.
 * Returns always TRUE.
 */
bool PreferencesWindow::update_widget(std::string name, int type, void *value)
{
  Gtk::Widget *widget = refXml->get_widget(name);
  if (!widget)
    return TRUE;
  Gtk::Entry  *entry  = dynamic_cast<Gtk::Entry*>(widget);
  if (entry) {
    entry->set_text(value ? (const gchar*)value : "");
    return TRUE;
  }
  Gtk::CheckButton *check = dynamic_cast<Gtk::CheckButton*>(widget);
  if (check) {
    check->set_active((gboolean)value);
    return TRUE;
  }
  return TRUE;
}


/******************************************************************************
 * Private
 ******************************************************************************/
/* Read the preferences window's glade file into an xml tree.
 */
void PreferencesWindow::create(void)
{
  // Load the Glade file and instantiate its widgets.
  try {
    refXml = Gnome::Glade::Xml::create(glade_file, glade_domain);
  }
  catch(const Gnome::Glade::XmlError & err) {
    std::cerr << err.what() << '\n';
    return;
  }
  
  // Get the Glade-instantiated Dialog.
  refXml->get_widget("preferences",         window);
  refXml->get_widget("viewport_categories", viewport);
  refXml->get_widget("notebook_main",       notebook);
  Gtk::Button *button = (Gtk::Button*)refXml->get_widget(
                                                    "button_preferences_close");
  g_assert(button != NULL);
  button->signal_clicked().connect(
                SigC::slot(*this, &PreferencesWindow::on_button_close_clicked));
  
  // Attach the category treeview to the scrolled window.
  viewport->add(preftree);
  preftree.show();
  preftree.set_headers_visible(FALSE);
  
  // Add all categories by iterating through all notebook tab labels.
  fill_tree();
  
  // Select the first category.
  Gtk::TreeModel::iterator iter = preftree.get_model()->children().begin();
  preftree.get_selection()->select(iter);
  
  connect_signals();
}


/* Adds all categories by iterating through all notebook tab labels.
 */
void PreferencesWindow::fill_tree(void)
{
  std::list<Gtk::Widget*>           list = notebook->get_children();
  std::list<Gtk::Widget*>::iterator iter = list.begin();
  while (iter != list.end()) {
    Gtk::Label *label       = (Gtk::Label*)notebook->get_tab_label(**iter);
    std::string catname     = label->get_text();
    std::string pixbufname  = PACKAGE_PIXMAPS_DIR + label->get_name();
    preftree.add(catname, pixbufname);
    iter++;
  }
}


/* Connect all widget callback signals.
 */
void PreferencesWindow::connect_signals(void)
{
  g_return_if_fail(window != NULL);
  std::list<Gtk::Widget*> list;
  get_children_recursive(notebook, &list);
  std::list<Gtk::Widget*>::iterator iter = list.begin();
  while (iter != list.end()) {
    Gtk::Widget *widget = *iter;
    Gtk::Entry  *entry  = dynamic_cast<Gtk::Entry*>(widget);
    if (entry) {
      entry->signal_changed().connect(
            SigC::bind(
              SigC::slot(*this, &PreferencesWindow::on_widget_changed), entry));
      iter++;
      continue;
    }
    Gtk::CheckButton *check = dynamic_cast<Gtk::CheckButton*>(widget);
    if (check) {
      check->signal_toggled().connect(
            SigC::bind(
              SigC::slot(*this, &PreferencesWindow::on_widget_changed), check));
      iter++;
      continue;
    }
    iter++;
  }
}


void PreferencesWindow::on_preftree_selection_changed(Gtk::TreeIter iter)
{
#ifdef _DEBUG_
  printf("PreferencesWindow::on_preftree_selection_changed(): Called.\n");
#endif
}


void PreferencesWindow::on_button_close_clicked(void)
{
  window->hide();
}


void PreferencesWindow::on_widget_changed(Gtk::Widget *widget)
{
#ifdef _DEBUG_
  printf("PreferencesWindow::on_widget_changed(): Called.\n");
  printf("Widget: %s\n", widget->get_name().c_str());
#endif
  Gtk::Entry *entry = dynamic_cast<Gtk::Entry*>(widget);
  if (entry) {
#ifdef _DEBUG_
    printf("Entry: %s\n", entry->get_text().c_str());
#endif
    signal_preference_changed.emit(entry->get_name(),
                                   G_TYPE_CHAR,
                                   (void*)entry->get_text().c_str());
    return;
  }
  Gtk::CheckButton *check = dynamic_cast<Gtk::CheckButton*>(widget);
  if (check) {
#ifdef _DEBUG_
    printf("Check: %s\n", check->get_active() ? "TRUE" : "FALSE");
#endif
    signal_preference_changed.emit(check->get_name(),
                                   G_TYPE_BOOLEAN,
                                   (void*)check->get_active());
    return;
  }
}


void PreferencesWindow::get_children_recursive(
                   Gtk::Container *container, std::list<Gtk::Widget*> *children)
{
  std::list<Gtk::Widget*>           newchildren = container->get_children();
  std::list<Gtk::Widget*>::iterator iter        = newchildren.begin();
  while (iter != newchildren.end()) {
    Gtk::Container *container = dynamic_cast<Gtk::Container*>(*iter);
    if (container)
      get_children_recursive(container, children);
    iter++;
  }
  newchildren.sort();
  children->merge(newchildren);
}
