;;; @file main.s
;;; Interrupt-driven file server for 8-bit Commodores
;;; @author Marko Mkel (msmakela@nic.funet.fi)
;;; @author Olaf Seibert (seibert@mbfys.kun.nl)

cmdinfo = 0	; code for requesting machine type info
cmdload = 1	; code for loading
cmdsave = 2	; code for saving
cmdjump = 3	; code for jumping to an address
cmdrun	= 4	; code for running a BASIC program

	header
mystart:	; A jump table for external programs
	jmp install		; Installs the interrupt handler.
	jmp deinstall		; Removes the interrupt handler.
	jmp prepare		; Prepare for using the data link.
	jmp send_switch		; Switches data direction and sends a byte.
	jmp send		; Sends a byte.
	jmp receive_switch	; Switches data direction and receives a byte.
	jmp receive		; Receives a byte.
	jmp exit		; Terminates an extension.

#include "cable.s"

install:
.(
	sei
	lda cinv	; check the original IRQ vector
	ldx cinv+1
	cmp #<irq
	bne irqinit
	cpx #>irq
	beq skipirq
irqinit:	; if it was different from our irq wedge, install the wedge
	sta oldirq	; first save the old IRQ vector
	stx oldirq+1
	lda #<irq	; and then install ours
	sta cinv
	lda #>irq
	sta cinv+1
skipirq:
.)
	initram
	initcable
	cli
	rts
prepare:
#if allslow			; C2N232 cable
	deactirq
#if host = 64 || host = 128 || host = 16
prepared:
	pla
	tax
	pla
	tay
	phslow
	jmp postpared
postpare:
	pla
	tax
	pla
	tay
	plslow
postpared:
	tya
	pha
	txa
	pha
#endif
#endif
	rts

exit:
#if allslow			; C2N232 cable
#if host = 64 || host = 128 || host = 16
	jsr postpare
#endif
	reactirq		; reactivate interrupts on the VIA/CIA
#endif
	jsr install		; reinstall the interrupt handler and
	irqexit			; exit from interrupt.

irq:	isrequest

#if allslow
#if host = 64 || host = 128 || host = 16
	jsr prepare
#endif
#endif
	jsr rereceive	; receive byte without requiring handshaking
	cmp #cmdinfo
	bne load	; branch if the client did not request for host info
	infoinit
	lda #host	; return the host info byte
	jsr send_switch
	lda #<mystart	; tell the driver start address
	jsr send
	lda #>mystart
	jsr send
	lda bstart	; and the BASIC start address
	jsr send
	lda bstart+1
	jsr send

#if allslow
#if host = 64 || host = 128 || host = 16
slreturn:
	jsr postpare
#endif
#endif
return:
oldirq = * + 1
	jmp *		; continue the normal irq (selfmodifying code)

load:	cmp #cmdload	; was it the command for loading?
.(
	beq nojump
	jmp jump	; no, skip
nojump:
.)
#if host = 205 || host = 210 || host = 220
	lda 1
	pha		; memorize the data bank
#endif
	jsr gethdr	; get the addresses
#if host = 205 || host = 210 || host = 220
	lda 1
	cmp #1
	bne nobasic	; bank not 1 => can't be basic
#endif
#if ramexp
	lda bank	; bank not the default => can't be basic
	bne nobasic
#endif
	lda start	; check if it is a basic program
	eor bstart
	bne nobasic
	lda start+1
	eor bstart+1
nobasic	pha		; accu is 0 if it is a basic program
#if allslow
#else
	phfast
#endif
	do_load
#if allslow
#else
	plfast
#endif

	pla
	bne nobasadj	; it was not a basic program
	lda end		; set the basic end address
	sta bend
	lda end+1
	sta bend+1
	jsr linkprg	; relink the program
#if host = 128
	lda #0
	sta $ff00	; restore the MMU configuration (linkprg changes it)
#endif
nobasadj
#if host = 205 || host = 210 || host = 220
	pla
	sta 1		; restore the data bank
#endif
#if allslow
#if host = 64 || host = 128 || host = 16
	jmp slreturn
#else
	jmp (oldirq)
#endif
#else
	jmp (oldirq)
#endif

jump:	cmp #cmdjump	; was it the command for jumping?
	bne save	; no, skip

	jsr receive	; get the memory configuration (for the c128)
#if host = 128
	sta bank	; and temporarily store it
#endif
	jsr receive	; get the jump address low
	tax
	jsr receive	; get the jump address high
#if allslow
#if host = 64 || host = 128 || host = 16
	tay
	plslow
	tya
#endif
#endif
	pha		; and store it
	txa
	pha		; store the jump address low
	lda #0
	pha		; clear the flags
	pha		; and the registers
	pha
	pha
#if host = 128
	lda bank
	pha		; store the MMU configuration
#endif
	jsr deinstall	; deinstall the IRQ routine
#if host - 128
	sta crsw	; turn the cursor off
#if host <> 16 && host <> 205 && host <> 210 && host <> 220
	sta blnsw
#endif
#endif
	jmp (oldirq)

save:
	cmp #cmdsave
.(
	beq yessave
	jmp run
yessave
.)

#if allslow
#else
	phfast
#endif

#if host = 205 || host = 210 || host = 220
	lda 1
	pha
#endif
	jsr gethdr	; get the pointers
	do_save

#if host = 205 || host = 210 || host = 220
	pla
	sta 1
#endif
#if allslow
#else
	plfast
#endif

return2:
#if allslow
#if host = 64 || host = 128 || host = 16
	jmp slreturn
#else
	jmp (oldirq)	; return from interrupt
#endif
#else
	jmp (oldirq)	; return from interrupt
#endif

run:
	cmp #cmdrun
	bne return2

	jsr deinstall		; Install the original interrupt routine
#if host = 16 || host = 128
#if host = 16
	sta crsw		; turn the cursor off
#endif	; host = 16
	cli			; process the pending interrupt
	jmp basrun
#else	; host = 16 || host = 128
	sta crsw		; disable the cursor blinking
#if host = 205 || host = 210 || host = 220
	cli			; process the interrupt
	lda #$fe
	sta $43			; BASIC status: program running
	jsr clr			; clr does stxpt
	jmp newstt
#else
	sta blnsw
	cli		; process the pending interrupt in a very kludgeous manner.
	jsr clr	; The stack won't run out, since the BASIC clr call
		; initializes the stack pointer.
	jsr stxpt	; Set the BASIC text pointer (to the beginning of the prg)
	jmp newstt	; BASIC warm start
#endif
#endif	; host = 16 || host = 128

deinstall:	; restore the original interrupt routine
	lda oldirq
	sta cinv
	lda oldirq+1
	sta cinv+1
	deinitcable
	rts

gethdr:
	jsr receive	; get the memory bank byte
#if host = 205 || host = 210 || host = 220
	sta 1
#endif
#if ramexp
	sta bank
#else
#if host = 128
	lsr
	ror
	ror
	ora #$3f
	sta bank
#endif		; host = 128
#endif		; ramexp
	lda #0	; send the acknowledgement code 0
	jsr send_switch
	jsr receive_switch ; get the file start and end vectors
	sta start
	jsr receive
	sta start+1
	jsr receive
	sta end
	jsr receive
	sta end+1
	rts

	incmptr_decl

#if ramexp
rptrinit:
	sec
	lda end+1
	ora #$3f
	sbc start+1	; get the amount of 16k bank crossings
	asl
	rol
	rol
	and #3
	tax		; and store it to x
.(
	bne nowrap	; check for address wraparound
	lda start
	cmp end		; if start >= end, the 16k bank is switched 4 times.
	lda start+1
	sbc end+1
	bcc nowrap
	ldx #4
nowrap:
.)
	lda start+1
	and #$3f
	ora #$40
	sta start+1
	lda end+1
	and #$3f
	ora #$40
	sta end+1
	rts
#endif		; ramexp
