/**
 * @file mem.c
 * Memory transfer primitives
 * @author Marko Mkel (msmakela@nic.funet.fi)
 */

/*
 * Copyright  1994-1996 Marko Mkel and Olaf Seibert
 * Copyright  2001 Marko Mkel
 * Original Linux and Commodore 64/128/Vic-20 version by Marko Mkel
 * Ported to the PET and the Amiga series by Olaf Seibert
 * Restructured by Marko Mkel
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "comm.h"
#include "info.h"

/** send a memory transfer command and a header
 * @param comm		the communication primitives
 * @param op		the operation (REQ_LOAD or REQ_SAVE)
 * @param bank		memory bank
 * @param start		start address
 * @param end		end address
 * @return		zero on success, nonzero on error
 */
static int
mem_op (const struct comm* comm,
	enum opcode op,
	unsigned bank, unsigned start, unsigned end)
{
  char req[4];
  req[0] = op;
  req[1] = bank;

  if ((*comm->comm_write) (req, 2))
    return 1;
  (*comm->comm_sr) ();
  if ((*comm->comm_read) (req, 1))
    return 2;
  if (*req)
    return 3;

  req[0] = start;
  req[1] = start >> 8;
  req[2] = end;
  req[3] = end >> 8;

  (*comm->comm_rs) ();
  if ((*comm->comm_write) (req, 4))
    return 4;

  return 0;
}

/** save a memory area
 * @param comm		the communication primitives
 * @param bank		memory bank (0=normal)
 * @param start		start address (inclusive)
 * @param end		end address (exclusive)
 * @param buf		(output) the saved data
 * @return		zero on success, nonzero on error
 */
int
save (const struct comm* comm,
      unsigned bank, unsigned start, unsigned end,
      void* buf)
{
  if (mem_op (comm, REQ_SAVE, bank, start, end))
    return 1;
  (*comm->comm_sr) ();
  if ((*comm->comm_read) (buf, end == start
			  ? 0x10000
			  : (end - start) & 0xffff))
    return 2;
  (*comm->comm_rs) ();
  return 0;
}

/** load a memory area
 * @param comm		the communication primitives
 * @param bank		memory bank (0=normal)
 * @param start		start address (inclusive)
 * @param end		end address (exclusive)
 * @param buf		the data to be loaded
 * @return		zero on success, nonzero on error
 */
int
load (const struct comm* comm,
      unsigned bank, unsigned start, unsigned end,
      const void* buf)
{
  if (mem_op (comm, REQ_LOAD, bank, start, end))
    return 1;
  if ((*comm->comm_write) (buf, end == start
			   ? 0x10000
			   : (end - start) & 0xffff))
    return 2;
  return 0;
}
